#include <stdio.h>
#include <fstream.h>
#include <iostream.h>

#include "global.h"
#include "neighborTable.h"
#include "vrt.h"
#include "gossipAgent.h"
#include "gossipHost.h"
#include "leaveAgent.h"
#include "controlAgent.h"
#include "connectivityAgent.h"
#include "dataAgent.h"
#include "gossipPayLoad.h"
#include "rttEstimationAgent.h"
#include "query.h"
#include "overheadStat.h"

GossipAgent::GossipAgent(GossipHost *gossipHostPtrIn, 
			 int numMembers, int *memberArrayPtr, 
			 int lowerDegreeBoundIn,int upperDegreeBoundIn, 
			 int sourceFlgIn) {

  gossipHostPtr  = gossipHostPtrIn;
  vrtPtr=new VirtualRoutingTable(MAX_ADDRESSES,
				 gossipHostPtr->GetMyAddr(),
				 numMembers,
				 memberArrayPtr);
  
  nbrTabPtr=new NbrTable();
  leaveMode=0;

  if (sourceFlgIn) {
    cout << "\n sessionSourceAddr = " << GetMyAddr() ;
  }

  queryPtr = new Query(this,vrtPtr,nbrTabPtr);

  lowerDegreeBound=lowerDegreeBoundIn;
  upperDegreeBound=upperDegreeBoundIn;
  sourceFlg=sourceFlgIn;

  connectivityAgentPtr = new ConnectivityAgent(this,
					       vrtPtr,
					       nbrTabPtr);

  dataAgentPtr = new DataAgent(this,vrtPtr,nbrTabPtr);
  rttEstimationAgentPtr = new RTTEstimationAgent(this,
						 vrtPtr,
						 nbrTabPtr,
						 sourceFlg);

  controlAgentPtr = new ControlAgent(this,vrtPtr,nbrTabPtr);
  
  leaveAgentPtr=NULL;

  overheadStatPtr = new OverheadStat(this,(verbosity > 0));

}

void GossipAgent::StartGossipAgent(){
  connectivityAgentPtr->StartJoin();
}

GossipAgent::~GossipAgent(){
  if(overheadStatPtr != NULL) delete overheadStatPtr;

  delete vrtPtr;
  delete nbrTabPtr;

  if(controlAgentPtr != NULL) delete controlAgentPtr;
  if(connectivityAgentPtr != NULL) delete connectivityAgentPtr;
  if(dataAgentPtr != NULL) delete dataAgentPtr;
  if(leaveAgentPtr != NULL) delete leaveAgentPtr;
  if(rttEstimationAgentPtr != NULL) delete rttEstimationAgentPtr;
  if(queryPtr != NULL) delete queryPtr;
}


void GossipAgent::InitLeave(int stayOnLeaveTime){
  delete connectivityAgentPtr;
  connectivityAgentPtr=NULL;
  leaveMode=1;
  leaveAgentPtr=new LeaveAgent(this,vrtPtr,nbrTabPtr);
  leaveAgentPtr->InitLeave(stayOnLeaveTime);
}

void GossipAgent::ReadyToLeave(){
  gossipHostPtr->ReadyToLeave();
}

void GossipAgent::SendToNetwork(int toAddr,
				int toPort,
				GossipMsgType msgType,
				SerializableObject *gossipMsgPtr){
  int priority;

  switch(msgType) {
  case DATA:
  case POKE_DATA:
  case SMALL_PROBE_DATA:
    priority = ((DataMsg *)gossipMsgPtr)->GetPriority();
    break;
  default:
    priority = PRIORITY_CNTL_DEFAULT;
  }

  GossipPayLoad *gossipPayloadPtr = new GossipPayLoad(msgType,gossipMsgPtr);
  PacketType packetType = gossipPayloadPtr->GetPacketType();
  overheadStatPtr->UpdateOverhead(packetType,0,gossipPayloadPtr->Size());
  gossipHostPtr->SendToNetwork(toAddr,toPort,gossipPayloadPtr,
			       packetType, priority);
}


void GossipAgent::RecvFromNetwork(int fromAddr,
				  int fromPort,
				  GossipPayLoad *gossipPacketPtr){

  PacketType packetType = gossipPacketPtr->GetPacketType();
  overheadStatPtr->UpdateOverhead(packetType,1,gossipPacketPtr->Size());

  /*******
  Bad programming practice: I am using downcasting here, which is 
  frowned upon. I don't know how to eliminate it without considerable 
  trouble, and this is not priority. Finally, this is "safe downcasting" as 
  it is done based on a code field.
  *******/

  int gossipPacketCode=gossipPacketPtr->GetCode();
  SerializableObject *msg=gossipPacketPtr->GetMsg();

  if(
     (gossipPacketCode != POKE_DATA) &&
     (gossipPacketCode != DATA) &&
     (gossipPacketCode != RTT_ESTIMATOR) 
     ){
    if(fromPort != GOSSIP_PORT){
      cout << "WRN:GossipAgent:" 
	   << "Received unknown packet from "
	   << GetNameByAddr(fromAddr)
	   << fromPort;
      return;
    }
  }
  
  switch(gossipPacketCode){
  case ADD_NBR_REQUEST:
    if(!leaveMode)
      connectivityAgentPtr->RecvAddNbrRequest(fromAddr,
					      fromPort,
					      (AddNbrRequestMsg *)msg);
    else return;
    break;
    
  case ADD_NBR_RESPONSE:
    if(!leaveMode){
      connectivityAgentPtr->RecvAddNbrResponse(fromAddr,
					       fromPort,
					       (AddNbrResponseMsg *)msg);
      
    }
    else return;
    break;
    
  case NBR_CONFIRM_REQUEST:
    if(!leaveMode){
      int ret=
	connectivityAgentPtr->RecvNbrConfirmRequest(fromAddr, 
						    fromPort, 
						    (NbrConfirmRequestMsg *)msg);
      
      if(ret == 1){
	controlAgentPtr->SendControlUpdateToAllNbrs();
      }
    }
    else
      return;
    break;

  case NBR_CONFIRM_RESPONSE:
    if(!leaveMode){
      int ret=
	connectivityAgentPtr->RecvNbrConfirmResponse(fromAddr, 
						     fromPort, 
						     (NbrConfirmResponseMsg *)msg);
      if(ret == 1){
	controlAgentPtr->SendControlUpdateToAllNbrs();
      }
    }
    else
      return;
    break;
    
  case CONTROL_UPDATE:
    controlAgentPtr->RecvControlUpdate(fromAddr,
				       fromPort,
				       (ControlUpdateMsg *)msg);
    break;
    
  case PROBE_REQUEST:
    if(!leaveMode)
      connectivityAgentPtr->RecvProbeRequest(fromAddr,
					     fromPort,
					     (ProbeRequestMsg *)msg);
    else return;
    break;

  case PROBE_RESPONSE:
    if(!leaveMode)
      connectivityAgentPtr->RecvProbeResponse(fromAddr,
					      fromPort,
					      (ProbeResponseMsg *)msg);
    else return;
    break;

  case JOIN_REQUEST:
    if(!leaveMode)
      connectivityAgentPtr->RecvJoinRequest(fromAddr,
					    fromPort,
					    (JoinRequestMsg *)msg);
    else return;
    break;
    
  case JOIN_RESPONSE:
    if(!leaveMode)
      connectivityAgentPtr->RecvJoinResponse(fromAddr,
					     fromPort,
					     (JoinResponseMsg *)msg);
    else return;
    break;
    
  case POKE_REQUEST:
  case POKE_DATA:
    if(!leaveMode)
      connectivityAgentPtr->RecvProbeMsg(fromAddr, fromPort, gossipPacketPtr);
    else return;
    break;

  case SMALL_PROBE_REQUEST:
  case SMALL_PROBE_DATA:
    if(!leaveMode)
      connectivityAgentPtr->RecvSmallProbeMsg(fromAddr, fromPort, 
					      gossipPacketPtr);
    else return;
    break;

  case INTENT_TO_CANCEL_NBR:
    controlAgentPtr->RecvIntentToCancelNbrMsg(fromAddr,
					      fromPort,
					      (IntentToCancelNbrMsg *)msg);
    break;

  case CANCEL_NBR:
    controlAgentPtr->RecvCancelNbrMsg(fromAddr,
				      fromPort,
				      (CancelNbrMsg *)msg);
    break;

  case ESTIMATE_DELAY_REQUEST:
    controlAgentPtr->RecvEstimateDelayRequest(fromAddr,
					      fromPort,
					      (EstimateDelayRequestMsg *)msg);
    break;

  case ESTIMATE_DELAY_RESPONSE:
    controlAgentPtr->RecvEstimateDelayResponse(fromAddr,
					       fromPort,
					       (EstimateDelayResponseMsg *)msg);
    break; 

  case DATA:

    dataAgentPtr->Forward(fromAddr,fromPort,(DataMsg *)msg);
    break;

  case RTT_ESTIMATOR:
    rttEstimationAgentPtr->RecvRTTEstimator(fromAddr,
					    fromPort,
					    (RTTEstimationMsg *)msg);
    break;
    
    
  default:
    MyError("GossipAgent::RecvFrom: wrong code\n");
    break;
  }
}

int GossipAgent::GetLowerDegreeBound(){
  return(lowerDegreeBound);
}

int GossipAgent::GetUpperDegreeBound(){
  return(upperDegreeBound);
}

int GossipAgent::GetMyAddr(){
  return(gossipHostPtr->GetMyAddr());
}

int GossipAgent::LeaveMode(){
  return leaveMode;
}

void GossipAgent::DoDebug(int myAddr){
  overheadStatPtr->ReportOverhead();

  if(verbosity > 0){
    vrtPtr->Print();
    nbrTabPtr->Print();
  }

  vrtPtr->DoDebug();
}

void GossipAgent::RecvDataForApp(const char *buf,int bufLen) {
  gossipHostPtr->RecvDataForApp(buf,bufLen);
}


/* yhchu */
void GossipAgent::SendDataForApp(const char *buf,int bufLen, int priority) {
  dataAgentPtr->SendData(buf,bufLen, priority);
}


Query *GossipAgent::GetQueryAgent(){
  return(queryPtr);
}


void GossipAgent::PrintPacketLoss(){
    dataAgentPtr->DoDebug();
}
