#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <assert.h>
#include <string.h>

#include "global.h"
#include "neighborTable.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "gossipAgent.h"
#include "dataAgent.h"
#include "gossipPayLoad.h"
#include "TIMERS/congestionCheckTimer.h"
#include "sourceEntry.h"

DataAgent::DataAgent(GossipAgent *gossipAgentPtrIn,
		     VirtualRoutingTable *vrtPtrIn,
		     NbrTable *nbrTabPtrIn
		     ) {
  gossipAgentPtr=gossipAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;

  seqNum=0;

}
 
void DataAgent::SendData(const char *buf,int bufLen, int priority) {
  assert(this != NULL); 
  
  if(gossipAgentPtr->LeaveMode()){
    return;
  }

  char *dataPtr = new char[bufLen];
  memcpy(dataPtr,buf,bufLen);
  
  long currTime=GetCurrTime();
  DataMsg *dataMsgPtr=new DataMsg(bufLen,
				  gossipAgentPtr->GetMyAddr(),
				  seqNum,
				  currTime,
				  UNDEFINED_RATE,
				  priority,
				  dataPtr);
  if(verbosity > 1){
    cout << "\n" << currTime << ":Sending Sequence Number " << seqNum;
  }
  if((verbosity == 1) && ((seqNum % SourceEntry::STATS_PERIOD) == 0)){
    cout << "\n" << currTime << ":Sending Sequence Number " << seqNum;
  }
  seqNum++;
   
  
  // Duplicate and send this message to all nbrs who use me as next 
  // hop to reach me
  ForwardDataMsg(dataMsgPtr);
  
  delete dataMsgPtr;
}


/* forward data to children */
void DataAgent::ForwardDataMsg(DataMsg *dataMsgPtr) {
  int sourceAddr = dataMsgPtr->GetSourceAddr();
  VRTEntryType *route= vrtPtr->GetRoute(sourceAddr);
  if(route == NULL) MyError("No route for source!!!");

  int numChildren=route->GetNumChildren();
  int *childAddrListPtr = new int[numChildren];
  route->GetChildList(childAddrListPtr);

  for(int i=0; i < numChildren; i++){
    nbrTabPtr->UpdateStatsSentToNbr(childAddrListPtr[i],
				    dataMsgPtr->GetDataSize());
    
    
    long rateToChild=nbrTabPtr->GetRateSentToNbr(childAddrListPtr[i]);
    DataMsg *dataMsgCopyPtr = dataMsgPtr->Duplicate();
    dataMsgCopyPtr->ResetLastHopSendRate(rateToChild);
    gossipAgentPtr->SendToNetwork(childAddrListPtr[i], 
				  GOSSIP_PORT, 
				  DATA, 
				  dataMsgCopyPtr);
  
  }
  
  delete [] childAddrListPtr;
}


/*
 * what is a flow?
 * - one TCP connection
 * - at most one flow from/to any pair of hosts
 *
 * what is available bw?
 * - per-flow
 * - aggregate flow
 *
 * which layer should monitor the flow?
 * - pokeMgr
 * 
 * per-flow
 * - the parent receives full source rate
 * - the parent receives partial source rate
 * - the link is "really" bad: no data can go through
 *
 * heuristics for per-flow
 * - sender marks each sending pkt with seq num
 * - receiver
 *   - ignore the first 3 seconds
 *   - 
 */

/*
 * - update nbr send rate
 * - update my rcv rate
 * - update bandwidth estimation
 * - check if source is known and valid
 * - check if pkt if good (not dup or late)
 * - forward data to children
 * - update source statistics
 * - hand off to application
 */
void DataAgent::Forward(int fromAddr,int fromPort,DataMsg *msgPtr){
  if((!nbrTabPtr->IsNbr(fromAddr)) ||
     (fromPort != GOSSIP_PORT) 
     )
    return;
  
  /* Update rate sent by the parent (from parent's dataAgent perspective, not
   * from TFRC or TCP perspective).  This rate is smoothed by the parent
   */
  long parentSendRate=msgPtr->GetLastHopSendRate();
  nbrTabPtr->UpdateRateSentByNbr(fromAddr,parentSendRate);
  
  /****************************
     Traditional DVMRP requires dropping packet unless it comes from source.
     I'm removing this, because if a node changes its parent during routing,
     it wants to keep receiving data from the old parent until the new parent
     starts forwarding packets. This can minimize loss. Duplication is 
     prevented by use of a sequential buffer.

   *******************************/ 
    
  //Now determine if a bandwidth report of parent must be made.
  VRTEntryType *routeParent= vrtPtr->GetRoute(fromAddr);
  if(routeParent == NULL)
    MyError("No route to parent!!");
  
  long lastBWReportTime=nbrTabPtr->GetLastBWReportTime(fromAddr);
  long timeSinceLastReport=GetCurrTime()-lastBWReportTime;
  if(
     (lastBWReportTime == NO_REPORT) ||
     (timeSinceLastReport > BW_REPORT_PERIOD) ){
    
    long parentSendRate=nbrTabPtr->GetRateSentByNbr(fromAddr);
    long rateRcvdFromParent=nbrTabPtr->GetRateRcvdFromNbr(fromAddr);
    int MIN_PARENT_RATE = MAX_SOURCE_RATE/4;
    
    if(
       (rateRcvdFromParent != UNDEFINED_RATE) &&
       (parentSendRate != UNDEFINED_RATE) &&
       (parentSendRate > MIN_PARENT_RATE) ){
      
      /***************  
	We have parentSendRate > MIN_PARENT_RATE to
	take care of the fact that the parent may not have sent any
        data along this link in the last few seconds, perhaps because
        it became a parent only recently (or due to a topology change), 
        and just started sending data to the client. As the counting times 
	are not precisely synchronized, the client has received only a
	small fraction of the packets. Thus the parent rate could be non-zero
	and the child rate could be very small (zero in the limit). 
        This would create a high loss rate 1 in the limit).
	The condition above ensures a certain minimum data flow for any 
	reasonable judgement to be made. The downside is if the child
	is behind a modem, and the parent is only receiving data at
	the minimum rate, no useful information is gleaned.

        We calculate the loss rate as below, i.e. 1 - child_bw/parent_bw
	If this is below some threshhold, then, lossrate is treated as 
	zero, and we assume that the rate we receive at is a lower
	bound on the available bandwidth along the path. If loss rate
	is sufficiently high, we treat it as though the available bandwidth
	is being fully tapped along the link.
       *****************/

      float MAX_LOSS_RATE;
      if(parentSendRate > MAX_SOURCE_RATE){
	MAX_LOSS_RATE=MAX_LOSS_RATE_LOW;
      }
      else{
	MAX_LOSS_RATE=MAX_LOSS_RATE_HIGH;
      }
      float lossRate = 1 - rateRcvdFromParent/ (float) parentSendRate; 
      if(lossRate < MAX_LOSS_RATE){
	if(verbosity > 0){
	  cout << "\nReport:" << rateRcvdFromParent << " " 
	       << parentSendRate << "LB";
	}
	routeParent->IntegrateBWResult(rateRcvdFromParent,BW_RESULT_LB);
      }
      else{
	if(verbosity > 0){
	  cout << "\nReport:" << rateRcvdFromParent << " " 
	       << parentSendRate << "Est";
	}
	routeParent->IntegrateBWResult(rateRcvdFromParent, BW_RESULT);
      }
      nbrTabPtr->UpdateLastBWReportTime(fromAddr);
    }
  }
  
  nbrTabPtr->UpdateStatsRcvdFromNbr(fromAddr,msgPtr->GetDataSize()); 

  int source=msgPtr->GetSourceAddr();
  VRTEntryType *routeSource= vrtPtr->GetRoute(source);
  
  if (routeSource == NULL) {
    cerr << "\nRcvd data from unknown source " << GetNameByAddr(source) 
	 << " seqNum " << msgPtr->GetSeqNum() << " discard";
    /* XXX as an optimization, instead of discarding the packet, we
     * could potentially create an entry, and accept and forward the
     * packet 
     */
    return;
  }

  if (! routeSource->IsSource()) {
    routeSource->SetSource();
  }
  
  SourceEntry *sourceEntryPtr = routeSource->GetSourceEntry();
  
  switch(sourceEntryPtr->RecvDataPkt(msgPtr)) {
  case BUF_DUP:
    if(verbosity > 1){  /* set to 1 for now to reduce processor load */
    cout << "\n" << GetCurrTime() 
	 << ":Receiving DUPLICATE Sequence Number " 
	 << msgPtr->GetSeqNum() << " from  " 
	 << GetNameByAddr(source)
	 << " (" << GetNameByAddr(fromAddr) << ") "; 
    }
    return;

  case BUF_LATE:
    return;

  case BUF_OK:
  case BUF_REORDER:
    break;  /* continue processing */
  default:
    assert(0);
  }
 

  if(!routeSource->IsNextHop(fromAddr)){ 
    if(verbosity > 1){
    cout << "\n" << GetCurrTime() << ":Receiving Sequence Number " 
	 << msgPtr->GetSeqNum() << " from  " 
	 << GetNameByAddr(msgPtr->GetSourceAddr())
	 << " (NON-PARENT = " << GetNameByAddr(fromAddr) << ") ";
    }
  }

 /* forward data to children */
  ForwardDataMsg(msgPtr);
  
  sourceEntryPtr->Update(msgPtr, fromAddr);
  
  // Now hand the packet to the application
  gossipAgentPtr->RecvDataForApp((const char *)msgPtr->GetData(),
				 msgPtr->GetDataSize());  
}

int DataAgent::GetMyAddr(){
  return(gossipAgentPtr->GetMyAddr());
}

void DataAgent::DoDebug(){
  if(verbosity > 0){
    cout << "\nSend Summary:Last Sequence number sent:" << seqNum-1 ;
  }
 
  for(VRTEntryType *route=vrtPtr->GetNextRouteInit(); route != NULL; 
      route=vrtPtr->GetNextRoute()) {
    if (! route->IsSource()) continue;
      
    SourceEntry *sourceEntryPtr = route->GetSourceEntry();
    assert(sourceEntryPtr != NULL);
    
    sourceEntryPtr->DoDebug();
  }
}
