#include <assert.h>
#include <iostream.h>

#include <Util/inputSerializableStream.h>
#include <Util/outputSerializableStream.h>

#include "latencyMetric.h"
#include "../vrtEntry.h"


LatencyMetric::LatencyMetric(){
  latency=NO_ROUTE;
}

LatencyMetric::LatencyMetric(int latencyIn){
  latency=latencyIn;
}

void LatencyMetric::SanityChecks(LatencyMetric *latencyMetric){

  assert(
	 latencyMetric->IsNoRoute() ||
	 latencyMetric->IsTempRoute() ||
	 latencyMetric->IsValidRoute()
	 );
}

int LatencyMetric::IsKnown(){
  if(latency >= 0){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::IsUnKnown(){
  if (latency == UNKNOWN_DELAY){
    return(1);
  }
  else{
    return(0);
  }
}

LatencyMetric* LatencyMetric::Merge(RoutingMetric *otherMetric){
  LatencyMetric *other = (LatencyMetric *)(otherMetric);
  
  SanityChecks(this);
  SanityChecks(other);
  
  assert(!this->IsNoRoute());
  assert(!other->IsNoRoute());
  
  int newLatency;
  
  if(
     this->IsTempRoute() ||
     other->IsTempRoute()
     ){
    newLatency=TEMPORARY_ROUTE;
  }
  
  else if(
	  this->IsUnKnown() || 
	  other->IsUnKnown()
	  ){
    newLatency=UNKNOWN_DELAY;
  }
  
  else{
    newLatency=this->latency + other->latency;
  }
  
  LatencyMetric *mergedMetric=new LatencyMetric(newLatency);
  return(mergedMetric);
}


int LatencyMetric::Compare(RoutingMetric *otherMetric){
  LatencyMetric *other= (LatencyMetric *)(otherMetric);
  
  SanityChecks(this);
  SanityChecks(other);
  
  assert(!this->IsNoRoute());
  assert(!other->IsNoRoute());

  if(this->IsTempRoute()){
    if (other->IsTempRoute()){
      return(0);
    }
    else{
      return(1);
    }
  }

  if(this->IsUnKnown()){
    if(other->IsTempRoute()){
      return(-1);
    }
    else if(other->IsUnKnown()){
      return(0);
    }
    else{
      return(1);
    }
  }
  
  if(this->IsKnown()){
    if(!(other->IsKnown())){
      return(-1);
    }
    else{
      if(this->latency < other->latency){
	return(-1);
      }
      else if(this->latency == other->latency){
	return(0);
      }
      else if(this->latency > other->latency){
	return(1);
      }
    }
  }
 
  MyError("Should not reach here!!");
  return(0);


}
 
void LatencyMetric::SetNoRoute(){
  latency=NO_ROUTE;
}

void LatencyMetric::SetTempRoute(){
  latency=TEMPORARY_ROUTE;
}

void LatencyMetric::SetMetric(RoutingMetric *otherMetric){
  LatencyMetric *other = (LatencyMetric *)(otherMetric);
  latency=other->latency;
}

void LatencyMetric::SetBest(){
  latency=0;
}

int LatencyMetric::IsNoRoute(){
  if(latency == NO_ROUTE){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::IsTempRoute(){
  if(latency == TEMPORARY_ROUTE){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::IsValidRoute(){
  if(
     IsKnown() ||
     IsUnKnown()
     ){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::IsBest(){
  return(latency == 0);
}

float 
LatencyMetric::ComputeUtilityGain(RoutingMetric *linkMetricToProbedNode,
				  RoutingMetric *routingMetricToProbedNode){

  LatencyMetric *linkMetric = (LatencyMetric *)(linkMetricToProbedNode);
  LatencyMetric *routeMetric = (LatencyMetric *)(routingMetricToProbedNode);
  
  assert(linkMetric->IsKnown());
  assert(routeMetric->IsValidRoute());
  assert(this->IsValidRoute());
  
  if(
     (this->IsUnKnown()) ||
     (routeMetric->IsUnKnown())
     ){
    return(0);
  }
  
  LatencyMetric *newMetric=routeMetric->Merge(linkMetric);
  float utilityGain=0;
   
  if(
     (newMetric->latency < this->latency) &&
     (this->latency != 0)
     ){
    utilityGain= 1 - ((newMetric->latency)/(float)(this->latency));
  }
  else{
    utilityGain=0;
  }

  delete newMetric;
  return(utilityGain);
}


float LatencyMetric::ComputeLinkQuality(){
  if(!this->IsKnown()){
    return(0);
  }
  
  if(latency <= 10){
    return(1);
  } else if (latency <= 20) {
    return(0.5);
  } else if (latency <= 40) {
    return(0.25);
  } else{
    return(0);
  }
}

void LatencyMetric::PhysicalToRouting(VRTEntryType *vrtEntryPtr){
  int physicalDelay=vrtEntryPtr->GetPhysicalDelay();
  if(physicalDelay < 0){
    latency=UNKNOWN_DELAY;
  }
  else{
    latency=physicalDelay;
  }
}

int LatencyMetric::IsReasonableDelay(){
  if(
     this->IsKnown() &&
     (latency <= REASONABLE_DELAY)
     ){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::IsLowDelay(){
  if(
     this->IsKnown() &&
     (latency <= CLOSE_NBR_DELAY)
     ){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::IsHighDelayPenalty(LatencyMetric *linkMetric){
  float penalty;
  
  if(!this->IsKnown()){
    return(0);
  }
  if(!linkMetric->IsKnown()){
    return(0);
  }
  
  if(linkMetric->latency != 0){
    penalty= latency/(float)linkMetric->latency;
  }
  else{
    penalty=latency;
  }
  
  if(penalty >= HIGH_DELAY_PENALTY){
    return(1);
  }
  else{
    return(0);
  }
}

int LatencyMetric::GetLatency(){
  return(latency);
}
    
LatencyMetric *LatencyMetric::Duplicate(){
  LatencyMetric *newLatencyMetric= new LatencyMetric(latency);
  return(newLatencyMetric);
}

void LatencyMetric::Print(){
  cout << "LatencyMetric: " << latency << " ";
}

int LatencyMetric::Size(){
  return(sizeof(latency));
}

void LatencyMetric::ReadFromStream(InputSerializableStream *osPtr){
  latency=osPtr->ReadInt();

}

void LatencyMetric::WriteToStream(OutputSerializableStream *isPtr){
  isPtr->WriteInt(latency);
}
