#include <assert.h>
#include <iostream.h>

#include <Util/inputSerializableStream.h>
#include <Util/outputSerializableStream.h>

#include "bwMetric.h"
#include "../bwLevelMgr.h"
#include "../vrtEntry.h"
#include "../global.h"

BWLevelMgr *BWMetric::bwLevelMgr = NULL;

void BWMetric::CreateBWLevelMgr(){
  if(bwLevelMgr == NULL){
    bwLevelMgr = new BWLevelMgr(MAX_SOURCE_RATE);
  }
}

BWMetric::BWMetric(){
  CreateBWLevelMgr();
  bw=NO_ROUTE;
  hysteresisFlag=0;
}

BWMetric::BWMetric(int bwIn){
  CreateBWLevelMgr();
  bw=bwIn;
  hysteresisFlag=0;
}

void BWMetric::SanityChecks(BWMetric *bwMetric){

  assert(
	 bwMetric->IsNoRoute() ||
	 bwMetric->IsTempRoute() ||
	 bwMetric->IsValidRoute()
	 );
}

int BWMetric::IsKnown(){

  if(bwLevelMgr->IsValidBW(bw)){
    return(1);
  }
  else{
    return(0);
  }
}

int BWMetric::IsUnKnown(){
  if (bw == UNKNOWN_BW){
    return(1);
  }
  else{
    return(0);
  }
}


BWMetric* BWMetric::Merge(RoutingMetric *otherMetric){
  BWMetric *other = (BWMetric *)(otherMetric);

  SanityChecks(this);
  SanityChecks(other);
  
  assert(!this->IsNoRoute());
  assert(!other->IsNoRoute());
  
  int newBW;
  
  if(
     this->IsTempRoute() ||
     other->IsTempRoute()
     ){
    newBW=TEMPORARY_ROUTE;
  }
  
  else if(
	  this->IsUnKnown() || 
	  other->IsUnKnown()
	  ){
    newBW=UNKNOWN_BW;
  }
  
  else{
    if(this->bw < other->bw) 
      newBW=this->bw;
    else 
      newBW=other->bw;
  }
  
  BWMetric *mergedMetric=new BWMetric(newBW);
  return(mergedMetric);
}


int BWMetric::Compare(RoutingMetric *otherMetric){
  BWMetric *other = (BWMetric *)(otherMetric);

  SanityChecks(this);
  SanityChecks(other);
  
  assert(!this->IsNoRoute());
  assert(!other->IsNoRoute());
  
  if(this->IsTempRoute()){
    if (other->IsTempRoute()){
      return(0);
    }
    else{
      return(1);
    }
  }

  if(this->IsUnKnown()){
    if(other->IsTempRoute()){
      return(-1);
    }
    else if(other->IsUnKnown()){
      return(0);
    }
    else{
      return(1);
    }
  }

  if(this->IsKnown()){
    if(!(other->IsKnown())){
      return(-1);
    }
    else{
      if(this->bw > other->bw){
	return(-1);
      }
      else if(this->bw == other->bw){
	return(0);
      }
      else if(this->bw < other->bw){
	return(1);
      }
    }
  }

  MyError("Should not come here!!!");
  return(0);

}

void BWMetric::SetNoRoute(){
  bw=NO_ROUTE;
}

void BWMetric::SetTempRoute(){
  bw=TEMPORARY_ROUTE;
}

void BWMetric::SetMetric(RoutingMetric *otherMetric){
  BWMetric *other = (BWMetric *)(otherMetric);
  bw=other->bw;
}

void BWMetric::SetBest(){
  bw=bwLevelMgr->GetMaxBWLevel();
}

int BWMetric::IsNoRoute(){
  if(bw == NO_ROUTE){
    return(1);
  }
  else{
    return(0);
  }
}

int BWMetric::IsTempRoute(){
  if(bw == TEMPORARY_ROUTE){
    return(1);
  }
  else{
    return(0);
  }
}

int BWMetric::IsValidRoute(){
  if(
     IsKnown() ||
     IsUnKnown()
     ){
    return(1);
  }
  else{
    return(0);
  }
}

int BWMetric::IsBest(){
  return(bw == bwLevelMgr->GetMaxBWLevel());
}

float BWMetric::ComputeUtilityGain(RoutingMetric *linkMetricToProbedNode,
				   RoutingMetric *routingMetricToProbedNode){

  BWMetric *linkMetric = (BWMetric *)(linkMetricToProbedNode);
  BWMetric *routeMetric = (BWMetric *)(routingMetricToProbedNode);
  
  assert(linkMetric->IsKnown());
  assert(routeMetric->IsValidRoute());
  assert(this->IsValidRoute());
  
  if(routeMetric->IsUnKnown()){
    return(0);
  }
  
  BWMetric *newMetric=routeMetric->Merge(linkMetric);
  float utilityGain=0;
  
  int compareBW=Compare(newMetric);
  if(compareBW > 0){
    return(1);
  }
  else{
    return(0);
  }
  
  delete newMetric;
  return(utilityGain);
}

float BWMetric::ComputeLinkQuality(){
  if(!this->IsKnown()){
    return(0);
  }

  if(bw >= bwLevelMgr->GetMaxBWLevel()){
    return(1);
  }
  else if (bw >= (bwLevelMgr->GetMaxBWLevel()/2)){
    return(0.5);
  }
  else{
    return(0);
  }
}

void BWMetric::PhysicalToRouting(VRTEntryType *vrtEntryPtr){

  int physicalBW=vrtEntryPtr->GetPhysicalBW();
  int numBWResultProbes=vrtEntryPtr->GetNumBWResultProbes();
  
  if(physicalBW < 0){
    bw=UNKNOWN_BW;
    return;
  }

  else if(hysteresisFlag == 0){
    int level=bwLevelMgr->ComputeLevelGivenBW(physicalBW);
    bw=bwLevelMgr->ComputeBWGivenLevel(level);

    if(numBWResultProbes == 1){
      hysteresisFlag=1;
    }
    return;
  }

  else{
    assert(this->IsKnown());
    bw=ComputeNewRoutingBW(physicalBW);
    return;
  }
}



BWMetric *BWMetric::Duplicate(){
  BWMetric *newBWMetric= new BWMetric(bw);
  return(newBWMetric);
}

void BWMetric::Print(){
  cout << "BWMetric: " << bw << " ";
}

int BWMetric::Size(){
  return(sizeof(bw));
}

int BWMetric::GetBW(){
  return(bw);
}

void BWMetric::ReadFromStream(InputSerializableStream *osPtr){
  bw=osPtr->ReadInt();
}

void BWMetric::WriteToStream(OutputSerializableStream *isPtr){
  isPtr->WriteInt(bw);
}


int BWMetric::ComputeNewRoutingBW(int physicalBW){
  /*********************
    Let LA1 be level of old advertised bw 
    Let LP be level of new physical bw.

    We need to compute LA, the new advertised bw

    If LP <= LA1, LA = LP. (if bw seems to have decreased, rush
                           down a level)
    If LP > LA1:
     let newLevel=LA1+1;
     if (physicalBW > 
        [bw_of_newLevel + (BETA) * (bw_of_newLevel+1 - bw_of_newLevel)])
	 then LA = newLevel
	
     [reasoning: if bw has increased, go up, 
                 but only by one level at a time]
  
  ************************/

  int levelRoutingBW;
  int levelPhyBW;

  levelRoutingBW=bwLevelMgr->ComputeLevelGivenBW(bw);
  levelPhyBW=bwLevelMgr->ComputeLevelGivenBW(physicalBW);
  
  if(levelPhyBW <= levelRoutingBW){
    return(bwLevelMgr->ComputeBWGivenLevel(levelPhyBW));
  }
  else{
    // Is the new estimate of physical bandwidth sufficiently 
    // up the threshold?
    
    int bw1,bw2;
    
    bw1=bwLevelMgr->ComputeBWGivenLevel(levelRoutingBW+1);
    bw2=bwLevelMgr->ComputeBWGivenLevel(levelRoutingBW+2);
    if(bw1 == bw2){
      /*********
	Must have hit the max level.  
      **********/
      assert(bw1 == bwLevelMgr->GetMaxBWLevel());
       if(((physicalBW - bw1)/(float) bw1) > BETA_MAX){
	return(bwLevelMgr->ComputeBWGivenLevel(levelRoutingBW+1));
      }
      else{
	return(bwLevelMgr->ComputeBWGivenLevel(levelRoutingBW));
      }
    }
    else{
      if (((physicalBW - bw1)/ (float) (bw2 - bw1)) > BETA){
	return(bwLevelMgr->ComputeBWGivenLevel(levelRoutingBW+1));
      }
      else{
	return(bwLevelMgr->ComputeBWGivenLevel(levelRoutingBW));
      }
    }
  }
}
