#include <assert.h>
#include <iostream.h>

#include <Util/inputSerializableStream.h>
#include <Util/outputSerializableStream.h>

#include "../vrtEntry.h"
#include "latencyMetric.h"
#include "bwMetric.h"
#include "bwLatencyMetric.h"

void BWLatencyMetric::SanityCheck(BWLatencyMetric *bwLatencyMetric){

  assert(
	 bwLatencyMetric->IsNoRoute() ||
	 bwLatencyMetric->IsTempRoute() ||
	 bwLatencyMetric->IsValidRoute()
	 );
}

BWLatencyMetric::BWLatencyMetric(){
  bwMetric = new BWMetric();
  latencyMetric = new LatencyMetric();
}

BWLatencyMetric::BWLatencyMetric(BWMetric *bwMetricIn,
				 LatencyMetric *latencyMetricIn){
  latencyMetric=latencyMetricIn;
  bwMetric=bwMetricIn;
}

BWLatencyMetric::~BWLatencyMetric(){
  delete latencyMetric;
  delete bwMetric;
}

BWLatencyMetric *BWLatencyMetric::Merge(RoutingMetric *otherMetric){
  BWLatencyMetric *other = (BWLatencyMetric *) otherMetric;

  SanityCheck(this);
  SanityCheck(other);
  
  assert(!this->IsNoRoute());
  assert(!other->IsNoRoute());
  

  LatencyMetric *newLatencyMetric=
    latencyMetric->Merge(other->latencyMetric);
  
  BWMetric *newBWMetric=
    bwMetric->Merge(other->bwMetric);
  
  BWLatencyMetric *newBWLatencyMetric = 
    new BWLatencyMetric(newBWMetric, newLatencyMetric);
  
  return(newBWLatencyMetric);
}


int BWLatencyMetric::Compare(RoutingMetric *otherMetric){
  BWLatencyMetric *other = (BWLatencyMetric *) otherMetric;
  
  SanityCheck(this);
  SanityCheck(other);
  
  assert(!this->IsNoRoute());
  assert(!other->IsNoRoute());
  
  
  int bwCompare=bwMetric->Compare(other->bwMetric);
  if(bwCompare != 0){
    return(bwCompare);
  }
  else{
    // bandwidths are equal
    int latencyCompare=latencyMetric->Compare(other->latencyMetric);
    return(latencyCompare);
  }
}

int BWLatencyMetric::GetBW(){
  return(bwMetric->GetBW());
}

int BWLatencyMetric::GetLatency(){
  return(latencyMetric->GetLatency());
}

void BWLatencyMetric::SetNoRoute(){
  bwMetric->SetNoRoute();
  latencyMetric->SetNoRoute();
}

void BWLatencyMetric::SetTempRoute(){
  bwMetric->SetTempRoute();
  latencyMetric->SetTempRoute();
}

void BWLatencyMetric::SetMetric(RoutingMetric *otherMetric){
  BWLatencyMetric *other = (BWLatencyMetric *)otherMetric;
  latencyMetric->SetMetric(other->latencyMetric);
  bwMetric->SetMetric(other->bwMetric);
}


void BWLatencyMetric::SetBest(){
  bwMetric->SetBest();
  latencyMetric->SetBest();
}

int BWLatencyMetric::IsNoRoute(){
  assert(bwMetric->IsNoRoute() == latencyMetric->IsNoRoute());
  return(bwMetric->IsNoRoute());
}

int BWLatencyMetric::IsTempRoute(){
  assert(bwMetric->IsTempRoute() == latencyMetric->IsTempRoute());
  return(bwMetric->IsTempRoute());
}

int BWLatencyMetric::IsValidRoute(){
  assert(bwMetric->IsValidRoute() == latencyMetric->IsValidRoute());
  return(bwMetric->IsValidRoute());
}

int BWLatencyMetric::IsBest(){
  return(
	 (bwMetric->IsBest()) &&
	 (latencyMetric->IsBest())
	 );
}


float BWLatencyMetric::ComputeUtilityGain(RoutingMetric *linkMetricToProbedNode,
					  RoutingMetric *routingMetricToProbedNode){

  BWLatencyMetric *linkMetric = (BWLatencyMetric *)(linkMetricToProbedNode);
  BWLatencyMetric *routeMetric = (BWLatencyMetric *)(routingMetricToProbedNode);

  assert(linkMetric->IsValidRoute());
  assert(routeMetric->IsValidRoute());
  assert(this->IsValidRoute());

  BWMetric *linkBWMetric=linkMetric->bwMetric;
  BWMetric *routeBWMetric=routeMetric->bwMetric;

  /*********
       If the link bandwidth metric is not known, then,
       we treat this as a latency only metric
  **********/

  if(linkBWMetric->IsKnown()){

    if(routeBWMetric->IsUnKnown()){
      return(0);
    }
    
    BWMetric *newBWMetric=routeBWMetric->Merge(linkMetric->bwMetric);
    int bwCompare=bwMetric->Compare(newBWMetric);
    delete newBWMetric;
    
    if(bwCompare < 0){
      // new bandwidth is worse
      return(0);
    }
    else if(bwCompare > 0){
      // new bandwidth is better
      return(1);
    }
    else{
      // bandwidth remains the same. utility gain is 
      // entirely due to latency
      return(latencyMetric->ComputeUtilityGain(linkMetric->latencyMetric,
					       routeMetric->latencyMetric));
    }
  }
  
  else{
    // Link Bandwidth not known. Treat as a Latency-Only Metric
    return(latencyMetric->ComputeUtilityGain(linkMetric->latencyMetric,
					     routeMetric->latencyMetric));
  }
}

float BWLatencyMetric::ComputeLinkQuality(){

  // Equal weightage to bandwidth and latency 

  float latencyQuality=latencyMetric->ComputeLinkQuality();
  float bwQuality=bwMetric->ComputeLinkQuality();
  
  float quality=((latencyQuality+bwQuality)/2.0);
  return(quality);
}

void BWLatencyMetric::PhysicalToRouting(VRTEntryType *vrtEntryPtr){
  bwMetric->PhysicalToRouting(vrtEntryPtr);
  latencyMetric->PhysicalToRouting(vrtEntryPtr);
}


void BWLatencyMetric::SetBestBW(){
  assert(latencyMetric->IsValidRoute());
  bwMetric->SetBest();
}

int BWLatencyMetric::IsBestBW(){
  return(bwMetric->IsBest());
}
	 
int BWLatencyMetric::IsReasonableDelay(){
  return(latencyMetric->IsReasonableDelay());
}

int BWLatencyMetric::IsLowDelay(){
  return(latencyMetric->IsLowDelay());
}

int BWLatencyMetric::IsHighDelayPenalty(BWLatencyMetric *linkMetric){
  return(latencyMetric->IsHighDelayPenalty(linkMetric->latencyMetric));
}


BWLatencyMetric *BWLatencyMetric::Duplicate(){
  BWMetric *newBWMetric=bwMetric->Duplicate();
  LatencyMetric *newLatencyMetric=latencyMetric->Duplicate();
  BWLatencyMetric *newBWLatencyMetric= new BWLatencyMetric(newBWMetric,newLatencyMetric);
  return(newBWLatencyMetric);
}

void BWLatencyMetric::Print(){
  bwMetric->Print();
  latencyMetric->Print();
}

int BWLatencyMetric::Size(){
  return(
	 bwMetric->Size() +
	 latencyMetric->Size()
	 );
}

void BWLatencyMetric::ReadFromStream(InputSerializableStream *osPtr){
  bwMetric->ReadFromStream(osPtr);
  latencyMetric->ReadFromStream(osPtr);
}

void BWLatencyMetric::WriteToStream(OutputSerializableStream *isPtr){
  bwMetric->WriteToStream(isPtr);
  latencyMetric->WriteToStream(isPtr);
}
