#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/shm.h>
#include <sys/param.h>
#include <assert.h>

#include "macros.h"
#include "packetBuffer.h"

// #define DEBUG_THIS

PacketBuffer::PacketBuffer(int maxQSize) {
  assert(maxQSize > 0);

  this->maxQSize = maxQSize;
  firstPtr = NULL;
  lastPtr = NULL;
  qSize = 0;
}


PacketBuffer::~PacketBuffer() {
  while (firstPtr != NULL) {
    SndBuf *tmp = firstPtr;
    firstPtr = firstPtr->next;
    free(tmp);
  }
}


/* if buffer is full, drop packets with the lowest priority and return -1
 * else return 0
 */
int PacketBuffer::Enqueue(const char *buf, int bufsize, int priority) {
  if (bufsize > MAX_UDP_PKTSIZE) {
    fprintf(stderr, "PacketBuffer: packet size %d exceeds max bufsize %d\n",
	    bufsize, MAX_UDP_PKTSIZE);
    assert(bufsize <= MAX_UDP_PKTSIZE);
  }

  // create a new sndbuf
  struct SndBuf *newBuf = (struct SndBuf *)(malloc(sizeof(struct SndBuf)));
  newBuf->bufsize = bufsize;
  newBuf->priority = priority;
  memcpy(newBuf->buf, buf, bufsize);
  newBuf->next = NULL;

  // insert to lastPtr
  if (firstPtr == NULL) {
    firstPtr = newBuf;
    lastPtr = newBuf;
    newBuf->prev = NULL;
  } else {
    newBuf->prev = lastPtr;
    lastPtr->next = newBuf;
    lastPtr = newBuf;
  }

  qSize++;
  if (qSize > maxQSize) {
    DropLowestPriority();
    CheckStatus();
    assert(qSize == maxQSize);
    return -1;
  } else {
    CheckStatus();
    assert(qSize <= maxQSize && qSize > 0);
    return 0;
  }
}

/*
 * Drop packets with the lowest priority
 * - among packets of same priority, drop head
 */
void PacketBuffer::DropLowestPriority() {
  // return immediate if no pkt in queue
  assert(qSize > 0);
  
  struct SndBuf *lowPtr = firstPtr;
  
  // scan from head of the queue
  for (SndBuf *curPtr = firstPtr; curPtr != NULL; curPtr = curPtr->next) {
    if (curPtr->priority < lowPtr->priority) {
      lowPtr = curPtr;
    }
  }
  
  if (verbosity > 1) {
    printf("\nPB: drop a packet of len %d priority %d due to sndbuf overflow",
	   lowPtr->bufsize, lowPtr->priority);
  }

  Dequeue(lowPtr);
}


/* returns 0 if success, -1 if buffer is empty */
int PacketBuffer::Peek(char **buf, int *bufsize) {
  CheckStatus();

  if (qSize == 0) {
    if (verbosity > 0) {
      printf("\nbuffer empty");
    }
    return -1;
  }
  
  if (buf != NULL) *buf = firstPtr->buf;
  if (bufsize != NULL) *bufsize = firstPtr->bufsize;
  return 0;
}


/* returns 0 if success, -1 if buffer is empty */
int PacketBuffer::Dequeue(char **buf, int *bufsize) {
  if (Peek(buf, bufsize) < 0) return -1;
  
  Dequeue(firstPtr);
  assert(qSize < maxQSize && qSize >= 0);

  CheckStatus();
  return 0;
}


void PacketBuffer::Dequeue(struct SndBuf *curPtr) {
  // dequeue pkt in lowPtr
  if (curPtr->prev != NULL) curPtr->prev->next = curPtr->next;
  if (curPtr->next != NULL) curPtr->next->prev = curPtr->prev;

  if (firstPtr == curPtr) firstPtr = curPtr->next;
  if (lastPtr == curPtr) lastPtr = curPtr->prev;

  // free memory
  free(curPtr);
  qSize--;  assert(qSize >= 0);
}


void PacketBuffer::CheckStatus() {
#ifdef DEBUG_THIS
  // reasonable size
  assert(qSize >= 0 && qSize <= maxQSize);

  // match size from head
  int size = 0;
  for (SndBuf *curPtr = firstPtr; curPtr != NULL; curPtr = curPtr->next) {
    size++;
  }
  assert(size == qSize);

  // match size from tail
  size = 0;
  for (SndBuf *curPtr = lastPtr; curPtr != NULL; curPtr = curPtr->prev) {
    size++;
  }
  assert(size == qSize);

  if (size == 0) {
    assert(firstPtr == NULL && lastPtr == NULL);
  } else {
    assert(firstPtr->prev == NULL);
    assert(lastPtr->next == NULL);

    if (size == 1) {
      assert(firstPtr == lastPtr);
    }
  }
#endif
}


int PacketBuffer::Size() {
  return qSize;
}


int PacketBuffer::IsFull() {
  return (qSize == maxQSize);
}


int PacketBuffer::IsEmpty() {
  return (qSize == 0);
}

