#include <sys/param.h>
#include <sys/socket.h>
#include <unistd.h>

#include <errno.h>
#include <stdio.h>
#include <iostream.h>
#include <netdb.h>
#include <assert.h>
#include <string.h>
#include <strings.h>
#include <stdarg.h>
#include <stdlib.h>

#include "inetMisc.h"
#include <sys/resource.h>  /* SUN rlimit */

#define MAXBUFENTRY 50

char *MyHostName = NULL;

int IsMulticastAddr(int addr) {
  int low = 0xe0000000;      /* 224.0.0.0 */
  int high = 0xefffffff;     /* 239.255.255.255 */

  return (addr >= low && addr <= high);
}

int SetsockNonBlocking(int fd) {
  int flags = Fcntl(fd, F_GETFL, 0);
  return Fcntl(fd, F_SETFL, flags | O_NONBLOCK);
}

int SetsockoptBufSize(int fd, int bufsize) {

  if (verbosity > 1) {
    int val;
    socklen_t optlen = sizeof(int);

    Getsockopt(fd, SOL_SOCKET, SO_RCVBUF, &val, &optlen);
    printf("\nRCVBUF was %d Bytes", val);

    Getsockopt(fd, SOL_SOCKET, SO_SNDBUF, &val, &optlen);
    printf("\nSNDBUF was %d Bytes", val);
  }

  /* should be at least 1 MTU size */
  assert(bufsize >= 1500 && bufsize < 65536);

  Setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &bufsize, sizeof(int));
  Setsockopt(fd, SOL_SOCKET, SO_SNDBUF, &bufsize, sizeof(int));

  if (verbosity > 1) {
    int val;
    socklen_t optlen = sizeof(int);
    
    Getsockopt(fd, SOL_SOCKET, SO_RCVBUF, &val, &optlen);
    printf("\nRCVBUF set to %d, now is %d Bytes", bufsize, val);
    
    /* some OS (Sun) set to the closest page size?, remove assert */
    /* assert(val == bufsize); */
    
    Getsockopt(fd, SOL_SOCKET, SO_SNDBUF, &val, &optlen);
    printf("\nSNDBUF set to %d, now is %d Bytes", bufsize, val);
    
    /* some OS (Sun) set to the closest page size?, remove assert */
    /* assert(val == bufsize); */
  }

  return 0;
}


/* addr and port are in host byte order */
struct sockaddr_in GetSockaddr(int addr, int port) {
  struct sockaddr_in sin;
  
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = htonl(addr);
  sin.sin_port = htons(port);
  
  return sin;
}

int GetFDWID() {
  int fd_wid;

#ifdef sun
  struct rlimit rlimit;
  getrlimit(RLIMIT_NOFILE, &rlimit);
  fd_wid = (int)rlimit.rlim_cur;
  return fd_wid;
#endif

#ifdef win   // XXX cygwin !!!
  fd_wid = 64;
  return fd_wid;
#endif

  fd_wid = getdtablesize();  /* max number of files a process can open. */ 
  return fd_wid;
}

int SetFDs(fd_set *fds, int fd_arr[], int size) {
  
  FD_ZERO(fds);

  for (int i=0; i<size; i++) {
    FD_SET(fd_arr[i], fds);
  }
  return GetFDWID();
}

char *getInetAddrIP(int addr) {
  static char out[MAXBUFENTRY][sizeof("xxx.xxx.xxx.xxx")+1];
  static int entry = 0;
  
  entry = (entry + 1) % MAXBUFENTRY;
  sprintf(out[entry], "%u.%u.%u.%u%c", (addr & 0xFF000000) >> 24, 
	  (addr & 0xFF0000) >> 16, (addr & 0xFF00) >> 8, addr & 0xFF, '\0');
  return out[entry];
}

/* return host byte order */
int getInetAddrByName(char *name) {
  struct hostent *host;

  /* cache */
  static char HostNameArr[MAXBUFENTRY][MAXHOSTNAMELEN];
  static int HostIPArr[MAXBUFENTRY];
  static int size=0;
  static int nextPtr=0;
  
  assert(strlen(name) <= MAXHOSTNAMELEN);

  /* init cache */
  if (size == 0) {
    bzero((char *)HostIPArr, MAXBUFENTRY*sizeof(int));
    bzero((char *)HostNameArr, MAXBUFENTRY*MAXHOSTNAMELEN);
  }

  /* access cache */
  for (int i=0; i<size; i++) {
    if (strcmp(name, HostNameArr[i]) == 0) {
      assert(HostIPArr[i] != 0);
      //fprintf(stderr, "serve from cache %d\n", i);
      return HostIPArr[i];
    }
  }

  /* query */
  if ((host = gethostbyname(name)) == NULL) {
    printf("\nhostname %s not found", name);
    return -1;
  }
  int ip = htonl(*(int *)host->h_addr);
  
  /* update cache on successful query, using FIFO */
  strcpy(HostNameArr[nextPtr], name);
  HostIPArr[nextPtr] = ip;
  nextPtr = (nextPtr+1) % MAXBUFENTRY;
  size = MIN(size+1, MAXBUFENTRY);
  
  return ip;
}


/* input addr: host byte order */
char *getInetAddrName(int addr) {
  static struct in_addr iaddr;
  static struct hostent *host;

  /* put it in buffer in case content of host gc 
   * also use it for cache
   */
  static char HostNameArr[MAXBUFENTRY][MAXHOSTNAMELEN];    
  static int nextPtr = 0;
  static int size = 0;
  static int HostIPArr[MAXBUFENTRY];

  /* init cache */
  if (size == 0) {
    bzero((char *)HostIPArr, MAXBUFENTRY*sizeof(int));
    bzero((char *)HostNameArr, MAXBUFENTRY*MAXHOSTNAMELEN);
  }

  /* access cache */
  for (int i=0; i<size; i++) {
    if (HostIPArr[i] == addr) {
      assert(strlen(HostNameArr[i]) > 0);
      // fprintf(stderr, "serve from cache %d\n", i);
      return HostNameArr[i];
    }
  }
  
  /* query */
  iaddr.s_addr = htonl(addr);
  char *hostname;
  if ((host = gethostbyaddr((char *)&iaddr, sizeof(iaddr), AF_INET)) == NULL) {
    // herror("gethostbyaddr");
    hostname = getInetAddrIP(addr);
  } else {
    hostname = (char *)(host->h_name);
  }

  /* update cache */
  assert(sizeof(hostname) <= MAXHOSTNAMELEN);
  strcpy(HostNameArr[nextPtr], hostname);
  HostNameArr[nextPtr][strlen(hostname)] = '\0';
  HostIPArr[nextPtr] = addr;
  
  nextPtr = (nextPtr + 1) % MAXBUFENTRY;
  size = MIN(size+1, MAXBUFENTRY);
  return hostname;
}


char toPrintableChar(const u_char value) {
  if (value >= 127 || value <= 31)
    return '?';
  else 
    return value;
}

char *buf2ReadableStr(const char *buf, int bufLen) {
  int lastZeroPos;
  static char internal_buf[1500];
  
  bzero((char *)internal_buf, 1500);
  
  for (lastZeroPos=(bufLen-1); lastZeroPos>=0; lastZeroPos--) {
    if (buf[lastZeroPos] != 0) break;
  }
  
  for (int i=0; i<lastZeroPos; i++) {
    internal_buf[i] = toPrintableChar(((u_char *)buf)[i]);
  }
  return internal_buf;
}

int toPrintableChar2(const u_char value, char *buf, int offset) {
  sprintf(buf+offset, "%d ", value);
  if (value < 10) return offset+2;
  else if (value < 100) return offset+3;
  else return offset+4;
}

char *buf2ReadableStr2(const char *buf, int bufLen) {
  static char internal_buf[1500];
  int lastZeroPos;
  
  bzero((char *)internal_buf, 1500);

  for (lastZeroPos=(bufLen-1); lastZeroPos>=0; lastZeroPos--) {
    if (buf[lastZeroPos] != 0) break;
  }
  
  int offset = 0;
  for (int i=0; i<lastZeroPos; i++) {
    offset = toPrintableChar2(buf[i], internal_buf, offset);
  }
  // if (lastZeroPos == (bufLen-1)) internal_buf[bufLen] = '\0';
  return internal_buf;
}


// extern int gethostname(char *name, int namelen);
// return a host name in host byte order
int GetMyAddr() {
  static int hostaddr = 0;  /* cache hostaddr */
  struct hostent *host;
  
  if (hostaddr == 0) {
    
    if (MyHostName != NULL) {
      host = gethostbyname(MyHostName);
      hostaddr = *(int *)host->h_addr;
      hostaddr = htonl(hostaddr);
    } else {
      
#ifdef sun2
      host = gethostent();
#else
      char hostname[MAXHOSTNAMELEN];
      
      gethostname(hostname, MAXHOSTNAMELEN);
      host = gethostbyname(hostname);
#endif
      hostaddr = *(int *)host->h_addr;
      hostaddr = htonl(hostaddr);
    }
    if(hostaddr == 0x7F000001){
      cerr << "MyHostName: " << MyHostName;
      cerr << "Host address is 127.0.0.1 !";
    }
  }
  return hostaddr;
}


int Socket(int domain, int type, int protocol) {
  int retval;

  if ((retval = socket(domain, type, protocol)) < 0) {
    perror("socket");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}

int Bind(int sockfd, struct sockaddr *my_addr, int addrlen) {
  int retval;
  errno = 0;

  if ((retval = bind(sockfd, my_addr, addrlen)) < 0) {
    perror("bind");

    int addr = htonl(((struct sockaddr_in *)my_addr)->sin_addr.s_addr);
    int port = htons(((struct sockaddr_in *)my_addr)->sin_port);
    fprintf(stderr, "\n error when bind() to %s %d\n", 
	    getInetAddrIP(addr), port);
    assert(0);
  }

  int addr = htonl(((struct sockaddr_in *)my_addr)->sin_addr.s_addr);
  int port = htons(((struct sockaddr_in *)my_addr)->sin_port);
  //fprintf(stderr, "\n bind() to %s %d OK\n", 
  //getInetAddrIP(addr), port);
  return retval;
}

int Sendto(int s, const char *msg, int len,  unsigned  int flags, 
	   const struct sockaddr *to, int tolen) {
  int retval;

  if (ETBSendDrop(htonl(((struct sockaddr_in *)to)->sin_addr.s_addr))==TRUE) {
    return retval;
  }

  if ((retval = sendto(s, msg, len, flags, to, tolen)) < 0) {
    perror("sendto");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}

/* return -1 if failed
 *    and print error message
 * else return # of bytes transmitted
 *    and it will assert if (bytes transmitted != len)
 *
 * Intended to use with non-blocking UDP socket
 */
int Sendto2(int s, const char *msg, int len,  unsigned  int flags, 
	    const struct sockaddr *to, int tolen, const char *errMsg) {
  int retval;
  errno = 0;

  /* emulate transient behavior: drop pkts probabilistically */
  if (ETBSendDrop(htonl(((struct sockaddr_in *)to)->sin_addr.s_addr))==TRUE) {
    return retval;
  }


  if ((retval = sendto(s, msg, len, flags, to, tolen)) < 0) {
    switch(errno) {
    case EAGAIN:  
      /* Resource temporarily unavailable (11), likely due to 
       * interface buffer full (?)
       */
      break;
    default:
      perror("sendto");
      fprintf(stderr, "%s: %s %d\n", errMsg, getInetAddrName(GetMyAddr()), 
	      errno);
      printf("Sendto %s: %s %d\n", errMsg, getInetAddrName(GetMyAddr()), 
	     errno);
    }
  } else {
    /* ensure no fragmentation */
    assert(retval == len);
  }

  return retval;
}


int Connect(int s, struct sockaddr *serv_addr, socklen_t addrlen) {
  int retval;

  if ((retval = connect(s, serv_addr, addrlen)) < 0) {
    perror("connect");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int Accept(int s, struct sockaddr *addr, socklen_t *addrlen) {
  int retval;

  if ((retval = accept(s, addr, addrlen)) < 0) {
    perror("accept");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int Listen(int s, int backlog){
  int retval;

  if ((retval = listen(s, backlog)) < 0) {
    perror("listen");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int Recv(int s, void *buf, int len, unsigned int flags) {
  int retval;

  if ((retval = recv(s, buf, len, flags)) < 0) {
    perror("recv");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int Send(int  s,  const  void *msg, int len, unsigned int flags) {
  int retval;

  assert(msg != NULL);
  assert(len > 0);

  if ((retval = send(s, msg, len, flags)) < 0) {
    perror("send");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}

int Recvfrom(int s, void *buf, int len, unsigned int flags,
	     struct sockaddr *from, socklen_t *fromlen) {
  int retval;
  errno = 0;
  
  assert(buf != NULL);
  assert(len > 0);

  if ((retval = recvfrom(s, (char *)buf, len, flags, from, fromlen)) < 0) {
    perror("recvfrom");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int Recvfrom2(int s, void *buf, int len, unsigned int flags,
	      struct sockaddr *from, socklen_t *fromlen) {
  int retval;
  errno = 0;

  assert(buf != NULL);
  assert(len > 0);
  
  retval = recvfrom(s, (char *)buf, len, flags, from, fromlen);

  return retval;
}



int Getsockopt(int s, int level, int optname, void *optval, socklen_t *optlen) {
  int retval;

  if ((retval = getsockopt(s, level, optname, optval, optlen)) < 0) {
    perror("getsockopt");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int Setsockopt(int s, int level, int optname, const void *optval, int optlen) {
  int retval;

  if ((retval = setsockopt(s, level, optname, optval, optlen)) < 0) {
    perror("setsockopt");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


int SetsockoptReuseAddrPort(int fd) {
  int opt = 1;

#ifdef SO_REUSEPORT
  Setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&opt, sizeof(opt));
#endif
  return Setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&opt, sizeof(opt));
}


int Getsockname(int s, struct sockaddr * name , socklen_t * namelen){
  int retval;

  if ((retval = getsockname(s,name,namelen) < 0)) {
    perror("getsockname");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}

int Fcntl(int fd, int cmd, long arg) {
  int retval;

  if ((retval = fcntl(fd, cmd, arg)) < 0) {
    perror("fcntl");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}


// extern int gethostname(char *name, int namelen);
// return a host name in host byte order
int IsMyAddr(int addr) {
  struct hostent *host;
  int i;

#ifdef sun2
    host = gethostent();
#else
    char hostname[MAXHOSTNAMELEN];

    gethostname(hostname, MAXHOSTNAMELEN);
    host = gethostbyname(hostname);
#endif
    for (i=0; host->h_addr_list[i] != NULL; i++){
      if(*(int *)host->h_addr_list[i] == addr){
	return(1);
      }
    }
    return(0);
}

int Select(int  n,  fd_set  *readfds,  fd_set  *writefds,
	   fd_set *exceptfds, struct timeval *timeout) {
  int retval;

  if ((retval = select(n, readfds, writefds, exceptfds, timeout)) < 0) {
    perror("select");
    assert(! "\n core dump due to socket error");
  }
  return retval;
}
