#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>
#include <netdb.h>
#include <sys/time.h>

#include "macros.h"
#include "estimateBandwidth.h"
#include "misc.h"

/* both in unit of seconds */
EstimateBandwidth::EstimateBandwidth(int estBwHistTime, int estBwUnitTime) {
  assert(estBwUnitTime > 0);
  assert(estBwHistTime > 0);
  assert(estBwHistTime % estBwUnitTime == 0);

  this->estBwUnitTime = estBwUnitTime;
  this->estBwHistTime = estBwHistTime;
  estBwUnitUs = estBwUnitTime * MICROSEC;

  startTime = getCurrTimeUs();
  nextTime = getCurrTimeUs() + estBwUnitUs;
  byteCnt = 0;
  
  statSize = 0;
  nextStat = 0;
  stat = new int[estBwHistTime];
  for (int i=0; i< estBwHistTime; i++) {
    stat[i] = 0;
  }

  ReportSeqNum = 0;
  totalByteCnt = 0;
}


EstimateBandwidth::~EstimateBandwidth() {
  delete [] stat;
}


int EstimateBandwidth::Report() {
  int sum = 0;
  
  Update(0);

  if (statSize < estBwHistTime) 
    return (NO_REPORT_AVAILABLE);

  for (int i=0; i<statSize; i++) {
    sum += stat[i];
  }
  
  int bytesInLastPeriod = sum + this->byteCnt;
  int bitsInLastPeriod = bytesInLastPeriod * 8;
  double lenOfLastPeriod = (estBwUnitTime * statSize) +  // seconds
    ((double)(getCurrTimeUs()-(this->nextTime-estBwUnitUs))/(double)MICROSEC);
  int rateInKbps = (int)((double)bitsInLastPeriod / lenOfLastPeriod / 1024.0);
  return rateInKbps;
}


/* returns 0 if session duration is less than 1 second
 */
int EstimateBandwidth::ReportSessionBw() {
  double dur = (getCurrTimeUs() - startTime)*1.0/MICROSEC;
  if (dur < 1) { return 0; }
  return (int)(totalByteCnt/1024*8 / dur);
}

int EstimateBandwidth::GetTotalByteCntInKB() {
  return totalByteCnt/1024;
}

void EstimateBandwidth::Update(int bytes) {
  UpdateInternal(bytes, FALSE);
}


/* returns a linked list of bandwidth report in a sorted
 * order of sequence number.  Called by both Update and UpdateAndReport
 */
struct EBReportList *EstimateBandwidth::UpdateAndReport(int bytes) {
  return UpdateInternal(bytes, TRUE);
}

struct EBReportList *EstimateBandwidth::UpdateInternal(int bytes, 
						       int isReport) {
  EBReportList *reportList = NULL;
  EBReportList *tailList = NULL;
  
  /* estimate status */
  int64 cur_time = getCurrTimeUs();
  for (;;) {

    if (cur_time - this->nextTime < 0) {
      break;
    }

    /* insert a report into the tail of the linked list */
    if (isReport) {
      EBReportList *newReport = 
	(struct EBReportList *)malloc(sizeof(EBReportList));
      newReport->time = (long)(nextTime/1000);
      newReport->seq = ReportSeqNum;
      newReport->byteCnt = byteCnt;
      newReport->next = NULL;
    
      if (reportList == NULL) {
	reportList = newReport;
	tailList = newReport;
      } else {
	tailList->next = newReport;
	tailList = newReport;
      }
    }


    ReportSeqNum++;
    stat[nextStat] = this->byteCnt;
    
    nextStat = (nextStat+1) % estBwHistTime;
    if (statSize < estBwHistTime) statSize++;
    
    this->byteCnt = 0;
    this->nextTime += estBwUnitUs;
  }
  
  this->byteCnt += bytes;
  this->totalByteCnt += bytes;

  return reportList;
}


#if 0

#include <Util/misc.h>
#include <Util/inetMisc.h>
int verbosity;

int main(int argc, char *argv[]) {
  EstimateBandwidth *bw = new EstimateBandwidth(10);

  for (int i=0; i<100000; i++) {
    long wait = 100;
    struct timeval wait_time = MsecToTimeval(wait);
    Select(1, (fd_set *)NULL, (fd_set *)NULL, (fd_set *)NULL, &wait_time);
    
    bw->Update(100000);  /* 8000 Kbps */

    if (i % (1000/wait) == 0) { 
      printf("%ld, %d\n", i / (1000/wait), bw->Report());
    }
  }
}

#endif
