#include "config_unix.h"
#include "config_win32.h"

#include <Util/inetMisc.h>
#include <Util/misc.h>
#include "tfrcDaemon.h"
#include <strings.h>

void TfrcDaemon::Init(int port) {
  fd = Socket(AF_INET, SOCK_DGRAM, 0);
  SetsockoptReuseAddrPort(fd);
  SetsockNonBlocking(fd);
  
  assert(port != 0);
  this->laddr = INADDR_ANY;     /* any addr */
  this->lport = port;           /* any port */
  
  struct sockaddr_in sin = GetSockaddr(this->laddr, this->lport);
  Bind(this->fd, (struct sockaddr *)&sin, sizeof(sin));
  
  this->firstPos = this->lastPos = 0;

  this->timer = NULL;
}



TfrcDaemon::TfrcDaemon(int port) {
  Init(port);
}


TfrcDaemon::TfrcDaemon(int port, int64 maxIdleTime) {
  Init(port);
  this->timer = new TfrcTimer(maxIdleTime, maxIdleTime);
}


TfrcDaemon::~TfrcDaemon() {
  Close();
  delete timer;
}

int TfrcDaemon::Close() {
  return close(this->fd);
}

int TfrcDaemon::SetFD(fd_set *rs, fd_set *) {
  FD_SET(this->fd, rs);

  int64 timeout = 60* MICROSEC;  /* def: 1 min */

  if (this->timer != NULL) timeout = this->timer->Expire();

  return (timeout/1000);
}


TfrcReceiver *TfrcDaemon::Accept() {
  char buf[MAX_UDP_PKTSIZE];
  struct sockaddr_in sin;
  int socklen = sizeof(struct sockaddr_in);
  
  int datalen = Recvfrom(this->fd, buf, MAX_UDP_PKTSIZE, 0, 
			 (struct sockaddr *)&sin, (socklen_t *) &socklen);
    
  int64 cur_time = get_time();
  int addr = htonl(sin.sin_addr.s_addr);
  int port = htons(sin.sin_port);

  for (int pos = firstPos; pos != lastPos; pos = (pos+1)%TFRC_DAEMON_PCSIZE) {
    /* XXX 3 second here is arguable!!! */
    if (PCArr[pos].time + (int64)TFRC_DAEMON_ACCEPT_TIME * (int64)MICROSEC 
	<= cur_time) { 
      firstPos = (firstPos+1) % TFRC_DAEMON_PCSIZE;
      continue;
    }
    
    if (PCArr[pos].addr == addr && PCArr[pos].port == port) {
      if (verbosity > 0) {
	printf("\n%ld: TfrcDaemon %s report dup connect pkt from %s", 
	       GetRealCurrTime(), getInetAddrName(GetMyAddr()), 
	       getInetAddrName(addr));
	//fprintf(stderr,"\n%ld: TfrcDaemon %s report dup connect pkt from %s", 
	//GetCurrTime(), getInetAddrName(GetMyAddr()), 
	//getInetAddrName(addr));
      }
      return NULL;
    }
  }


  /* this is to prevent duplicated probe packets */
  PCArr[lastPos].addr = addr;
  PCArr[lastPos].port = port;
  PCArr[lastPos].time = get_time();
  lastPos = (lastPos+1) % TFRC_DAEMON_PCSIZE;
  
  return new TfrcReceiver(buf, datalen, (struct sockaddr *)&sin, socklen);
}


int TfrcDaemon::Accept2(char *buf, int buflen, struct sockaddr *sin) {
  int socklen = sizeof(struct sockaddr_in);
  
  return Recvfrom2(this->fd, buf, buflen, 0, sin, (socklen_t *)&socklen);
}


/* return -1 if connection is believed to be terminated
 * 0 if nothing to process
 * 1 if connection is established (calll Accept)
 */
int TfrcDaemon::Process(fd_set *rs, fd_set *) {
  if (FD_ISSET(this->fd, rs)) {
    FD_CLR(this->fd, rs);

    if (this->timer != NULL) this->timer->Update();
    return 1;
  }

  if (this->timer != NULL && this->timer->Expire() == 0) {
    printf("\nTFRC DAEMON Connection aborted or timer expired");
    fprintf(stderr, "\nTFRC DAEMON Connection aborted or timer expired");
    return -1;
  }

  return 0;
}

