#include <assert.h>
#include <iostream.h>
#include <stdio.h>
#include "traceGraph.h"
#include "simulationDriver.h"

TraceGraph::TraceGraph(int sourceAddrIn, 
		       SimulationDriver *simDriverPtrIn){
  sourceAddr=sourceAddrIn;
  simDriverPtr=simDriverPtrIn;
  graph=NULL;
  numVisitedNodes=0;
  visitedArray= new int[simDriverPtr->GetNumMembers()];
}

TraceGraph::~TraceGraph(){
  Link *prevPtr,*linkPtr;
  
  while(linkPtr != NULL){
    prevPtr=linkPtr;
    linkPtr=linkPtr->next;
    delete prevPtr;
  }

  graph=NULL;
  delete [] visitedArray;
}

void TraceGraph::UpdateTraceGraph(){

  int newGraphFlag=0;
  MarkUnvisited();

  for(   
      int addr=simDriverPtr->GetFirstMemberAddr(); 
      addr != -1; 
      addr=simDriverPtr->GetNextMemberAddr(addr)){
    
    int numChildren=simDriverPtr->GetNumChildren(sourceAddr,addr);
    int *childListPtr = new int[numChildren];
    
    simDriverPtr->GetChildList(childListPtr,sourceAddr,addr);
    
    for(int i=0; i < numChildren; i++){
      if(InsertLink(childListPtr[i],addr)){
	cout << "\n" << GetCurrTime() 
	     << ":Insert:" << GetNameByAddr(childListPtr[i])
	     << ":" << GetNameByAddr(addr);
	newGraphFlag=1;
      }
    }

    delete [] childListPtr;
  }
  
  if(DeleteUnvisited()){
    newGraphFlag=1;
  }
  
  if(newGraphFlag){
    numVisitedNodes=0;
    
    cout << "\n" << GetCurrTime()
	 << ":Begin TraceTree: Source "
	 << GetNameByAddr(sourceAddr);

    PrintGraph(sourceAddr,sourceAddr,0,0);
    cout << "\n" << "End TraceTree" << "\n";
    
  }
}


void TraceGraph::PrintGraph(int root, 
			    int parent,
			    int delayFromParent,
			    int inconsistentFlag){

  int i;

  int physicalPerf=simDriverPtr->GetPhysicalPerf(sourceAddr,root);
  int routingPerf = simDriverPtr->GetRoutingPerf(sourceAddr,root);
  
  cout << "\n"
       << GetNameByAddr(root) << " "
       << GetNameByAddr(parent) << " "
       << delayFromParent << " "
       << inconsistentFlag << " "
       << physicalPerf << " " 
       << routingPerf << " ";
  
  
  for(int i=0; i < numVisitedNodes; i++){
    if(root == visitedArray[i]){
      return;
    }
  }
  
  visitedArray[numVisitedNodes++]=root;
  int numChildren=simDriverPtr->GetNumChildren(sourceAddr,root);
  int *childListPtr = new int[numChildren];
  simDriverPtr->GetChildList(childListPtr,sourceAddr,root);
  
  for(i=0; i < numChildren; i++){
    int inconsistentFlag=0;
    
    int delayToChild = 
      simDriverPtr->GetPhysicalPerf(root,childListPtr[i]);
    
    int parentAddr=
      simDriverPtr->GetParentAddr(sourceAddr,childListPtr[i]);
    
    if(root != parentAddr){
      inconsistentFlag=1;
    }
    
    PrintGraph(childListPtr[i],
	       root,
	       delayToChild,
	       inconsistentFlag);
    
  }
  
  delete [] childListPtr;
}

void TraceGraph::MarkUnvisited(){
  for(Link *linkPtr=graph; linkPtr != NULL; linkPtr=linkPtr->next){
    linkPtr->visitedFlag=0;
  }
}

int TraceGraph::DeleteUnvisited(){
  int linksDeletedFlag=0;
  Link *prevPtr=NULL;
  Link *linkPtr=graph;

  while(
	(linkPtr != NULL) && 
	(linkPtr->visitedFlag != 1)
	){
    prevPtr=linkPtr;
    linkPtr=linkPtr->next;
    cout << "\n" << GetCurrTime() 
	 << ":Delete:" 
	 << GetNameByAddr(prevPtr->child)  << ":" 
	 << GetNameByAddr(prevPtr->parent);
    delete prevPtr;
    linksDeletedFlag=1;
  }
  
  graph=linkPtr;
  
  if(graph == NULL){
    return(linksDeletedFlag);
  }
  else{
    prevPtr=graph;
    linkPtr=graph->next;
  }

  while(linkPtr != NULL){
    if(linkPtr->visitedFlag != 1){
      prevPtr->next=linkPtr->next;
      cout << "\n" << GetCurrTime()
	   << ":Delete:" << GetNameByAddr(linkPtr->child) 
	   << ":" << GetNameByAddr(linkPtr->parent);
      delete linkPtr;
      linksDeletedFlag=1;
      linkPtr=prevPtr->next;
    }
    else{
      prevPtr=linkPtr;
      linkPtr=linkPtr->next;
    }
  }

  SanityCheck();
  return(linksDeletedFlag);
}
    
void TraceGraph::SanityCheck(){
  Link *linkPtr=graph;

  if(linkPtr == NULL){
    return;
  }

  while(linkPtr->next != NULL){
    assert(
	   (linkPtr->child < linkPtr->next->child) ||
	   (
	    (linkPtr->child == linkPtr->next->child) &&
	    (linkPtr->parent < linkPtr->next->parent)
	    )
	   );
    linkPtr=linkPtr->next;
  }
}
    
int TraceGraph::InsertLink(int child, int parent){
  Link *linkPtr;
  Link *prevPtr;

  prevPtr=NULL;
  linkPtr=graph; 

  while( 
	(linkPtr != NULL) &&
	(linkPtr->child < child)
	){
    prevPtr=linkPtr;
    linkPtr=linkPtr->next;
  }

  while( 
	(linkPtr != NULL) &&
	(linkPtr->child == child) &&
	(linkPtr->parent < parent)
	){
    prevPtr=linkPtr;
    linkPtr=linkPtr->next;
  }

  if(
     (linkPtr != NULL) &&
     (linkPtr->child == child) &&
     (linkPtr->parent == parent)
     ){
    linkPtr->visitedFlag=1;
    return(0);
  }
  
  Link *newLinkPtr= new Link();
  newLinkPtr->child=child;
  newLinkPtr->parent=parent;
  newLinkPtr->visitedFlag=1;
  newLinkPtr->next=linkPtr;

  if(prevPtr == NULL){
    graph=newLinkPtr;
  }
  else{
    prevPtr->next=newLinkPtr;
  }

  return(1);
}


/***********************
void main(){
  TraceGraph *graph=new TraceGraph(1,NULL);
  
  graph->InsertLink(2,5);
  graph->InsertLink(2,4);
  graph->InsertLink(1,4);
  graph->InsertLink(1,8);
  graph->InsertLink(8,17);
  graph->InsertLink(3,7);


  graph->PrintGraph();

  graph->MarkUnvisited();

  graph->PrintGraph();

  cout.flush();

  graph->InsertLink(2,5);
  graph->InsertLink(2,4);
  graph->InsertLink(1,4);
  graph->InsertLink(8,17);
  graph->InsertLink(3,7);
  graph->InsertLink(9,5);

  graph->PrintGraph();

  cout.flush();

  graph->DeleteUnvisited();
  graph->PrintGraph();


}

**************************/
