#include <iostream.h>
#include <iomanip.h>
#include <stdio.h>
#include <fstream.h>
#include <assert.h>
#include <string.h>

#include <Gossip/global.h>
#include <Gossip/gossipPayLoad.h>
#include <Gossip/overheadStat.h>
#include <Gossip/query.h>

#include <TimerEvents/event.h>
#include <TimerEvents/eventQueue.h>

#include "simGlobal.h"
#include "simulationDriver.h"
#include "routingTable.h"
#include "simGossipHost.h"
#include "packet.h"
#include "traceGraph.h"

SimulationDriver::SimulationDriver(EventQueue *eventQueueIn){
  numRouters=numMembers=numNodes=numLinks=0;
  routingTabPtr=NULL;
  addrToNameMapping=NULL;
  gossipHostPtrArray=NULL;
  unsuccessfulMemberJoinArray=0;
  numActiveMembers=0;
  eventQueue=eventQueueIn;
  traceGraphPtr=NULL;
  if(overheadFlg){
    overheadStat = new OverheadStat(overheadPrintPeriod,1);
  }

}

SimulationDriver::~SimulationDriver(){
  delete routingTabPtr;
  for(int i=0;i<numNodes;i++){
    delete [] addrToNameMapping[i];
    delete gossipHostPtrArray[i];
  }
  delete [] addrToNameMapping;
  delete [] gossipHostPtrArray;
  delete [] unsuccessfulMemberJoinArray;
}


/********************************************************************
 Format of topology file is as below:

 numRouters
 numMembers
 numLinks
 List-of-"R router-name"
 List-of-"M member-name"
 List-of-"L node1-name node-2-name link-delay link-loss"
 
 R's and M's may be interspersed, L's are at the end.
 *********************************************************************/
 
ifstream ifs;
ifstream ifs1;

void 
SimulationDriver::ParseTopologyFileAndComputeRoutingTable(char *topologyFile){


  ifs.open(topologyFile,ios::in);
  if(ifs == NULL) MyError("Error opening topologyFile");
  
  ifs >> numRouters;
  ifs >> numMembers;
  ifs >> numLinks;



  numNodes=numRouters+numMembers;

  if(numNodes >= MAX_ADDRESSES) MyError("Too many Nodes!!! \n");
  routingTabPtr = new RoutingTable(numNodes,numRouters);
  addrToNameMapping = new (char *)[numNodes];
  gossipHostPtrArray = new (SimGossipHost *)[numNodes];
  for(int i=0; i < numNodes; i++)
    gossipHostPtrArray[i]=NULL;
  unsuccessfulMemberJoinArray = new char[numNodes]; 

  // Relevant only for members, put here for convenience 
  // to map from addr to flag

  for(int i=0; i < numNodes; i++)
    unsuccessfulMemberJoinArray[i]=0;

  char networkElementType;
  int linkCount=0,routerCount=0,memberCount=numRouters;
  
  while(ifs >> networkElementType){
    switch(networkElementType){
    case 'R':
      addrToNameMapping[routerCount] = new char[MAX_HOSTNAME_LEN];
      ifs >> addrToNameMapping[routerCount];
      routerCount++;
      if(routerCount > numRouters) MyError("Invalid topology file");
      break;
      
    case 'M':
      addrToNameMapping[memberCount]= new char[MAX_HOSTNAME_LEN];
      ifs >> addrToNameMapping[memberCount];
      memberCount++;
      if(memberCount > numNodes) MyError("Invalid topology file");
      break;
      
    case 'L':
      char node1Name[MAX_HOSTNAME_LEN];
      char node2Name[MAX_HOSTNAME_LEN];
      int node1Idx,node2Idx;
      int linkDelay;
      int linkLoss;
      //float linkLoss;
      
      ifs >> node1Name >> node2Name;
      linkCount++;
      if(linkCount > numLinks) MyError("Invalid topology file");

      for(node1Idx=0; (node1Idx < numNodes) && 
	    (strcmp(GetNameByAddr(node1Idx),node1Name));node1Idx++);
      for(node2Idx=0; (node2Idx < numNodes) && 
	    (strcmp(GetNameByAddr(node2Idx),node2Name));node2Idx++);
      if((node1Idx >= numNodes) || (node2Idx >= numNodes)) MyError("Invalid topology file");
      ifs >> linkDelay >> linkLoss;
      routingTabPtr->Link(node1Idx,node2Idx,linkDelay,linkLoss);
      break;

    default:
      MyError("Invalid topology File");
    }
  }


  if((routerCount != numRouters) || (memberCount != (numMembers+numRouters)) || (linkCount != numLinks))
    MyError("Invalid topology file");

  
  printf("Loading Routing Table!!!");
  fflush(stdout);
  
  if(inputRoutingTableMode)
    routingTabPtr->LoadRoutingTable();
  else
    routingTabPtr->CalculateRoutingTable();
  
  printf("Done loading!!!!");
  fflush(stdout);
  
  // routingTabPtr->PrintRoutingTable();
}

/***********************
 EventFile configuration:

 numMembersipChanges
 time member J/D/L (J:Join,D:Die.L:Leave)
 if Join,
 time member Join lowerDegreeBound upperDegreeBound sourceFlg(1/0)
 if sourceFlg == 1
 time member Join lowerDegreeBound upperDegreeBound 1 dataPeriod dataSize
 ***********************/

void SimulationDriver::ParseMembershipChangeFile(char *membershipChangeFile){



  ifs1.open(membershipChangeFile,ios::in);
  if(ifs1 == NULL) MyError("Error opening membershipChangeFile");

  long time;
 
  while(ifs1 >> time){
    MembershipChangeType *ptr = new MembershipChangeType;
    ptr->time=time;

    char name[MAX_HOSTNAME_LEN];
    ifs1 >> name;

    int addr;
    for(addr=numRouters; (addr < numNodes) && 
	  (strcmp(GetNameByAddr(addr),name));addr++);
    if(addr >= numNodes) MyError("Invalid membership change file");
    ptr->memberAddr=addr;

    ifs1 >> ptr->changeCode;
    if(ptr->changeCode == 'J'){
      ifs1 >> ptr->lowerDegreeBound;
      ifs1 >> ptr->upperDegreeBound;
      ifs1 >> ptr->sourceFlg;
      if(ptr->sourceFlg){
	ifs1 >> ptr->dataPeriod;
	ifs1 >> ptr->dataSize;
      }
      else{
	ptr->dataPeriod=-1;
	ptr->dataSize=-1;
      }
    }
    Event *eventPtr = new Event(this,time,ptr);
    eventQueue->InsertEvent(eventPtr);
  }
}

void SimulationDriver::ReadyToLeave(int addr){
  assert(gossipHostPtrArray[addr]->gossipAgentPtr != NULL);
  /***************
  if(
     (gossipHostPtrArray[addr]->gossipAgentPtr->dataAgentPtr != NULL) &&
     (gossipHostPtrArray[addr]->gossipAgentPtr->dataAgentPtr->seqBuffer != NULL)
     ){
    if(resetStatisticsTime < 0){
      packetLossStream << GetNameByAddr(addr) << " " 
       << gossipHostPtrArray[addr]->gossipAgentPtr->dataAgentPtr->seqBuffer->ComputeLoss() << "\n";
    }
  }
  ***************/

  cout << "\n" << GetCurrTime() << ":"
       << "Leave:" << GetNameByAddr(addr);
	  
  numActiveMembers--;
  delete gossipHostPtrArray[addr];
  gossipHostPtrArray[addr]=NULL;
  UpdateTrace();
}


/***********
 This function should never be called after merge of partition 
 and join agents. Sanjay, April 2002
***********/
void SimulationDriver::JoinUnSuccessful(int addr){
  assert(addr >= numRouters);
  assert(addr < numNodes);
  
  cout << "\n" << addr << ": JOIN UNSUCCESSFUL!!!!";
  numActiveMembers--;
  unsuccessfulMemberJoinArray[addr]=1;
  delete gossipHostPtrArray[addr];
  gossipHostPtrArray[addr]=NULL;

  assert(0 == 1); // Temporarily put here, to alert if a join was unsuccessful.
}


void SimulationDriver::DumpRoutingTable(){
  routingTabPtr->DumpRoutingTable();
}

void SimulationDriver::SendPacket(int fromAddr, 
				  int toAddr, 
				  Packet *packetPtr, 
				  PacketType packetType){

  assert(fromAddr >= numRouters);
  assert(fromAddr < numNodes);
  assert(toAddr >= numRouters);
  assert(toAddr < numNodes);
  assert(gossipHostPtrArray[fromAddr] != NULL); // Receiver may be dead!!

  if(overheadFlg){
    overheadStat->UpdateOverhead(packetType,0,packetPtr->Size());
  }

  char *msgCodeString=
    ((GossipPayLoad *)(packetPtr->GetPayLoad()))->GetMsgCodeString();

  if (
      (
       (verbosity > 0) &&
       (
	(strcmp(msgCodeString,"DATA")) &&
	(strcmp(msgCodeString,"POKE_DATA")) &&
	(strcmp(msgCodeString,"ESTIMATE_DELAY_REQUEST")) &&
	(strcmp(msgCodeString,"ESTIMATE_DELAY_RESPONSE"))
	)
       )
      
      ||
      
      (verbosity > 1)
      )
    
    {
      
      cout << "\n" << GetCurrTime() 
	   << ":Send " <<  msgCodeString
	   << " from host : " << GetNameByAddr(fromAddr) 
	   << " to " << GetNameByAddr(toAddr);
      
    }
  
  int delay=routingTabPtr->Route(fromAddr,toAddr,packetPtr->Size(),packetType);
  if((delay < 0) || (gossipHostPtrArray[toAddr] == NULL)){
    // Packet lost, not delivered, or the recepient dead
    delete packetPtr;
    return;
  }
  
  Event *eventPtr=
    new Event(gossipHostPtrArray[toAddr],GetCurrTime()+delay,packetPtr);
  eventQueue->InsertEvent(eventPtr);
}

void SimulationDriver::Simulate(){

  while((GetCurrTime() < simulationTime) && (!eventQueue->IsEmpty())){
    Event *eventPtr=eventQueue->PopNextEvent();
    long time=eventPtr->GetTimeToFire();
    assert(time >= GetCurrTime());
    SetCurrTime(time); // Progress time to time of next event
    EventReceiver *eventReceiverPtr=eventPtr->GetReceiver();
    eventReceiverPtr->PrintEvent(eventPtr->GetMsgToDeliver());
    cout.flush();
    
    if(GetCurrTime() == debugTime){
      cout << "\n About to debug events at time: " << debugTime;
      cout.flush();
      cout << "\n EVENT QUEUE START!!!!";
      verbosity=2;
      eventQueue->Print();
      cout << "\n EVENT QUEUE END!!!!";
      cout.flush();
      verbosity=1;

    }
    eventReceiverPtr->RecvEvent(eventPtr->GetMsgToDeliver());
    delete eventPtr;

    if(eventReceiverPtr == this){
      UpdateTrace();
    }
  }
  if(debugFlg) DoDebug();
  if(latencyFlg) DoComputeLatency();
  if(stressFlg) DoComputeStress();
  if(fanoutFlg) DoComputeFanout();
  if(knowledgeFlg) DoComputeKnowledge();

  cout << "\n" << GetCurrTime() << ":SimulationEndTime";

  if (verbosity > 0){
    cout << "\n ****** ROUTING TABLE FINALLY *********";
    routingTabPtr->PrintRoutingTable();
  }

}


void SimulationDriver::RecvEvent(void *eventMsg){
  
  MembershipChangeType *membershipChangePtr = (MembershipChangeType *) eventMsg;  
  // A cast, is it bad practice? Don't know.
  if(unsuccessfulMemberJoinArray[membershipChangePtr->memberAddr]) return; 
  // Ignore events from unsuccessful joiners.

  switch(membershipChangePtr->changeCode){
  case 'J':
    {

      int *memberAddrListPtr = new int[numActiveMembers];
      int count=0;
      for(int addr=numRouters;addr < numNodes; addr++){
	if(gossipHostPtrArray[addr] != NULL){
	  memberAddrListPtr[count]=addr;
	  count++;
	}
      }

      cout << "\n" << GetCurrTime() 
	   << ":JoinStart:" << GetNameByAddr(membershipChangePtr->memberAddr)
	   << ":" << membershipChangePtr->sourceFlg;
      
      cout << "\n" << GetCurrTime() 
	   << ":MeshJoin:" << GetNameByAddr(membershipChangePtr->memberAddr);


      if(membershipChangePtr->sourceFlg){
	if(traceGraphPtr == NULL){
	  traceGraphPtr = new TraceGraph(membershipChangePtr->memberAddr,
					 this);
	}
	else{
	  MyWarning("Tracing two sources!!! Disregarding %d!",
		    membershipChangePtr->memberAddr);
	}
      }

      assert(count == numActiveMembers);
      gossipHostPtrArray[membershipChangePtr->memberAddr] = 
	new SimGossipHost(membershipChangePtr->memberAddr,this,eventQueue);
      numActiveMembers++;
      gossipHostPtrArray[membershipChangePtr->memberAddr]->JoinGroup(
					     count,
					     memberAddrListPtr,
					     membershipChangePtr->lowerDegreeBound, 
					     membershipChangePtr->upperDegreeBound,
					     membershipChangePtr->sourceFlg,
					     membershipChangePtr->dataPeriod,
					     membershipChangePtr->dataSize);
      break;
    }
  case 'D':
    {
      if(gossipHostPtrArray[membershipChangePtr->memberAddr] != NULL){
	/**************
	if(gossipHostPtrArray[membershipChangePtr->memberAddr]->gossipAgentPtr->dataAgentPtr->seqBuffer != NULL){
	  cout << "\n" << "PacketLoss: " 
	       << gossipHostPtrArray[membershipChangePtr->memberAddr]->gossipAgentPtr->dataAgentPtr->seqBuffer->ComputeLoss();
	}
	**************/
	delete gossipHostPtrArray[membershipChangePtr->memberAddr];
	gossipHostPtrArray[membershipChangePtr->memberAddr]=NULL;
	numActiveMembers--;

	cout << "\n" << GetCurrTime() 
	     << ":Die:" << GetNameByAddr(membershipChangePtr->memberAddr);

      }
      break;
    }
  case 'L':
    {
      /**XXX: Sanjay: stayTimeOnLeave set to 0. Make it parameter ***/
      if(gossipHostPtrArray[membershipChangePtr->memberAddr] != NULL){
	gossipHostPtrArray[membershipChangePtr->memberAddr]->InitLeave(0);


      }
      break;
    }
  default:
    MyError("Invalid code!!");
  }
  delete membershipChangePtr;
}


/***************************
 DoComputePacketLoss: Disabled for now...! Sanjay, Nov 5 2001
 Need to redesign this function
**************************/
void SimulationDriver::DoComputePacketLoss(){
  /***
  ofstream ofs(packetLossFileName,ios::out);
  if(ofs == NULL) MyError("Error opening packet loss file");
  ****/

  /*************
  for(int addr=numRouters; addr < numNodes; addr++){
    if(gossipHostPtrArray[addr] == NULL) continue;
    if(gossipHostPtrArray[addr]->gossipAgentPtr->dataAgentPtr->seqBuffer != NULL)
      packetLossStream << GetNameByAddr(addr) << " " 
	  << gossipHostPtrArray[addr]->gossipAgentPtr->dataAgentPtr->seqBuffer->ComputeLoss() << "\n";
  }
  *****************/
}

/***********
 DoComputeFanout:

 <memberName> <# of neighbors> (including temporary neighbors
*************/
void SimulationDriver::DoComputeFanout(){
  ofstream ofs(fanoutFileName,ios::out);
  if(ofs == NULL) MyError("Error opening fanout file");

  for(int addr=numRouters;addr < numNodes;addr++){
    if(gossipHostPtrArray[addr] == NULL)
      continue;
    Query *queryPtr=gossipHostPtrArray[addr]->GetQueryAgent();
    ofs << GetNameByAddr(addr) << " " 
	<< queryPtr->GetNumNbrs() << "\n";
  }
}
     
     
     
void SimulationDriver::DoDebug(){
  MyError(0);
  /******* Sanjay: Nov 6, 2001:
	   This function may need work, but I am not clear on whether
	   it is useful any more. So, avoid using for now.
  ********************/

  /*************
  for(int addr=numRouters;addr < numNodes; addr++){
    if(gossipHostPtrArray[addr] != NULL){
      gossipHostPtrArray[addr]->DoDebug();
    }
  }
  
  ofstream ofs(debugFileName,ios::out);
  if(ofs == NULL) MyError("Error opening debug file");
  ************/
  
  /*******Sanjay: XXX : To compile
  for(int i=0; i < debugNumEdges;i++){
    ofs << "\n " << GetNameByAddr(debugEdgeArray1[i]) << " " << 
      GetNameByAddr(debugEdgeArray2[i]) << " " << 
      routingTabPtr->GetDelay(debugEdgeArray1[i],debugEdgeArray2[i]);
  }
  ***************/
}


void SimulationDriver::DoComputeKnowledge(){
  /*** Sanjay: Nov 6, 2001:
       This function looks buggy. If it is useful, we can fix it later.
       For now, no need to support it.
  ******************/

  MyError(0);

  /***********************
  ofstream ofs(knowledgeFileName,ios::out);
  if(ofs == NULL) MyError(" Error opening knowledge file");

  for(int addr1=numRouters; addr1 < numNodes; addr1++){
    if(gossipHostPtrArray[addr1] == NULL) continue;
    if(gossipHostPtrArray[addr1]->gossipAgentPtr->joinMode) continue;
    ofs << GetNameByAddr(addr1) << ":"; 
    for(int addr2=numRouters; addr2 < numNodes; addr2++){
      if(gossipHostPtrArray[addr2] == NULL) continue;
      if(gossipHostPtrArray[addr2]->gossipAgentPtr->joinMode) continue;
      if(gossipHostPtrArray[addr1]->gossipAgentPtr->vrtPtr->table[addr2] != NULL)
	ofs << "1 ";
      else
	ofs << "0 ";
    }
    ofs << "\n";
  }
  ***************************/
}


/**********
DoComputeLatency: 

Prints output in the form:
      host1 host2 routingDelay physicalDelay Penalty

-Prints symmetric pairs consecutively.
-Does not print a value if a member does not know the other member
-Prints a penalty of 100000 if physicalDelay is 0

For negative routing delays, prints some meaningless negative values
**********/

void SimulationDriver::DoComputeLatency(){
  ofstream ofs(latencyFileName,ios::out);
  if(ofs == NULL) MyError("Error opening debug file");
  
  int addr1,addr2;
  for(addr1=numRouters; (addr1 < numNodes); addr1++){
    if(gossipHostPtrArray[addr1] == NULL) continue;
    for(addr2=addr1+1; (addr2 < numNodes); addr2++){
      if(gossipHostPtrArray[addr2] == NULL) continue;
      int routingDelay=-1;
      
      Query *queryPtr =
	gossipHostPtrArray[addr1]->GetQueryAgent();

      int ret=queryPtr->GetRoutingDelay(&routingDelay,addr2);
      assert(ret == 0);

      int physicalDelay=routingTabPtr->GetDelay(addr1,addr2);
      float delayPenalty;
      if(physicalDelay==0)
	delayPenalty=100000;
      else delayPenalty=routingDelay/(float)physicalDelay;
      
      ofs << GetNameByAddr(addr1)  << " " << GetNameByAddr(addr2) << " " 
	  << routingDelay << " "   << physicalDelay << " " 
	  << setprecision(3) << (float)delayPenalty << "\n";

      routingDelay=-1;
      queryPtr=gossipHostPtrArray[addr2]->GetQueryAgent();
      ret=queryPtr->GetRoutingDelay(&routingDelay,addr1);
      assert(ret == 0);

      physicalDelay=routingTabPtr->GetDelay(addr2,addr1);
      if(physicalDelay==0)
	delayPenalty=100000;
      else delayPenalty=routingDelay/(float)physicalDelay;
      
      ofs << GetNameByAddr(addr2) << " " << GetNameByAddr(addr1) << " " 
	  << routingDelay << " " 
	  << physicalDelay << " " 
	  << setprecision(3) << delayPenalty << "\n";
    }
  }
}


void SimulationDriver::CountVirtualPath(StrategyType strategy,int source, int destination){
  int currentNode=source;

  while(currentNode != destination){
    int nextHop=routingTabPtr->table[currentNode][destination].successor;
    assert(routingTabPtr->table[currentNode][nextHop].linkFlg == 1);
    
    if(strategy == GOSSIP_STRATEGY)
      stressTabPtr[currentNode][nextHop].gossipCount++;
    else if(strategy == UNICAST_STRATEGY)
      stressTabPtr[currentNode][nextHop].unicastCount++;
    else if(strategy == DVMRP_STRATEGY)
      stressTabPtr[currentNode][nextHop].multicastCount |= 0x1;
    else
      MyError("Invalid Strategy!!");

    currentNode=nextHop;
  }
}

/*********
 DoComputeStress:

 if (stressFlg == 2): prints information only for the sources
 if (stressFlg == 1): prints information for every member.

 Format:

 For each source:

 Begin Source <sourcename>
 ... entries
 End Source <sourcename>

 Each entry has the following format:
 Print each *physical* link that has a non-zero stress value for Gossip, DVMRP, or unicast

<Node1> <Node2> unicastStress MulticastStress GossipStress
<Node1>,<Node2> are routers, or end hosts.

Important note: we treat physical links *without* direction.
Thus, a packet from R1-R2, or R2-R1, both contributes a stress of 1 to that link

We would also print error messages:
<Host1> UNAWARE OF <SOURCE>, or
<Host1> NO ROUTE TO <SOURCE>, 
if in Gossip <host1> does not have an entry for, or has no route to <SOURCE>




**************/

void SimulationDriver::DoComputeStress(){
  ofstream ofs(stressFileName,ios::out);
  if(ofs == NULL) MyError("Error opening debug file");

  stressTabPtr= new (StressAnalyzeType *)[numNodes];
  for(int i=0; i < numNodes; i++)
    stressTabPtr[i] = new (StressAnalyzeType) [numNodes];
  
  for(int source=numRouters;source < numNodes; source++){


    if(gossipHostPtrArray[source] == NULL) continue;
    if(stressFlg == 2){
      Query *queryPtr = gossipHostPtrArray[source]->GetQueryAgent();
      if(!queryPtr->IsSource())
	continue;
    }
    ofs << "Begin Source " << " " << GetNameByAddr(source) << "\n";

    // Initialize Stress Counts
    for(int i=0; i < numNodes; i++){
      for(int j=0; j < numNodes; j++){
	stressTabPtr[i][j].unicastCount=0;
	stressTabPtr[i][j].multicastCount=0;
	stressTabPtr[i][j].gossipCount=0;
      }
    }
    
    //Compute gossipCount
    for(int childAddr=numRouters;childAddr < numNodes; childAddr++){
      if(
	 (gossipHostPtrArray[childAddr] == NULL) || 
	 (childAddr == source)
	 ) continue;

      Query *queryPtr
	=gossipHostPtrArray[childAddr]->GetQueryAgent();

      if(queryPtr->IsAware(source)){
	ofs << GetNameByAddr(childAddr) 
	    << " UNAWARE OF " 
	    << GetNameByAddr(source) << "\n";
      }
      else  if(queryPtr->IsValidRoute(source)){
	ofs << GetNameByAddr(childAddr) 
	    << " NO ROUTE TO " 
	    << GetNameByAddr(source) << "\n";
      }
      else{
	int parentAddr;
	int ret;

	ret=queryPtr->GetParentAddr(&parentAddr,source);
	assert(ret == 0);

	CountVirtualPath(GOSSIP_STRATEGY,parentAddr,childAddr);
      }
    }
    
    //Compute unicastCount

    for(int addr=numRouters; addr < numNodes; addr++){
      if((gossipHostPtrArray[addr] == NULL) || (addr == source)) continue;
      CountVirtualPath(UNICAST_STRATEGY,source,addr);
    }
     
    //Compute DVMRPCount
    for(int addr=numRouters; addr < numNodes; addr++){
      if((gossipHostPtrArray[addr] == NULL) || (addr == source)) continue;
      CountVirtualPath(DVMRP_STRATEGY,source,addr);
    }

    for(int i=0; i < numNodes; i++){
      for(int j=i+1; j < numNodes; j++){

	int unicastCount=
	  stressTabPtr[i][j].unicastCount + 
	  stressTabPtr[j][i].unicastCount;

	int multicastCount=
	  (stressTabPtr[i][j].multicastCount | 
	   stressTabPtr[j][i].multicastCount);

	int gossipCount=
	  stressTabPtr[i][j].gossipCount +
	  stressTabPtr[j][i].gossipCount;
	  
	if(
	   (unicastCount > 0)||
	   (multicastCount > 0) ||
	   (gossipCount > 0)
	   )
	  ofs << GetNameByAddr(i) << " " << GetNameByAddr(j) 
	      << " " << unicastCount
	      << " " << multicastCount
	      << " " << gossipCount << "\n";
      }
    }
    ofs << "End Source " << " " << GetNameByAddr(source) << "\n";
  }
}

void SimulationDriver::PrintEvent(void *eventMsg){
  struct MembershipChangeType *ptr;

  ptr=(MembershipChangeType *)eventMsg;
  if(verbosity > 0){
    if (ptr->changeCode == 'D')
      cout << "\n" << GetCurrTime() << ":DEATH " << GetNameByAddr(ptr->memberAddr);
    else if(ptr->changeCode == 'L')
      cout << "\n" << GetCurrTime() << ":LEAVE " << GetNameByAddr(ptr->memberAddr);
    else if(ptr->changeCode == 'J'){
      printf("\n %ld:JOIN: %s(<%d,%d>) %d",GetCurrTime(), GetNameByAddr(ptr->memberAddr),ptr->lowerDegreeBound,ptr->upperDegreeBound,ptr->sourceFlg);
      if(ptr->sourceFlg) printf(" <%d %d>",ptr->dataPeriod,ptr->dataSize);
      fflush(stdout);
    }
    else cout << "Invalid Change code: " << ptr->changeCode;
  }
}


  
int SimulationDriver::GetFirstMemberAddr(){
  int nextAddr;

  for(nextAddr=numRouters; 
      (nextAddr < numNodes) && (gossipHostPtrArray[nextAddr] == NULL); 
      nextAddr++);

  if(nextAddr >= numNodes){
    return(-1);
  }
  else{
    return(nextAddr);
  }
}

int SimulationDriver::GetNextMemberAddr(int addr){
  int nextAddr;
  
  for(nextAddr=addr+1; 
      (nextAddr < numNodes) && (gossipHostPtrArray[nextAddr] == NULL); 
      nextAddr++);

  if(nextAddr >= numNodes){
    return(-1);
  }
  else{
    return(nextAddr);
  }
}

int SimulationDriver::GetNumChildren(int sourceAddr, int addr){
  if(gossipHostPtrArray[addr] == NULL){
    return(0);
  }
  
  Query *queryPtr=gossipHostPtrArray[addr]->GetQueryAgent();
  return(queryPtr->GetNumChildren(sourceAddr));
  
}

void SimulationDriver::GetChildList(int *childListPtr,
				   int sourceAddr, 
				   int addr){

  if(gossipHostPtrArray[addr] == NULL){
    return;
  }
  
  Query *queryPtr=gossipHostPtrArray[addr]->GetQueryAgent();
  queryPtr->GetChildList(childListPtr,sourceAddr);
  
}

int SimulationDriver::GetParentAddr(int sourceAddr,
				    int addr){
  
  int nextHop=-1;

  if(gossipHostPtrArray[addr] == NULL){
    return(-1);
  }
  
  
  Query *queryPtr=gossipHostPtrArray[addr]->GetQueryAgent();
  queryPtr->GetParentAddr(&nextHop, sourceAddr);

  return(nextHop);
}

int SimulationDriver::GetPhysicalPerf(
				      int sourceAddr,
				      int addr
				      ){

   int physicalDelay=routingTabPtr->GetDelay(sourceAddr,addr);
   return(physicalDelay);
}



int SimulationDriver::GetRoutingPerf(int sourceAddr, 
				     int addr){

  if(gossipHostPtrArray[addr] == NULL){
    return(-1);
  }

  int routingPerf=-1;

  Query *queryPtr=gossipHostPtrArray[addr]->GetQueryAgent();
  int ret=queryPtr->GetRoutingDelay(&routingPerf,sourceAddr);
  assert(ret == 0);

  return(routingPerf);
}




int SimulationDriver::GetNumMembers(){
  return(numMembers);
}

void SimulationDriver::UpdateTrace(){
  if(traceGraphPtr != NULL){
    traceGraphPtr->UpdateTraceGraph();
  }
}  
