#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>
#include <netdb.h>
#include <signal.h>

#include <arpa/inet.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/resource.h>  /* SUN rlimit */
#include <netinet/in.h>	/* sockaddr_in{} and other Internet defns */

#include <Util/inetMisc.h>  /* MAX_PAYLOAD_SIZE defined */

#include "../Gossip/gossipAgent.h"
#include "dataMgrTfrc.h"

#include <Tfrc/tfrcDaemon.h>
#include <Tfrc/tfrcSender.h>
#include <Tfrc/tfrcReceiver.h>

DataMgrTfrc::DataMgrTfrc(int DaemonPort) : DataMgrUdp(DaemonPort) {

  if (verbosity > 0){
    printf("\ndata mgr is TFRC");
  }

  /* XXX: if you change TFRC daemon, you must change
   * the filter port in tfrcReceiver accordingly
   */
  daemon = new TfrcDaemon(DaemonPort+1);
  connMgr = new ConnMgr();
}


DataMgrTfrc::~DataMgrTfrc() {
  delete daemon;
}


/* returns the timeout value in msec */
int DataMgrTfrc::SetFD(fd_set *rs, fd_set *ws) {
  int timeout;
  
  timeout = DataMgrUdp::SetFD(rs, ws);
  
  timeout = daemon->SetFD(rs, ws);

  for (struct TfrcConn *conn = connMgr->GetFirst();
       conn != NULL; conn = connMgr->GetNext()) {
    switch(conn->type) {
    case TFRC_SND:
      timeout = min(timeout, conn->dir.snd->SetFD(rs, ws));
      break;
    case TFRC_RCV:
      timeout = min(timeout, conn->dir.rcv->SetFD(rs, ws));
      break;
    default:
      assert(! "wrong type");
    }
  }

  return timeout;
}

#if 1
int DataMgrTfrc::ReadFromNetwork(fd_set *rs, fd_set *ws, char *buf, 
				 int maxBufLen, int *fromAddr, int *fromPort) {

  int ret = DataMgrUdp::ReadFromNetwork(rs, ws, buf, maxBufLen, 
					fromAddr, fromPort);
  if (ret >= 0) return ret;
  
  /* handle new data connection request */
  if ((ret = daemon->Process(rs, ws)) < 0) {
    /* TFRC daemon quits */
    assert(0);
  }
  
  if (ret > 0) {
    /*
    char buf[MAX_UDP_PKTSIZE];
    int socklen = sizeof(struct sockaddr_in);
    struct sockaddr_in sin;
    
    daemon->Accept2(buf, MAX_UDP_PKTSIZE, 
    */

    TfrcReceiver *receiver = daemon->Accept();

    /* if receiver is NULL, dup connection pkt */
    if (receiver != NULL) {
      int addr, port;
      assert(receiver->GetRemoteProcess(&addr, &port) >= 0);
      connMgr->Add(addr, TFRC_RCV, receiver);
    }
  }


  /* snd requests */
  for (struct TfrcConn *conn = connMgr->GetFirst();
       conn != NULL; conn = connMgr->GetNext()) {
    switch(conn->type) {
    case TFRC_SND: {
      if (conn->dir.snd->Process(rs, ws) < 0) {
	connMgr->Remove(conn);
      }
      continue;
      break;
    }      
    case TFRC_RCV: {
      int pktlen = conn->dir.rcv->Process(rs, ws, buf, maxBufLen);
      
      if (pktlen == 0) {
	/* two possiblities that pktlen == 0:
	 * (i): someone sends a 0-byte packet
	 * (ii): unsuccessful read from the socket, but does not
	 *       warrent removing the connection
	 */
	continue;
      }

      if (pktlen < 0) {
	connMgr->Remove(conn);
	continue;
      }
      
      *fromAddr = conn->addr;
      *fromPort = DaemonPort;  /* bogus */
      
      return pktlen;
      break;
    }      
    default:
      assert(! "\nunknown type");
    }
  }
  
  return -1;    /* no data */
}

#else

int DataMgrTfrc::ReadFromNetwork(fd_set *rs, fd_set *ws, char *buf, 
				 int maxBufLen, int *fromAddr, int *fromPort) {

  int ret;
  
  if ((ret = DataMgrUdp::ReadFromNetwork(rs, ws, buf, maxBufLen, 
					 fromAddr, fromPort)) >= 0) {
    return ret;
  }
  
  /* handle new data connection request */
  if ((ret = daemon->Process(rs, ws)) < 0) {
    /* TFRC daemon quits */
    assert(0);
  }
  
  if (ret > 0) {
    char buf[MAX_UDP_PKTSIZE];
    struct sockaddr_in sin;
    int socklen = sizeof(struct sockaddr_in);
    
    int datalen = daemon->Accept2(buf, MAX_UDP_PKTSIZE, 
				  (struct sockaddr *)&sin);
    
    if (datalen > 0) {
      int addr = htonl(sin.sin_addr.s_addr);
      int port = htons(sin.sin_port);
      
      /* check if it is already handled */
      struct TfrcConn *conn = connMgr->Get(addr, TFRC_RCV);
      
      if (conn == NULL) {
	TfrcReceiver *receiver = new TfrcReceiver(buf, datalen, 
						  (struct sockaddr *)&sin, socklen);
	connMgr->Add(addr, TFRC_RCV, receiver);
      } else {
	/* old receiver? (match port) */
	
	assert(conn->addr == addr);
	TfrcReceiver *receiver = conn->dir.rcv;
	
	if (receiver->GetRemotePort() == port) {
	  /* XXX: hack send data to the receiver buffer */
	  fprintf(stderr, "dataMgrTfrc: %s passing a packet over to tfrcReceiver\n", getInetAddrName(GetMyAddr()));
	  receiver->InjectData(buf, datalen, (struct sockaddr *)&sin, socklen);
	} else {
	  /* remove the old receiver */
	  connMgr->Remove(conn);
	  TfrcReceiver *receiver = new TfrcReceiver(buf, datalen, 
						    (struct sockaddr *)&sin, 
						    socklen);
	  connMgr->Add(addr, TFRC_RCV, receiver);
	}
      }
    }
  }

  /* snd requests */
  for (struct TfrcConn *conn = connMgr->GetFirst();
       conn != NULL; conn = connMgr->GetNext()) {
    switch(conn->type) {
    case TFRC_SND: {
      if (conn->dir.snd->Process(rs, ws) < 0) {
	connMgr->Remove(conn);
      }
      continue;
      break;
    }      
    case TFRC_RCV: {
      int pktlen = conn->dir.rcv->Process(rs, ws, buf, maxBufLen);
      
      if (pktlen == 0) { 
	continue;
      }

      if (pktlen < 0) {
	connMgr->Remove(conn);
	continue;
      }
      
      *fromAddr = conn->addr;
      *fromPort = DaemonPort;  /* bogus */
      
      return pktlen;
      break;
    }      
    default:
      assert(! "\nunknown type");
    }
  }
  
  return -1;    /* no data */
}
#endif


/* return 0 if succeed, < 0 if failed */
int DataMgrTfrc::SendToNetwork(int toAddr, int toPort, const char *buf, 
			       int bufLen, PacketType packetType,
			       int priority) {

  /* handle non-TFRC stuff */
  switch(packetType) {
  case CONTROL:
  case PROBE:
    return DataMgrUdp::SendToNetwork(toAddr, toPort, buf, bufLen, 
				     packetType, priority);
    break;
  case MEASUREMENT:
  case DATAPACKET:
  case POKE:
    break;  /* continue process */
  default:
    MyError("DataMgrTfrc: unknown case");
    break;
  }
  
  /* get/establish the connection */
  struct TfrcConn *conn = connMgr->Get(toAddr, TFRC_SND);

  if (conn == NULL) {

    /* don't open a TFRC connection just for measurement traffic 
     * a hack XXX
     */
    if (packetType == MEASUREMENT) {
      return DataMgrUdp::SendToNetwork(toAddr, toPort, buf, bufLen, 
				       packetType, priority);
    }

    /* establish a new connection if it does not exist */
    if ((conn = EstablishConn(toAddr)) == NULL) {
      return -1;
    }
  }
  
  if (packetType == MEASUREMENT) {
    if (conn->dir.snd->GetSessionDuration() >= 5 &&
	conn->dir.snd->GetBwExpected() <= 10.0) { /* Kbps */

      if (verbosity > 0){
	printf("\nTFRC Garbage Collection: %d %d",
	       conn->dir.snd->GetSessionDuration(),
	       conn->dir.snd->GetBwExpected());
      }
      /* gc the connection */
      connMgr->Remove(conn);
      
      return DataMgrUdp::SendToNetwork(toAddr, toPort, buf, bufLen, 
				       packetType, priority);
    }
  }


  conn->dir.snd->Send(buf, bufLen, priority);
  
  return 0;
}


/* return a new TfrcConn object
 */
struct TfrcConn *DataMgrTfrc::EstablishConn(int addr) {

  TfrcSender *sender = new TfrcSender(addr, DaemonPort+1);
  assert(sender != NULL);

  sender->SetBwExpected(5, 1);
  sender->SetBwActual(5, 1);

  struct TfrcConn *conn = connMgr->Add(addr, TFRC_SND, sender);
  assert(conn != NULL);

  return conn;
}

  

