#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>
#include <netdb.h>

#include <arpa/inet.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>	/* sockaddr_in{} and other Internet defns */

#include <Gossip/global.h>

#include <Util/inetTypes.h>
#include <Util/inetMisc.h>
#include <Util/misc.h>
#include <Util/estimateBandwidth.h>

#include <Internet/inetGossipHost.h>

#include "inetProxy.h"
#include "proxyClient.h"


ProxyClient::ProxyClient(int clientAddr) {
  //  : ApplicationClient(int clientAddr) {

  this->clientAddr = clientAddr;
  numPorts = 0;

  for (int i=0; i<MAX_PORTS_PROXY_LISTENS; i++) {
    this->port[i] = -1;
    sendFd[i] = -1;
  }

  bwSentToApp = new EstimateBandwidth(10, 5);

  if (IsMulticastAddr(clientAddr)) {
    type = MULTICAST;
  } else {
    type = UNICAST;
  }
}


void ProxyClient::SetTTL(int ttl) {
  assert(type == MULTICAST);
  this->ttl = ttl;
}


ProxyClient::~ProxyClient() {
  delete bwSentToApp;
}


/* Open a connected UDP socket to the client port
 * return FALSE if connect failed, TRUE otherwise
 */
int ProxyClient::AddPort(int port) {
  int localSendPort = -1;
  int fd;

  switch(type) {
  case UNICAST: {
    fd = Socket(AF_INET, SOCK_DGRAM, 0);
    SetsockNonBlocking(fd);
    SetsockoptBufSize(fd, 65535);
    
    struct sockaddr_in sock;
    bzero((char *)(&sock), sizeof(sock));
    sock.sin_family = AF_INET;
    sock.sin_addr.s_addr = htonl(GetAddr());
    sock.sin_port = htons(port);
    
    errno = 0;
    if (connect(fd, (struct sockaddr *)&sock, 
		sizeof(sock)) < 0) {
      switch(errno) {
      case ECONNREFUSED:
      default:  // should not happen
	MyWarning("ConnectToUnicastClient");
	perror("connect");
	return FALSE;
	break;
      }
    }
    break;
  }

  case MULTICAST: {
    fd = Socket(AF_INET, SOCK_DGRAM, 0);
    SetsockNonBlocking(fd);
    SetsockoptBufSize(fd, 65535);
    
    struct sockaddr_in sock;
    bzero((char *)(&sock), sizeof(sock));
    sock.sin_family = AF_INET;
    sock.sin_addr.s_addr = htonl(INADDR_ANY);
    sock.sin_port = 0;
    Bind(fd, (struct sockaddr *)&sock, sizeof(sock));
    
    /* get the ephemeral port */
    socklen_t len = sizeof(sock);
    Getsockname(fd, (struct sockaddr *)&sock, &len);
    localSendPort = htons(sock.sin_port);
    
    /* set loopback and ttl */
    Setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL, 
	       (char *)&ttl,sizeof(ttl));
    
    unsigned char loopback=1;
    Setsockopt(fd, IPPROTO_IP, IP_MULTICAST_LOOP, 
	       (char *)&loopback, sizeof(loopback));
    break;
  }
  
  default:
    MyError("ProxyClient: type %d not implemented", type);
    break;
  }

  assert(GetPortID(port) == -1);

  this->sendFd[numPorts] = fd;
  this->port[numPorts] = port;
  this->localSendPort[numPorts] = localSendPort;
  numPorts++;
  return TRUE;
}


/* in vic/vat cntl and data ports come in pair XXX: to remove*/
int ProxyClient::GetSisterPort(int i) {
  int j; if (i%2==0) { j=i+1; } else {j=i-1;}
  
  return j;
}


/* get port ID
 * returns -1 if port not found
 */
int ProxyClient::GetPortID(int port) {
  for (int i=0; i<numPorts; i++) {
    if (this->port[i] == port) return i;
  }
  return -1;
}


void ProxyClient::RemovePort(int port) {
  int i = GetPortID(port);
  assert(i >= 0);

  numPorts--;
  assert(numPorts >= 0);

  close(sendFd[i]);
  sendFd[i] = -1;
  this->port[i] = -1;

  sendFd[i] = sendFd[numPorts];
  this->port[i] = this->port[numPorts];

  return;
}

int ProxyClient::IsChannelOpen(int port) {
  return (GetPortID(port) >= 0);
}



/* Open two connected UDP sockets (cntl, data) to the client port
 * return FALSE if connect failed, TRUE otherwise */
int ProxyClient::AddChannel(int port) {
  MyWarning("Channel %d added", port);
  assert(port != 0);

  if (AddPort(port) == FALSE) {
    assert(numPorts % 2 == 0);
    return FALSE;
  }

  if (AddPort(GetSisterPort(port)) == FALSE) {
    RemovePort(port);
    assert(numPorts % 2 == 0);
    return FALSE;
  }

  assert(numPorts % 2 == 0);
  return TRUE;
}


void ProxyClient::DropChannel(int port) {
  MyWarning("Channel %d dropped", port);

  RemovePort(port);
  RemovePort(GetSisterPort(port));
  assert(numPorts % 2 == 0);
}


int ProxyClient::GetAddr() {
  return clientAddr;
}


void ProxyClient::ReportPath() {
  if (monitorSource == -1) {
    monitorSource = GetQueryAgent()->GetSourceAddr();
  }
  
  if (monitorSource == GetMyAddr()) {
    return;
  }

  if (monitorSource == -1) {
    return;
  }

  int pathLen = GetQueryAgent()->GetPathLen(monitorSource);
  
  fprintf(stderr, "\nPATH %s->%s (%d): ", 
	  GetNameByAddr(monitorSource), GetNameByAddr(GetMyAddr()),
	  pathLen);
  
  if (pathLen < 0) {
    fprintf(stderr, "NO ROUTE");
    return;
  } 

  int *path = new int[pathLen];
  GetQueryAgent()->GetPath(path, monitorSource);
  
  
  for (int i=0; i< pathLen; i++) {
    fprintf(stderr, "%s ", GetNameByAddr(path[i]));
  }
  delete [] path;  
}


/* sends data to the client (buf is already decapsulated)
 * returns -1 if data cannot be sent (client drops out)
 *   case 1: client drops out: the channel is internally marked closed
 *   case 2: client does not subscribe to this channel, pkt is not sent
 * returns bufLen if data is sent successfully
 */
int ProxyClient::SendData(int pktDstPort, const char *buf, int bufLen) {
  /* XXX (need to do gc) */
  int i = GetPortID(pktDstPort);
  if (i < 0) return -1;

  switch(type) {
  case UNICAST: {
    
    errno = 0;
    if (send(sendFd[i], buf, bufLen, 0) < 0) {
      
      switch(errno) {
      case ECONNREFUSED:
	/* client left */
	
	DropChannel(pktDstPort);
	return -1;
	break;
      default:
	perror("send");
	MyError("error in send");
	break;
      }
    }
    break;
  }

  case MULTICAST: {
    // XXX: can try to do connect on this multicast socket...
    struct sockaddr_in sock;
    bzero((char *)(&sock), sizeof(sock));
    sock.sin_family = AF_INET;
    sock.sin_addr.s_addr = htonl(clientAddr);
    sock.sin_port = htons(pktDstPort);
    
    errno = 0;
    if (sendto(sendFd[i], buf, bufLen, 0,
	       (struct sockaddr *)&sock, sizeof(sock)) < 0) {
      
      switch(errno) {
      default:
	perror("send");
	MyError("error in send");
	return -1;
      }
    }
    break;
  }

  default:
    MyError("ProxyClient: type %d not implemented", type);
    break;
  }


  if (verbosity > -1) {
    struct EBReportList *reportList = bwSentToApp->UpdateAndReport(bufLen);
    while (reportList != NULL) {
      
      printf("\n%ld:BwSentToApp %s %s %d %d", 
	     reportList->time, 
	     getInetAddrName(clientAddr),
	     getInetAddrIP(clientAddr),
	     reportList->seq, reportList->byteCnt);
      
      struct EBReportList *next = reportList->next;
      free(reportList);
      reportList = next;
      
      if (reportList == NULL) {
	ReportPath();  /* print path */
      }
    }
  }

  return bufLen;
}


/* 
 * implementation
 * - hybernation primitive on proxy
 * - leader election among proxies
 *
 * this is actually quite complex:
 *
 * proxies multicast keepalive on another cntl channel
 * case 1: data is from yourself (loopback) -> drop
 * case 2: data is from your co-located client -> forward
 * case 3: data is from another co-located client -> forward
 * case 3: data is from another client on the LAN -> forward
 * case 4: data is from another proxy -> election between two proxies
 * case 5: data is from a client served by another proxy -> drop
 *
 * Now I assume ttl is always 0
 * XXX
 */

int ProxyClient::IsLoopbackPacket(int pktSrcAddr, int pktSrcPort) {

  if (pktSrcAddr == GetMyAddr()) {
    for (int i=0; i<numPorts; i++) {
      if (localSendPort[i] == pktSrcPort) {
	return TRUE;
      }
    }
  }

  return FALSE;
}


