#include <assert.h>
#include <stdlib.h>
#include <stdarg.h> /* va_list in FreeBSD and SUN */
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>
#include <netdb.h>

#include <arpa/inet.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/resource.h>  /* SUN rlimit */
#include <netinet/in.h>	/* sockaddr_in{} and other Internet defns */

#include <Gossip/global.h>
#include <Gossip/gossipAgent.h>  // GOSSIP_PORT
#include <Gossip/query.h>

#include <Util/inetTypes.h>
#include <Util/inetMisc.h>
#include <Util/misc.h>
#include <Util/packetBuffer.h>  // where MAX_UDP_PKTSIZE defined
#include <Internet/inetGossipHost.h>
#include <Internet/dataMgr.h>
#include <Internet/dataMgrUdp.h>
#include <Internet/dataMgrTcp.h>
#include <Internet/dataMgrTfrc.h>
#include <Internet/correlatedCongestion.h>

/* specific to proxies */
#include <Util/estimateBandwidth.h>
#include "listenPort.h"
#include "clientMgr.h"
#include "inetProxy.h"
#include "streamSelectorMgr.h"

/* porting from inetDrive note -- yhchu 
 * remove source sending
 * remove ETB
 */

/* global in inetDriver */
InetGossipHost *gossipHost;
DataMgr *dataMgrPtr;
CorrelatedCongestion *correlatedCongestionPtr;

const int INFINITE_TIME=100000;
int KILL_PORT=3333;
void Usage();

/* ETB declarations */
void ETBParse(char *optarg);
void ETBAction(float timeSinceStart);
int IsETBEnabled();

/* proxy-specific declaration */
EstimateBandwidth *bwRcvdFromNet;
int monitorSource = -1;

ListenPort *listenPortPtr[MAX_PORTS_PROXY_LISTENS];
int numListenPorts = 0;

ClientMgr *clientMgrPtr;

StreamSelectorMgr *streamSelectorMgrPtr = NULL;


/* common function calls */
void OutOfMemory(){
  cerr << "\n" << getInetAddrName(GetMyAddr());
  MyError("Ran out of memory!");
}

Query *GetQueryAgent(){
  return(gossipHost->GetQueryAgent());
}

void MyError(const char *format, ...) {
  va_list ap;

  fprintf(stderr, "\nERR - existing: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  fflush(stderr);

  printf("\nERR - exiting: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stdout, format, ap);
  fflush(stdout);

  assert(0);
  exit(1);
}


void MyWarning(const char *format, ...) {
  va_list ap;

  fprintf(stderr, "\nWRN: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  fflush(stderr);

  printf("\nWRN: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stdout, format, ap);
  fflush(stdout);
}


void DoExit() { 
  /* process about to exit, print status */
  printf("\nSTOP INTERNET RUN");
  
  /* XXX will cause some extra data to be printed */
  gossipHost->DoDebug();
  delete gossipHost;
  cout.flush();
  printf("\n");
  exit(0);
}


void FuncSendToNetwork(int toAddr, int toPort, const char *buf, int bufLen,
		       PacketType packetType, int priority) {


  dataMgrPtr->SendToNetwork(toAddr, toPort, buf, bufLen, packetType, priority);

  /*
    printf("\nProxy: SendToNetwork: write %d bytes to %s/%d network",
    bufLen, getInetAddrName(toAddr), toPort);
  */
}


void FuncRecvDataForApp(const char *encapBuf, int encapBufLen) {
  ApplicationIDType *appIDPtr = (ApplicationIDType *)encapBuf;
  int pktSrcAddr = htonl(appIDPtr->srcAddr);
  int pktDstPort = htons(appIDPtr->port);
  int pktSeqNum = htonl(appIDPtr->seqNum);

  if (verbosity > 1) {
    printf("\nProxy:SendToApp: port: %d", pktDstPort);
  }

  int i; for(i=0; i < numListenPorts; i++){
    if (pktDstPort == listenPortPtr[i]->GetPort()) {
      break;
    }
  }

  if(i >= numListenPorts){
    MyError("Packet to invalid port %d", pktDstPort);
  }

  
  int redirectPort = pktDstPort;
  if (streamSelectorMgrPtr != NULL) {
    redirectPort = streamSelectorMgrPtr->Recv(pktSrcAddr, 
					      pktDstPort, 
					      pktSeqNum);
    
    if (redirectPort == FALSE) return;
  }
  
  // send to all clients
  clientMgrPtr->SendData(redirectPort,
			 encapBuf+sizeof(ApplicationIDType), 
			 encapBufLen - sizeof(ApplicationIDType));
}


/*
 * translate DNS names of active members to IP addresses
 * if a hostname cannot be resolved, it will be taken out from the member list
 *
 * returns the adjusted number of active members
 */
int CreateMemberAddrList(int numActiveMembers,
			 char **memberNameList,
			 int *memberAddrListPtr) {

  int cnt = 0;
  
  for (int i=0; i< numActiveMembers; i++) {
    int remote_addr;

    if ((remote_addr = getInetAddrByName(memberNameList[i])) == -1) {
      MyWarning("hostname %s not found", memberNameList[i]);
      continue;
    }

    memberAddrListPtr[cnt] = remote_addr;
    if(verbosity > 0){
      printf("\nKNOWN_MEMBER %s %d",getInetAddrName(remote_addr),remote_addr);
    }

    cnt++;
  }
  
  return cnt;
}


struct MembershipChangeType{
  long time;
  int memberAddr;
  char changeCode;   // 'D' for death, 'J' for Join, 'I' for ignore
  int lowerDegreeBound;
  int upperDegreeBound;
};


void CreateNewDataMgr(char *dataMgrType, int wndsize){
  if (strcmp(dataMgrType, "udp") == 0) {
    dataMgrPtr = new DataMgrUdp(GOSSIP_PORT);
  } else if (strcmp(dataMgrType, "tcp") == 0) {
    dataMgrPtr = new DataMgrTcp(GOSSIP_PORT, wndsize);
  } else if (strcmp(dataMgrType, "tfrc") == 0) {
    dataMgrPtr = new DataMgrTfrc(GOSSIP_PORT);
  } else {
    printf("unknown transport option -T %s, expect -T {udp | tcp}\n\n", 
	   dataMgrType);
    Usage();
    exit(-1);
  }
}


void MiscInit(){
  /* test timezone */
  /* Also set random seed */
  struct timeval tv;
  struct timezone tz;
  
  
  if (verbosity > 1) {
    printf("\nMy host addr is %s, host name %s",
	   getInetAddrIP(GetMyAddr()), getInetAddrName(GetMyAddr()));
  }

  gettimeofday(&tv, &tz);
  if(verbosity > 0){
    printf("\nTIME: %ld %ld %d %d", 
	   tv.tv_sec, tv.tv_usec, tz.tz_minuteswest, tz.tz_dsttime);
  }
  srand(tv.tv_sec);

  if(verbosity > 0){
    printf("\nHOSTNAME %s", getInetAddrName(GetMyAddr()));
  }
}
  
void CheckParamsJoinLeaveTime(int joinTimeInSec,
			      int leaveTimeInSec){

  if(IsETBEnabled()){
    assert(joinTimeInSec == 0);
    assert(leaveTimeInSec == INFINITE_TIME);
  }

  if(joinTimeInSec < 0){
    MyError("JoinTime must be >= 0");
  }

  if(leaveTimeInSec < 0){
    MyError("LeaveTime must be >= 0");
  }
  
  if(joinTimeInSec > leaveTimeInSec){
    MyError("Join time must be smaller than leave time!!");
  }
}


void DoKill(int numActiveMembers, char **memberNameList) {
  struct sockaddr_in srcSock;
  const int on = 1;
  
  int net_fd = Socket(AF_INET, SOCK_DGRAM, 0);
  bzero((char *)&srcSock, sizeof(srcSock));
  srcSock.sin_family = AF_INET;
  srcSock.sin_addr.s_addr = htonl(INADDR_ANY);
  srcSock.sin_port = htons(KILL_PORT);
  Setsockopt(net_fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));
  Bind(net_fd, (struct sockaddr *)&srcSock, sizeof(srcSock));
    
  /* send kill UDP datagram 10 times */
  for (int j=0; j<10; j++) {
    for (int i=0; i< numActiveMembers; i++) {
      struct sockaddr_in dstSock;
      char buf[] = "KILL";
      int remote_addr;
      
      if ((remote_addr = getInetAddrByName(memberNameList[i])) == -1) {
	printf("\nhostname %s not found", memberNameList[i]);
	MyError("\nhostname not found");
	continue;
      }
      
      dstSock.sin_family = AF_INET;
      dstSock.sin_addr.s_addr = htonl(remote_addr);
      dstSock.sin_port = htons(GOSSIP_PORT);
      
      sendto(net_fd, (const char *)&buf, sizeof(buf), 0, 
	     (struct sockaddr *)&dstSock, sizeof(dstSock));

      if (j == 0)
	printf("\nKill Narada daemon on host %s", memberNameList[i]);
    }
  }
  printf("\n");
}


void Usage() {
  fprintf(stderr, "-h print out help\n");
  
  fprintf(stderr, "\nDebug:\n");
  fprintf(stderr, "-v [verbosity level], def=1\n");
  fprintf(stderr, "-V enable per-packet verbosity\n");

  fprintf(stderr, "\nLocal host config:\n");
  fprintf(stderr, "-N [host IP name or address] (if default to 127.0.0.1)\n");
  
  fprintf(stderr, "\nNarada config:\n");
  fprintf(stderr, "-D use static propagation delay instead of dynamic latency\n");
  fprintf(stderr, "-M [Protocol Metric] def=BW_LATENCY\n");
  fprintf(stderr, "-B [Probe Method] def=RTT\n");
  fprintf(stderr, "-P [gossip_port], def=10673 \n");
  fprintf(stderr, "-r [max source rate in Kbps], def=1200Kbps \n");
  fprintf(stderr,"-m [active member IP addr]+ (may set multiple -m flags)\n");
  fprintf(stderr, "-l [lower degree bound], def=3\n");
  fprintf(stderr, "-u [upper degree bound], def=6\n");
  fprintf(stderr, "-T [udp|tcp|tfrc] transport layer option for data forwarding, def=tcp\n");
  fprintf(stderr, "-w [TCP window size], def=65535 bytes\n");
  fprintf(stderr, "-H [performance history input filename]\n");
  fprintf(stderr, "-W [performance history output filename]\n");

  fprintf(stderr, "\nExperiment config:\n");
  fprintf(stderr, "-S [running time in minutes], def=20\n");
  fprintf(stderr, "-J [join time in sec], def=0\n");
  fprintf(stderr, "-L [join time in sec], def=INFINITE\n");
  fprintf(stderr, "-R [stay time on leave in sec], def=60 (collaborative leave mode)\n");
  fprintf(stderr, "-K kill remote Narada daemons \n");
  fprintf(stderr, "-Q [what port kill message will be sent from], def=3333\n");
  fprintf(stderr, "-X [paramters for ETB]\n");

  fprintf(stderr, "\nProxy-specific config:\n");
  // parameters for proxy that talks to the sender
  fprintf(stderr, "-y [proxy listen port:priority] (priority is optional)\n");
  
  // parameters proxy that talks to the receiver
  fprintf(stderr, "-x [correlation of ports]+\n");
  fprintf(stderr, "-t [multicast TTL value], def=0 (if ttl==1, avoid multiple proxies on same LAN)\n");
  fprintf(stderr, "-Y [proxy IP address to talk to the application] (for cygwin, use loopback 127.0.0.1)\n");
  fprintf(stderr, "-Z [source host] (for local data path monitoring)\n");
  
  // parameters for both sender+receiver proxy
  fprintf(stderr, "-g [Multicast group of application]\n");

  fprintf(stderr, "\n");
  fprintf(stderr, "example: \n");
  fprintf(stderr, "  - one audio (5000) two video streams (5002, 5004)\n");
  fprintf(stderr, "  - prioritize audio over video, and 5002 over 5004\n");
  fprintf(stderr, "\n");

  fprintf(stderr, "./proxy -y 5000:3 -y 5001:3 -y 5002:2 -y 5003:2 -y 5004:1 -y 5005:1 -x 5002:5004\n");
  fprintf(stderr, "./proxy -m marin.cmcl.cs.cmu.edu -y 5000:3 -y 5001:3 -y 5002:2 -y 5003:2 -y 5004:1 -y 5005:1 -x 5002:5004\n");
}


void InitiateLeave(int stayTimeOnLeave){
  gossipHost->LeaveGroup(stayTimeOnLeave);
}

int IsReadyToLeave(){
  return(gossipHost->IsReadyToLeave());
}

void SleepUntilJoinTime(int joinTimeInSec){
  if(joinTimeInSec > 0){
    fd_set fd;
    struct timeval tv;
    int retVal;
    
    cout << " Sleeping for " << joinTimeInSec << " seconds \n";
    FD_ZERO(&fd);
    tv.tv_sec = joinTimeInSec;
    tv.tv_usec = 0;
    
    retVal=select(0, NULL, NULL, NULL, &tv);
    if(retVal < 0){
      MyError("Select: Error");
    }
  }
}


int main(int argc, char *argv[]) {

  /***** Setting up default values *******/

  /*** Individual Gossip Host Params *****/
  // argc is the upper bound on how many members in the argument list 
  char **memberNameList = new (char *)[argc];
  int numActiveMembers = 0;
  int lowerDegreeBound = 3;
  int upperDegreeBound = lowerDegreeBound * 2;

  protocolMetric=BW_LATENCY;
  probeMethod = RTT;
  MAX_SOURCE_RATE=1200;
  verbosity = 0;
  
  /**** Parameters pertaining to dynamic membership ******/
  int joinTimeInSec=0;
  int leaveTimeInSec=INFINITE_TIME;
  int stayTimeOnLeaveInSec=60;

  /**** Parameters pertaining to Internet use ******/
  char *dataMgrType = "tcp";
  int wndsize = 65535;

  /**** Parameters pertaining to stopping Inetdriver ******/
  int stopTimeInMin = 20;  // default 20 min 
  int doKill = FALSE;

  /* proxy-specific parameters */
  int multicastAddr = -1; 
  char *localRecvAddrStr = NULL;
  int portsProxyListens[MAX_PORTS_PROXY_LISTENS];  // local to main
  int priorityPorts[MAX_PORTS_PROXY_LISTENS];

  int multicastTTL=0;
  int numPortsProxyListens=0;
  
  bwRcvdFromNet = new EstimateBandwidth(2, 1);
  
  if (argc <= 1) {
    fprintf(stderr, "If you want to invoke proxy without arg, use ./proxy -z\n");
    fprintf(stderr, "\n");
    Usage();
    exit(-1);
  }
  

  /* parse arguments */
  {
    char optstr[]="v:N:VDM:B:P:r:m:l:u:J:L:R:T:w:KQ:S:zhCH:W:y:t:Y:g:Z:x:";
    char c;
  
    while((c=getopt(argc,argv,optstr)) != -1){
      switch(c){

	/******** Parameters for Util Library ***********/
      case 'v':
	verbosity=atoi(optarg);
	break;
	
      case 'N':
	MyHostName = optarg; 
	break;
	
	/*********  Parameters for Gossip Library *********/
	
      case 'V':
	perPacketVerbosity=1;
	break;
	
      case 'D':
	PROPAGATION_DELAY_FLAG=1;
	break;
	
      case 'M':
	ParseProtocolMetric(optarg);
	break;

      case 'B':
	ParseProbeMethod(optarg);
	break;

      case 'P':
	GOSSIP_PORT=atoi(optarg);
	break;
	
      case 'r':
	MAX_SOURCE_RATE=atoi(optarg);
	break;

	/***** Parameters to tune individual Gossip Host ******/

      case 'm':
	memberNameList[numActiveMembers] = optarg;
	numActiveMembers++;
	break;

      case 'l':
	lowerDegreeBound=atoi(optarg);
	break;

      case 'u':
	upperDegreeBound=atoi(optarg);
	break;

	/******* Parameters to control dynamic join/leave *******/
	
      case 'J':
	joinTimeInSec=atoi(optarg);
	break;
      case 'L':
	leaveTimeInSec=atoi(optarg);
	break;
      case 'R':
	stayTimeOnLeaveInSec=atoi(optarg);
	break;

	/******* Parameters to configure Internet usage *****/
      case 'T':
	dataMgrType = optarg;
	break;
	
      case 'w':  /* update wndsize */
	wndsize = atoi(optarg); 
	break;

	/****** Parameters for InetDriver ******/
      case 'K':
	doKill = TRUE;
	break;
	
      case 'Q':
	KILL_PORT=atoi(optarg);
	break;

      case 'S':
	stopTimeInMin=atoi(optarg); /* min->ms */
	break;

	/****** Parameters for usage *********/
      case 'z':
	break;
      case 'h':
	Usage();
	exit(-1);

      case 'C': {
	/* used exclusively for ABone:
	 * create a binary copy of the executable, and rename to "proxy"
	 * this is because in ABone, the executable is erased once the program
	 * exits.
	 */
	char *binStr = "proxy";
	char srcStr[1000];
	char dstStr[1000];
	
	strcpy(srcStr, argv[0]);
	strcpy(dstStr, argv[0]);
	
	/* remove the characters after the last '/' */
	char *substr = strrchr(dstStr, '/');
	if (substr == NULL) {
	  dstStr[0] = '\0';
	} else {
	  substr[1] = '\0';
	}
	
	/* concat with the binary name "binStr" */
	sprintf(dstStr, "%s%s", dstStr, binStr);
	assert(strcmp(srcStr, dstStr) != 0);
	
	/* create and execute the command */
	char command[1000];
	sprintf(command, "/bin/cp -f %s %s", argv[0], dstStr);
	fprintf(stderr, "make a copy %s\n", command); 
	system(command);
	exit(0);
      }
	/****** Performance History *******/
      case 'H': 
	PerformanceHistoryReadFile = optarg;
	break;
      case 'W':
	PerformanceHistoryWriteFile = optarg;
	break;
	
	/* proxy-specific arguments */

      case 'y': {
	char bak[1000];
	strcpy(bak, optarg);

	portsProxyListens[numPortsProxyListens] = atoi(strtok(bak, ":"));
	char *str = strtok(NULL, ":");
	if (str != NULL) {
	  priorityPorts[numPortsProxyListens] = atoi(str);
	} else {
	  priorityPorts[numPortsProxyListens] = PRIORITY_DATA_DEFAULT;
	}
	numPortsProxyListens++;
	break;
      }

      case 'g':
	multicastAddr = getInetAddrByName(optarg);
	if (multicastAddr == -1) {
	  MyError("cannot resolve %s", optarg);
	}
	break;
	
      case 't':
        multicastTTL=atoi(optarg);
        if((multicastTTL < 0) || (multicastTTL > 1)){
           MyError("\n Invalid multicast TTL [must be 0 or 1]");
        }
        break;

	/* XXX: currently a hack for cygwin: specify the IP address of
	 * the interface that the proxy uses to listen on the
	 * (unicast) player's RTCP request (typically 127.0.0.1)
	 */
      case 'Y':
	localRecvAddrStr = optarg;
	break;

	// specify video quality: currently works for 2 only
      case 'x': {
	//int numQuality = 0;
	int qualityPorts[100];
	
	char bak[1000];
	strcpy(bak, optarg);
	qualityPorts[0] = atoi(strtok(bak, ":"));
	int numPorts = 1;

	char *str;
	while ((str = strtok(NULL, ":")) != NULL) {
	  qualityPorts[numPorts] = atoi(str);
	  numPorts++;
	}

	streamSelectorMgrPtr = new StreamSelectorMgr(qualityPorts, numPorts);
	break;
      }

      case 'Z':
	if ((monitorSource = getInetAddrByName(optarg)) == -1) {
	  MyWarning("hostname %s not found", optarg);
	}
	break;
	
      default:
	Usage();
	MyError("Invalid Option %c", c);
      }
    }
  }
  
  
  CheckParamsJoinLeaveTime(joinTimeInSec,leaveTimeInSec);

  /* start clock tick */
  long startTimeInMs = GetCurrTime();

  if (doKill) {
    DoKill(numActiveMembers, memberNameList);
    DoExit();
  }
  
  MiscInit(); //initializes random seed, checks time zone, some printfs
  CreateNewDataMgr(dataMgrType,wndsize); // creates appropriate dataMgr

  /* start a gossipHost */
  gossipHost = new InetGossipHost(GetMyAddr(), 
				  FuncSendToNetwork,
				  FuncRecvDataForApp);

  correlatedCongestionPtr = new CorrelatedCongestion(GOSSIP_PORT+2);

  /* open a UDP port for the Narada proxy 
   * three types of listen address (localRecvAddrStr):
   * - group address
   * - local interface address
   * - no address (use default GetMyAddr())
   */

  /* start a gossipHost */
  gossipHost = new InetGossipHost(GetMyAddr(), 
				  FuncSendToNetwork,
				  FuncRecvDataForApp);
  correlatedCongestionPtr = new CorrelatedCongestion(GOSSIP_PORT+2);

  {
    /*** JOIN: Sequence of actions taken till member joins.******/


    SleepUntilJoinTime(joinTimeInSec);
    
    int *memberAddrListPtr = new int[numActiveMembers];
    numActiveMembers = CreateMemberAddrList(numActiveMembers,
					    memberNameList,
					    memberAddrListPtr);

    gossipHost->JoinGroup(numActiveMembers, 
			  memberAddrListPtr,
			  lowerDegreeBound, 
			  upperDegreeBound,
			  1);  /* always a source??? XXX */
  }

  clientMgrPtr = new ClientMgr();

  int localRecvAddr = GetMyAddr();
  /* local receive addr */
  if (localRecvAddrStr != NULL) {
    if ((localRecvAddr = getInetAddrByName(localRecvAddrStr))==-1){
      MyError("cannot resolve %s", localRecvAddrStr);
    }
    
    /* HACK! XXX when QuickTime starts, it never send "probe" RTP/RTCP data
     * and so the proxy cannot discover QuickTime exists.
     * The solution: if localRecvAddrStr is set to 127.0.0.1, always blast
     * data to GetMyAddr()
     */
    if (localRecvAddr == INADDR_LOOPBACK) {
      int i; for(i=0; i < numPortsProxyListens; i++){
	int port = portsProxyListens[i];
	
	clientMgrPtr->UpdateUnicastClientAndChannel(GetMyAddr(),
						    port);
      }
    }
  }

  // unicast
  int i; for(i=0; i < numPortsProxyListens; i++){
    listenPortPtr[numListenPorts] = new ListenPort(localRecvAddr, 
						   portsProxyListens[i],
						   priorityPorts[i], 
						   gossipHost);
    numListenPorts++;
  }
  
  if (multicastAddr != -1) {

    for(int i=0; i < numPortsProxyListens; i++){
      int port = portsProxyListens[i];
      
      clientMgrPtr->UpdateMulticastClientAndChannel(multicastAddr,
						    multicastTTL, 
						    port);

      listenPortPtr[numListenPorts] = new ListenPort(multicastAddr, 
						     port,
						     priorityPorts[i], 
						     gossipHost);
      numListenPorts++;
    }
  }

  int fd_wid = GetFDWID();
  int leaveMode=0;
  fd_set rs, ws;
  for (;;) {
    float timeSinceStart = (GetCurrTime() - startTimeInMs)/1000.0;

    /***********
		Terminate if time is up, 
    ***************/
    if (timeSinceStart > (stopTimeInMin * 60)) {
      DoExit();
    }

    /**************
		NOTE: Currently two ways for leave are supported.
		Either ETB, or join/leave. So, there is some redundancy
		that can be cleaned up in the future.
    ****************/

    if ((!leaveMode) && (timeSinceStart > leaveTimeInSec)) {
      gossipHost->LeaveGroup(stayTimeOnLeaveInSec * 1000);
      leaveMode=1;

      cout << "\n" << GetCurrTime() << ":LEAVE " 
	   << GetNameByAddr(GetMyAddr()) << " COOPERATIVE IN "
	   << stayTimeOnLeaveInSec;
    }
    
    if(leaveMode && (gossipHost->IsReadyToLeave())){
      cout << "\n" << GetCurrTime() << ":LEAVE " 
	   << GetNameByAddr(GetMyAddr()) << " COOPERATIVE NOW "
	   << stayTimeOnLeaveInSec;
      DoExit();
      break; 
    }

    FD_ZERO(&rs); 
    FD_ZERO(&ws);

    /* proxy client */
    for (int i=0; i < numListenPorts; i++){
      listenPortPtr[i]->SetFD(&rs, &ws);
    }

    /**** pop the next event from the queue, if needed */
    long wait = gossipHost->PopNextEvent();
    wait = MIN(wait, dataMgrPtr->SetFD(&rs, &ws));
    wait = MIN(wait, correlatedCongestionPtr->SetFD(&rs, &ws));

    struct timeval wait_time = MsecToTimeval(wait);

    /* set wait_time to NULL if wait indefinitely */
    if (select(fd_wid, &rs, &ws, (fd_set *) NULL, &wait_time) < 0) {
      switch(errno) {
      case EINTR:
	break;  
      default:
	perror("select");
	assert(0);
      }
    }
    
    correlatedCongestionPtr->ReadFromNetwork(&rs, &ws);

    while (TRUE) {
      int fromAddr, fromPort, bufLen;
      char buf[MAX_PAYLOAD_SIZE];

      if ((bufLen=dataMgrPtr->ReadFromNetwork(&rs, &ws, buf, MAX_PAYLOAD_SIZE, 
					      &fromAddr, &fromPort)) < 0) {
	break;
      }

      /* XXX a hack to intercept and process the KILL message */
      if (fromPort == KILL_PORT) {
	printf("\n Stopping because of Kill signal");
	fprintf(stderr, "\nWRN: %s stopped because of Kill signal",
		getInetAddrName(GetMyAddr()));
	fflush(stdout);
        DoExit();
      }
      if(verbosity > 1){
	printf("\nNetwork: read %d bytes from %s/%d network",
	       bufLen, getInetAddrName(fromAddr), fromPort);
      }
      
      gossipHost->RecvFromNetwork(buf, bufLen, fromAddr, fromPort);
    }
    
    for(int i=0; i < numListenPorts; i++){ 
      listenPortPtr[i]->ReadFromNetwork(&rs, &ws);
    }
  }
  
  cerr << "CAME HERE!!!!!!";
  DoExit();
}
 


