#ifndef VRT_ENTRY_H
#define VRT_ENTRY_H

#include "global.h"

class ChildList;
class SourceEntry;
class ProbeResponseMsg;

enum BWResultType {BW_RESULT, BW_RESULT_LB};

class BWLevelMgr;
class RoutingMetric;
class PhysicalDelay;
class PhysicalBW;
class MyPath;

class VRTEntryType{

 private:

  friend class VirtualRoutingTable;

  static const int MAX_INITIAL_CONTROL_SEQ_NUM = 1000;

  int addr;
  int lastControlSeqNum;
  long time;

  
  RoutingMetric *routingMetric;
  RoutingMetric *linkMetric;
  PhysicalDelay *physicalDelay;
  PhysicalBW *physicalBW;
  MyPath *path;

  int numRTProbesSent; 
  int numRTProbesRcvd;
  int lastRTProbeSentTime; 
  int lastRTProbeRcvdTime;
  ProbeResponseMsg *lastRTProbe;
  
  int bwTestFailureFlag;
  int timeOfLastBWTestFailure;

  ChildList *childListPtr;
  int deadFlg;

  /* handle multiple sources */
  SourceEntry *sourceEntryPtr;


  int SetNoRouteEntry(int candidateNextHop);
  void UpdateRouteEntry(int candidateNextHop,
			RoutingMetric *metricViaCandidate,
			MyPath *pathFromCandidateToAddr
			);
  

  int SanityChecks(RoutingMetric *metricToCandidate,
		   RoutingMetric *metricFromCandidateToAddr,
		   MyPath *pathFromCandidateToAddr
		   );
  
  void Init(int addrIn);


 public:

  static const int NO_CONTROL_INFORMATION = -5;

  VRTEntryType(int addrIn,int myAddrFlg); 
  VRTEntryType(int addrIn, int lastControlSeqNumIn, long timeIn);
  ~VRTEntryType();
  


  /***************************
     The following functions change the routing (and other) state
     of the entry if external events are received.

     -New routing information is available
     -Neighbor is deleted
     -Neighbor becomes temporary
     -No control messages received from a member
  *****************************/

  /**********
   UpdateRoutingInformation:
   
   Return 1 if I should preemptively send a CONTROL_UPDATE to sender
   of UPDATE message.

   This happens in the following cases:
   -Update occurs: I start using sender as next hop
   -If I feel the sender will benefit a lot by sending message
   earlier. (currently: if he has NO_ROUTE, but I have a route).
   -If sender thinks I am next hop, but I have NO_ROUTE

   Return 2 if I should send a "triggered update" to all my neighbors.
   This happens if:
   -I was using sender as next hop, and find I have no route.
   -I was having NO_ROUTE, but now I start using sender as next hop.
   
   Returns 0 otherwise.


   *****************************************************
   Sanjay: I updated this for purposes of Sigcomm2002: 
   Triggers are eliminated (used only for fresh entries!!)
   ******************************************************

   Returns 1 if:

   -Update occurs: I start using sender as next hop 
                 -Even if I had NO_ROUTE before
   -If I feel the sender will benefit a lot by sending message
   earlier. (currently: if he has NO_ROUTE, but I have a route).
   -If sender thinks I am next hop, but I have NO_ROUTE


  *************/

  
  int UpdateRoutingInformation(int candidateNextHop,
			       int tempNbrFlag,
			       RoutingMetric *metricToCandidate,
			       RoutingMetric *metricFromCandidateToAddr,
			       int myAddr,
			       MyPath *pathFromCandidateToAddr
			       );
  
  int UpdateRoutingInformationActual(int candidateNextHop,
				     int tempNbrFlag,
				     RoutingMetric *metricToCandidate,
				     RoutingMetric *metricFromCandidateToAddr,
				     int myAddr,
				     MyPath *pathFromCandidateToAddr
				     );

  /*************
   UpdateSelfRoutingInformation:

   Might want to add message later in the following case:
   Neighbor has poor performance to me (say NO_ROUTE), and 
   preemptively send CONTROL_UPDATE. But not done at present
  ***************/
  
  void UpdateSelfRoutingInformation(int nbr,int nbrUsesMeAsNextHopFlg);
  
  /*************
   UpdateOnNbrDeletion:
   
   Returns 1 if deleted nbr destroys my route to someone. In this
   case, a triggered update is sent.
   Returns 0 otherwise.
  ****************/
  int UpdateOnNbrDeletion(int nbrDeleted);

  /************
  UpdateOnNbrTemp:
  
  Returns 1 if nbr being made TEMP affects this route to some other member.
  In future, this may result in origination of triggered update.
  Returns 0 otherwise.
  ***************/
  int UpdateOnNbrTemp(int nbrBecomeTemp);
  
  /*********
	   Might need to be updated to process triggered updates
  **********/
  void UpdateOnControlTimeout();
  int IsValidRoute();
  int IsNoRoute();

  /***************
	    Functions that identify a node as dead
  ****************/
  void MarkDead();
  void MarkAlive();
  int IsDead();


  /***********
	    Miscellaneous functions
  ***********/
  int GetAddr();
  int IsNextHop(int addr);
  int GetNextHop(int *nextHop);


  /********
      Both functions return a newly allocated copy of actual 
      routing metrics. Caller can mess around with returned 
      values, but must remember to free it.
  **********/
  RoutingMetric *GetRoutingMetric();
  RoutingMetric *GetLinkMetric();
  
  
  
  /************
	Functions pertaining to the control information stored for
	this address
  **************/

  void IncrementMyControlSeqNum(void);
  int GetLastControlSeqNum();
  int UpdateControlInformation(int lastControlSeqNumUpdate,long timeOfUpdate); 
  long GetTimeOfLastUpdate();
  
  /*************** 
	Functions that are handed over to PhysicalDelay.
	GetPhysicalDelay(), GetTimeSinceLastDelayProbe()
	return negative if no probe has been done, positive
	otherwise. IntegrateDelayResult() requires a non-negative
	delay.
  *****************/
  void IntegrateDelayResult(int roundTripDelay);
  int GetPhysicalDelay();
  long GetTimeSinceLastDelayProbe();
  int GetNumDelayProbes();
  
   /*************** 
	Functions that are handed over to PhysicalBW.
	GetPhysicalBW(), GetTimeSinceLastDelayProbe()
	return negative if no probe has been done, positive
	otherwise. GetNumBWResultProbes() returns the number of
	bandwidth probes that were actual estimates (as opposed
	to lower bounds). GetNumBWProbes() returns total number
	of all bandwidth probes.

	IntegrateBWResult requires a non-negative estBW value.
  *****************/

  void IntegrateBWResult(long estBW, BWResultType type);
  int GetPhysicalBW();
  int GetNumBWProbes();
  int GetNumBWResultProbes();
  long GetTimeSinceLastBWProbe();


  /*****************
      Functions that are handed over to the ChildList class.
      GetChildList() requires that adequate space be allocated
      for new children by the caller

      Children: People to whom data must be forwarded if packets
      are received for this particular source.
  ******************/

  int GetNumChildren();
  void GetChildList(int *childAddrList);
  int IsInChildList(int addr);
  
  /****************
	   SourceEntry is a class that keeps important statistics regarding
     addr if it is a source of data. The following functions pertain
     to this aspect.
     
     Warning: GetSourceEntry() returns a pointer to SourceEntry, which
     is private. Thus, modifications to the return value by the caller
     affect the entry itself.
  *****************/
  int IsSource();
  void SetSource();
  SourceEntry *GetSourceEntry();
  
  /*****************
    Functions that handle the "path" from the member to the address.
  ******************/
  MyPath *GetPath(); 

  /*****************
    Functions that keep track of routing table probes
  ******************/
  int GetTimeSinceLastRTProbeSent();
  int GetTimeSinceLastRTProbeRcvd();
  int GetNumRTProbesSent();
  int GetNumRTProbesRcvd();
  void UpdateOnRTProbeRcvd(ProbeResponseMsg *msgPtr);
  void UpdateOnRTProbeSent();
  ProbeResponseMsg *GetLastRTProbe();

  int GetTimeSinceLastBWTestFailure();
  void SetBWTestFailure();

  void Print();
  void PrintRouteChange();

};

#endif
