#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "global.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "gossipPayLoad.h"
#include "bwLevelMgr.h"
#include "METRICS/routingMetric.h"
#include "path.h"
#include "performanceHistory.h"

VirtualRoutingTable::VirtualRoutingTable(int maxTableSizeIn, 
					 int myAddrIn,
					 int numMembers,
					 int *memberArrayPtr){
  maxTableSize=maxTableSizeIn;
  myAddr=myAddrIn;
  
  table = new (EntryType *) [maxTableSize+1];
  for(int i=0; i <= maxTableSize; i++)
    table[i]=NULL;

  
  int hashValue=Hash(myAddr);
  VRTEntryType *vrtEntry=new VRTEntryType(myAddr,1);
  vrtEntry->IncrementMyControlSeqNum();
  EntryType *entryPtr=new EntryType;

  entryPtr->prev=NULL;
  entryPtr->next=NULL;
  entryPtr->hashValue=hashValue;
  entryPtr->vrtEntry=vrtEntry;
  table[hashValue]=entryPtr;
  firstEntry= entryPtr;
  

  numEntries=1;
  numDeadRoutes=0;

  /* do not add performance history for myself */
  performanceHistoryPtr = new PerformanceHistory(this);
  
  if (PerformanceHistoryReadFile != NULL) {
    assert(performanceHistoryPtr->ReadFromFile(PerformanceHistoryReadFile) == TRUE);
  }

  for(int i=0; i < numMembers; i++){
    AddRoute(memberArrayPtr[i]);
  }

  getNextRouteNextEntry=NULL;

  delete [] memberArrayPtr;
}

VirtualRoutingTable::~VirtualRoutingTable(){
  for(EntryType *entry=firstEntry; entry != NULL;){
    assert(entry->vrtEntry != NULL);
    delete entry->vrtEntry;
    EntryType *tempPtr=entry;
    entry=entry->next;
    delete tempPtr;
  }
  
  delete [] table;

  //printf("End of VRT deletion: %d",myAddr);
  /**********
  if(performanceHistoryPtr != NULL){
    delete performanceHistoryPtr;
  }
  ***********/
}

int VirtualRoutingTable::GetNumRoutes(){
  return(numEntries-numDeadRoutes);
}


/* return Route EntryNum, or -1 if not found
 */
VirtualRoutingTable::EntryType *VirtualRoutingTable::GetRouteEntryApproximate(int addr){
  
  int hashValue=Hash(addr);
  int i;

  for(i=hashValue; i >= 0; i--){
    if(table[i] != NULL){
      break;
    }
  }
  if(i < 0){
    return(NULL);
  }
  EntryType *entryPtr=table[i];
  assert(entryPtr != NULL);
  while(
	(entryPtr->hashValue == i) &&
	(entryPtr->vrtEntry->GetAddr() != addr)
	){
    if(entryPtr->next != NULL){
      entryPtr=entryPtr->next;
    }
    else{
      return(entryPtr);
    }
  }

  assert(entryPtr != NULL);

  if(entryPtr->hashValue != i){
    return(entryPtr->prev);
  }
  else{
    return(entryPtr);
  }
  
}


VirtualRoutingTable::EntryType *VirtualRoutingTable::GetRouteEntry(int addr){
  EntryType *entryPtr=GetRouteEntryApproximate(addr);
  if(entryPtr == NULL){
    return(NULL);
  }
  if(entryPtr->vrtEntry->GetAddr() == addr){
    return(entryPtr);
  }
  else{
    return(NULL);
  }
}



/* return the route corresponding the addr,
 * or NULL if route not found
 * a dead route will be returned
 */
VRTEntryType *VirtualRoutingTable::GetRoute(int addr){
  EntryType *routeEntry = GetRouteEntry(addr);
  if (routeEntry == NULL) return NULL;
  else return routeEntry->vrtEntry;
}


VRTEntryType *VirtualRoutingTable::GetNextRouteInit() {
  getNextRouteNextEntry = firstEntry;
  return GetNextRoute();
}

VRTEntryType *VirtualRoutingTable::GetNextRouteIncludingDeadInit() {
  getNextRouteNextEntry = firstEntry;
  return GetNextRouteIncludingDead();
}
/*********
       Skips over dead nodes
***********/

VRTEntryType *VirtualRoutingTable::GetNextRoute () {

  while(getNextRouteNextEntry != NULL) {
    VRTEntryType *route = getNextRouteNextEntry->vrtEntry;
    assert(route != NULL);
    if (route->IsDead()) {
      getNextRouteNextEntry=getNextRouteNextEntry->next;
      continue;
    }
    getNextRouteNextEntry=getNextRouteNextEntry->next;
    return route;
  }
  return NULL;
}

/*********
       Does not skip over dead nodes
***********/

VRTEntryType *VirtualRoutingTable::GetNextRouteIncludingDead() {

  while(getNextRouteNextEntry != NULL) {
    VRTEntryType *route = getNextRouteNextEntry->vrtEntry;
    assert(route != NULL);
    getNextRouteNextEntry=getNextRouteNextEntry->next;
    return route;
  }
  return NULL;
}

/*********
return NULL if route can't be added (exceed max addr),
       or if route already exists.
returns pointer to new entry if succeess 
***********/

VRTEntryType *VirtualRoutingTable::AddRoute(int addr){
  assert(addr != 0);  /* yhchu assert, debug only */

  EntryType *routeEntry = GetRouteEntryApproximate(addr);

  if(
     (routeEntry != NULL) &&
     (routeEntry->vrtEntry->GetAddr() == addr)
     ){
    return(NULL);
  }

  VRTEntryType *routeToAdd = new VRTEntryType(addr,0);

  EntryType *newEntry=new EntryType;
  newEntry->prev=NULL;
  newEntry->next=NULL;
  newEntry->hashValue=Hash(addr);
  newEntry->vrtEntry=routeToAdd;

  if(routeEntry == NULL){
    //printf("%d: routeEntry is NULL \n",addr);
    newEntry->next=firstEntry;
    if(firstEntry != NULL){
      firstEntry->prev=newEntry;
    }
    firstEntry=newEntry;
    
    assert(table[newEntry->hashValue] == NULL);
    table[newEntry->hashValue]=newEntry;
  }
  else  if (routeEntry != NULL) {
    //printf("%d: routeEntry is %d \n",addr,routeEntry->vrtEntry->GetAddr());
    newEntry->next=routeEntry->next;
    newEntry->prev=routeEntry;
    if((routeEntry->next) != NULL){
      (routeEntry->next)->prev=newEntry;
    }
    routeEntry->next=newEntry;
    if(table[newEntry->hashValue] == NULL){
      table[newEntry->hashValue]=newEntry;
    }
  }
  
  /* performance history */
  if (PerformanceHistoryReadFile != NULL) {
    performanceHistoryPtr->SetVrtEntry(routeToAdd);
  }

  numEntries++;
  return(routeToAdd);
}

/* delete the route, if exists */
void VirtualRoutingTable::DeleteRoute(int addr){
  EntryType *routeEntry = GetRouteEntry(addr);
  if (routeEntry != NULL) {
    delete routeEntry->vrtEntry;

    if(routeEntry->prev != NULL){
      (routeEntry->prev)->next=routeEntry->next;
    }
    else{
      firstEntry=routeEntry->next;
    }

    if(routeEntry->next != NULL){
      (routeEntry->next)->prev=routeEntry->prev;
    }

    if(table[routeEntry->hashValue] == routeEntry){
      if(
	 (routeEntry->next != NULL) &&
	 (routeEntry->next->hashValue == routeEntry->hashValue)
	 ){
	table[routeEntry->hashValue]=routeEntry->next;
      }
      else{
	table[routeEntry->hashValue]=NULL;
      }
    }
    
    delete routeEntry;
    numEntries--;
  }
  return;
}

void VirtualRoutingTable::MarkRouteDead(int addr){
  EntryType *routeEntry=GetRouteEntry(addr);
  if (routeEntry == NULL) return;
  if(!(routeEntry->vrtEntry->IsDead())){
    if(verbosity > 0){
      cout << "\n" << GetCurrTime() << ":" 
	   << GetNameByAddr(myAddr) 
	   << " marks " 
	   << GetNameByAddr(addr)
	   << " DEAD";
    }
    routeEntry->vrtEntry->MarkDead();
    numDeadRoutes++;
  }
}

void VirtualRoutingTable::MarkRouteAlive(int addr){
  EntryType *routeEntry=GetRouteEntry(addr);
  if (routeEntry == NULL) return;
  if(routeEntry->vrtEntry->IsDead()){
    routeEntry->vrtEntry->MarkAlive();
    numDeadRoutes--;
  }
}

/* modified for multi-source */
/***************
void VirtualRoutingTable::QuickPrint(){
  cout << "\n" << GetCurrTime() << ":VRT Begin 1";

  for(int i=0; i < numEntries; i++){
    VRTEntryType *route = table[i];
    assert(route != NULL);

    if (! route->IsSource()) continue;

    cout << "\n Addr " << route->GetAddr() 
	 << " Name " << GetNameByAddr(route->GetAddr()) << ":"; 
    route->Print();
  }

  cout << "\n VRT End";
}
********************/

int VirtualRoutingTable::PRINT_SOURCE_ALONE_FLAG=0;

void VirtualRoutingTable::Print(){
  cout << "\n";
  cout << "\n" << GetCurrTime() << ":Table of " << GetNameByAddr(myAddr);

  cout << "\n" << GetCurrTime() << ":VRT Begin " << numEntries;

  for(EntryType *entry=firstEntry; entry != NULL; entry=entry->next){
    if(PRINT_SOURCE_ALONE_FLAG){
      if(entry->vrtEntry->IsSource()){
	entry->vrtEntry->Print();
      }
    }
    else{
      entry->vrtEntry->Print(); 
    }
  }
  cout << "\n VRT End";

  cout << "\n";
}

/************
 UpdateTable:

 Returns 1 if must send Control-Update to sender of control-update
 Returns 2 if must trigger control updates to all neighbors
**************/

int VirtualRoutingTable::UpdateTable(int numRecords,
				     UpdateRecord *updateRecordPtr,
				     int nbrAddr,
				     int tempNbrFlg) {

  Check();

  int shouldTriggerFlag=0;

  VRTEntryType *nbrRoute = GetRoute(nbrAddr);
  if(nbrRoute == NULL) MyError("No route to a nbr!");
  RoutingMetric *metricToNbr=nbrRoute->GetLinkMetric();
  
  for(int recordNum=0; recordNum < numRecords; recordNum++){
    
    int addr,lastControlSeqNumOfAddrRcvdByNbr,aliveFlag;
    RoutingMetric *routingMetricFromNbrToAddr;
    MyPath *pathFromNbrToAddr;
    
    addr=updateRecordPtr[recordNum].GetAddr();
    lastControlSeqNumOfAddrRcvdByNbr=
      updateRecordPtr[recordNum].GetLastControlSeqNum();
    routingMetricFromNbrToAddr=
      updateRecordPtr[recordNum].GetRoutingMetric();
    pathFromNbrToAddr=
      updateRecordPtr[recordNum].GetPath();
    aliveFlag=
      updateRecordPtr[recordNum].GetAliveFlag();
    
    
    int nbrUsesMeAsNextHopToAddrFlg = 
      (pathFromNbrToAddr->IsNextHop(myAddr));
    
    if(!aliveFlag){
      VRTEntryType *updateRoute = GetRoute(addr);
      if(updateRoute != NULL){
	if(updateRoute->IsDead()){
	  updateRoute->UpdateControlInformation(
					lastControlSeqNumOfAddrRcvdByNbr,
					GetCurrTime()
					);
	}
	else{
	  int lastControlSeqNumOfAddrRcvdByMe=
	    updateRoute->GetLastControlSeqNum();
	  
	  if(
	     (
	      lastControlSeqNumOfAddrRcvdByNbr >=
	      lastControlSeqNumOfAddrRcvdByMe
	      )
	     ){
	    MarkRouteDead(updateRoute->GetAddr());
	  }
	}
      }
      continue;
    }


    VRTEntryType *updateRoute = GetRoute(addr);
    if (addr == myAddr) {
      assert(updateRoute != NULL);
      // I could be leaving.. assert(! updateRoute->IsDead());
      updateRoute->UpdateSelfRoutingInformation(nbrAddr,
						nbrUsesMeAsNextHopToAddrFlg);
      continue;
    } 

    if (updateRoute == NULL) {
      updateRoute=AddRoute(addr);
      assert(nbrUsesMeAsNextHopToAddrFlg==0);
    }

    if (updateRoute->IsDead()){
      //Allowing Rejoin
      if(
	 (updateRoute->GetLastControlSeqNum()) >=
	 lastControlSeqNumOfAddrRcvdByNbr
	 ){
	continue;
      }
      else{
	MarkRouteAlive(updateRoute->GetAddr());
      }
    }
    

    // route exists and is not dead
    if(!tempNbrFlg){
      // For a fresh entry, control info was not being updated earlier.
      // I added this now, but check for bugs etc.
      
      int oldControlStatus=
	(updateRoute->GetLastControlSeqNum() == 
	 VRTEntryType::NO_CONTROL_INFORMATION);
      
      updateRoute->UpdateControlInformation(lastControlSeqNumOfAddrRcvdByNbr,
					    GetCurrTime());
      
      int newControlStatus=
	(updateRoute->GetLastControlSeqNum() == 
	 VRTEntryType::NO_CONTROL_INFORMATION);
      
      if(oldControlStatus && !newControlStatus){
	shouldTriggerFlag=2;
      }
    }
    int routeTriggerFlag = 
      updateRoute->UpdateRoutingInformation(nbrAddr,
					    tempNbrFlg,
					    metricToNbr,
					    routingMetricFromNbrToAddr,
					    myAddr,
					    pathFromNbrToAddr
					    );

    assert(
	   (routeTriggerFlag == 0) ||
	   (routeTriggerFlag == 1)
	   );

    if(routeTriggerFlag == 1){
      if(shouldTriggerFlag == 0){
	shouldTriggerFlag=1;
      }
    }

    /************
    if(routeTriggerFlg > shouldTriggerFlag){
      shouldTriggerFlag=routeTriggerFlg;
    }
    ****************/
  }
  delete metricToNbr;
  Check();
  return(shouldTriggerFlag);
}

UpdateRecord *VirtualRoutingTable::CreateUpdateRecord(int *numRecords){

  //int numRoutes=GetNumRoutes();
  UpdateRecord *updateRecord = new UpdateRecord[numEntries];
  VRTEntryType *route=GetNextRouteIncludingDeadInit();

  for(int routeNum=0; routeNum < numEntries; routeNum++){
    int addr=route->GetAddr();
    int lastControlSeqNumOfAddrIRcvd=route->GetLastControlSeqNum();
    int aliveFlag=!(route->IsDead());
    RoutingMetric *routingMetric= route->GetRoutingMetric();
    MyPath *path=route->GetPath();

    updateRecord[routeNum].SetAddr(addr);
    updateRecord[routeNum].SetAliveFlag(aliveFlag);
    updateRecord[routeNum].SetLastControlSeqNum(lastControlSeqNumOfAddrIRcvd);
    updateRecord[routeNum].SetRoutingMetric(routingMetric);
    updateRecord[routeNum].SetPath(path);
    
    route=GetNextRouteIncludingDead();
  } 

  *numRecords=numEntries;
  return(updateRecord);
}

ProbeResponseRecord *VirtualRoutingTable::CreateProbeResponseRecord(int *numRecords){
  int numRoutes=GetNumRoutes();
  ProbeResponseRecord *probeResponseRecordPtr = 
    new ProbeResponseRecord[numRoutes];
  
  VRTEntryType *route=GetNextRouteInit();
  for(int routeNum=0; routeNum < numRoutes; routeNum++){
    
    int addr=route->GetAddr();
    int lastControlSeqNumOfAddrIRcvd=route->GetLastControlSeqNum();
    RoutingMetric *routingMetricToAddr= 
      route->GetRoutingMetric();

     probeResponseRecordPtr[routeNum].SetAddr(addr);
     probeResponseRecordPtr[routeNum].SetLastControlSeqNum(lastControlSeqNumOfAddrIRcvd);
     probeResponseRecordPtr[routeNum].SetRoutingMetric(routingMetricToAddr);

     route=GetNextRoute();
  }
  
  *numRecords=numRoutes;
  return(probeResponseRecordPtr);
}

   
   
int VirtualRoutingTable::UpdateOnNbrDeletion(int nbr){
  VRTEntryType *route;
  int connectivityLossFlag=0;
  for(route=GetNextRouteInit(); route != NULL;route=GetNextRoute()){
    if(route->UpdateOnNbrDeletion(nbr)){
      connectivityLossFlag=1;
    }
  }
  return(connectivityLossFlag);
}

void VirtualRoutingTable::UpdateOnNbrTemp(int nbr){
  VRTEntryType *route;
  int routesMadeTempFlag=0;
  for(route=GetNextRouteInit(); route != NULL;route=GetNextRoute()){
    if(route->UpdateOnNbrTemp(nbr)){
      routesMadeTempFlag=1;
    }
  }
  //return(routesMadeTempFlag);
}

int VirtualRoutingTable::GetMyAddr(){
  return(myAddr);
}

void VirtualRoutingTable::DoDebug() {
  if (PerformanceHistoryWriteFile != NULL) {
    assert(performanceHistoryPtr->WriteToFile(PerformanceHistoryWriteFile) == TRUE);
  }
}

int VirtualRoutingTable::Hash(int addr){
  int hash;
  
  if(addr < 0){
    hash=((-1) * addr) % maxTableSize;
  }
  else{
    hash=addr % maxTableSize;
  }
  
  return(hash);
}


void VirtualRoutingTable::Check(){

int numTotal=0;
int numDead=0;

VRTEntryType *route=GetNextRouteIncludingDeadInit();

 for(; route != NULL; route=GetNextRouteIncludingDead()){
   numTotal++;
   if(route->IsDead()){
     numDead++;
   }
 }

 assert(numTotal == numEntries);
 assert(numDead == numDeadRoutes);
}
