#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <assert.h>

#include "gossipPayLoad.h"
#include "pokeAgent.h"
#include "pokeMgr.h"

/*
 * 4/22/01, yhchu: 
 *    fixed POKE_DATA_SIZE over-estimate computation
 */


SmallProbeAgent::SmallProbeAgent(SmallProbeMgr *pokeMgrPtr, int remoteAddr, 
				 int remotePort,
				 int nonce, SmallProbeAgentType type,
				 int pokeRate) {
  
  /******************
    Rate pktsize period

    8192 1024    1
    4096 1024    2
    2048 1024    4
    1024 1024    8
    512  1024    16
    256  1024    32
    128  1024    64
    64   1024    128
    32   1024    256
   ********************/



  /*********
   pokeRate = pokeDataSize/pokeDataPeriod
   
   pokeRate is in KBits per second (or, Bits per millisecond)
   pokeDataSize is in Bytes, pokeDataPeriod is in milliseconds

   The algorithm below:
   -probes at a minimum rate of 32 Kbps (which involves a period of 256ms)
   -probes at a max rate of 8Mbps (pokeDataSize=1024,Period=1)
   -over estimates the rate at which it needs to probe:
    probes at:  (pokeDataSize)/(floor(pokeDataSize/pokeRate))

  ********/

  if (pokeRate <= 32) {
    pokeRate=32;
  }

  POKE_DATA_SIZE = 1024;
  POKE_DATA_PERIOD = (int) ((POKE_DATA_SIZE * 8.0) / (float) pokeRate);
  if(POKE_DATA_PERIOD < 1){
    POKE_DATA_PERIOD=1;
  }

  cout << "\n Starting SmallProbeAgent with:" 
       << " dataSize: " << POKE_DATA_SIZE
       << " dataPeriod: " << POKE_DATA_PERIOD
       << " actualSmallProbeRate: " << (POKE_DATA_SIZE/POKE_DATA_PERIOD) * 8;
    
  this->pokeMgrPtr = pokeMgrPtr;
  this->remoteAddr = remoteAddr;
  this->remotePort = remotePort;
  this->type = type;
  pokeTimer = new SmallProbeTimer(this);
  this->dataBuf = new char[POKE_DATA_SIZE];
  
  this->nonce = nonce;
  StartSmallProbe();
}


SmallProbeAgent::~SmallProbeAgent() {
  delete pokeTimer;
  delete [] dataBuf;
}



void SmallProbeAgent::StartSmallProbe() {
  switch(type) {
  case POKE_RECEIVER: {

    /* init */
    startTime = endTime = seqNum = sumDataLen = 0;
    sumSlowStartDataLen = 0;
    
    /* don't poke if saturation level is high 
     * but would continue waiting for some time for timeout
     */
    if (IsLocalCongestionSignal()) {
      printf("\nSmallProbeAgent Master: abort due to local congestion %s",
	     GetNameByAddr(remoteAddr));
    } else {
      if (verbosity > 0) {
	printf("\nSmallProbeAgent Master: start bw estimation to %s",
	       GetNameByAddr(remoteAddr));
      }

      SmallProbeRequestMsg *msg=new SmallProbeRequestMsg(nonce);
      pokeMgrPtr->SendToNetwork(remoteAddr, remotePort, POKE_REQUEST, msg);
    }

    /* wake up after probe finishes */
    pokeTimer->SetTimer(POKE_MAX_LAG_TIME*1000);
    break;
  }

  case POKE_SENDER: {

    if (verbosity > 0) {
      printf("\nSmallProbeAgent Slave: start bw estimation to %s",
	     GetNameByAddr(remoteAddr));
    }

    seqNum = 0;
    startTime = GetCurrTime();
    nextTime = startTime + POKE_DATA_PERIOD;
    endTime = startTime + POKE_SND_DURATION * 1000;
    
    pokeTimer->SetTimer(POKE_DATA_PERIOD);
    break;
  }

  default:
    MyError("unknown pokeAgent type");
    break;
  }
}


void SmallProbeAgent::Timeout(){
  
  switch(type) {
  case POKE_RECEIVER: {
    long curTime = GetCurrTime();
    
    /* hasn't started */
    if (startTime == 0) {
      if (verbosity > 0) {
	printf("\nSmallProbeAgent Master: bw estimation from %s failed (timeout)",
	       GetNameByAddr(remoteAddr));
      }

      pokeMgrPtr->Finish(this, -1);
      return;
    }

    assert(curTime >= (startTime + POKE_RCV_DURATION*1000));

    float dur = (lastTime - startTime)/1000.0 - POKE_SLOW_START_DURATION;
    
    /* don't report if you don't receive enough packets */
    if (dur < POKE_MIN_ESTIMATE) {
      printf("\nSmallProbeAgent Master: bw estimation from %s failed (did not receive dat long enough (%.2f)",
	     GetNameByAddr(remoteAddr), dur);
      
      pokeMgrPtr->Finish(this, -1);
      return;
    }
    
    /* I use dur instead of POKE_RCV_DURATION for accuracy */
    int rate = (int)(sumDataLen / dur * 8.0 / 1024.0);
    
    int rateSlowStart = (int)(((float)sumSlowStartDataLen) 
			      / POKE_SLOW_START_DURATION * 8.0 / 1024.0);
    
    if (verbosity > 0) {
      printf("\nSmallProbeAgent Master: bw estimation to %s is %d Kbps (%ld - %ld: %.2f) %d %d",
	     GetNameByAddr(remoteAddr), rate, startTime, lastTime, 
	     dur, rateSlowStart, sumDataLen);
    }

    pokeMgrPtr->Finish(this, rate);

    break;
  }

  case POKE_SENDER: {
    long currTime = GetCurrTime();
    
    /* add a safeguard where pokeAgent will send at most 100 packets
     * in this while loop (otherwise we may end up in an infinite loop
     */
    int seqNumBeforeLoop = seqNum;
    
    while (nextTime - currTime <= 0) {

      /* abort poke if saturation level is high */
      if (IsLocalCongestionSignal()) {
	/* XXX should wait for some time before returning
	 * or it'll end up emptying the poke queue
	 */
	printf("\nSmallProbeAgent Slave: abort due to local congestion %s",
	       GetNameByAddr(remoteAddr));
	
	pokeMgrPtr->Finish(this);
	return;
      }
      

      if (seqNum - seqNumBeforeLoop >= 100) {
	/* now we are really lagging behind */
	nextTime = currTime;

	fprintf(stderr, "\nWRN: SmallProbeAgent: Slave: %s cannot send poke data to %s fast enough", GetNameByAddr(GetMyAddr()), GetNameByAddr(remoteAddr));
	break;
      }

      /* send a probe data */
      SmallProbeDataMsg *msg=new SmallProbeDataMsg(POKE_DATA_SIZE, nonce, seqNum,
				       GetCurrTime(), POKE_DATA_RATE,
				       new char[POKE_DATA_SIZE]);
      pokeMgrPtr->SendToNetwork(remoteAddr, remotePort, POKE_DATA, msg);
      seqNum++;

      nextTime += POKE_DATA_PERIOD;

      // debug! XXX
      if (verbosity > 1 && seqNum % 100 == 0) {
	printf("\nSmallProbeAgent: Slave: %ld send data %d to %s",
	       currTime, seqNum, GetNameByAddr(remoteAddr));
      }
    }
    
    /* continue if time is not up yet */
    if (currTime - endTime <= 0) {
      pokeTimer->SetTimer(nextTime - currTime);
    } else {
      pokeMgrPtr->Finish(this);
    }
    
    break;
  }
  default:
    MyError("should not be here");
  }
}


void SmallProbeAgent::RecvSmallProbeData(SmallProbeDataMsg *msg){
  /* check nonce */
  int nonce = msg->GetNonce();
  if (nonce != this->nonce) {
    printf("\nSmallProbeAgent Master: wrong nonce %d %d", nonce, this->nonce);
    return;
  }
  
  long currTime = GetCurrTime();

  if (startTime == 0) {
    startTime = currTime;
    pokeTimer->DeleteTimer();
    pokeTimer->SetTimer(POKE_RCV_DURATION*1000);
  }
  
  lastTime = currTime;
  
  /* do not deal with duplicated packets */
  int dataSize = msg->GetDataSize();
  assert(POKE_DATA_SIZE == dataSize);

  if (((currTime - startTime)/1000.0) >= POKE_SLOW_START_DURATION) {
    sumDataLen += dataSize;
  } else {
    sumSlowStartDataLen += dataSize;
  }
}


int SmallProbeAgent::GetAddr() {
  return remoteAddr;
}


int SmallProbeAgent::GetPort() {
  return remotePort;
}


int SmallProbeAgent::GetType() {
  return type;
}


int SmallProbeAgent::GetMyAddr(){
  return(pokeMgrPtr->GetMyAddr());
}

