#include <assert.h>
#include <fstream.h>
#include <iostream.h>
#include <stdio.h>
#include "vrt.h"
#include "vrtEntry.h"
#include "neighborTable.h"
#include "gossipPayLoad.h"
#include "rttEstimationAgent.h"
#include "gossipAgent.h"
#include "path.h"
#include "TIMERS/rttEstimationTimer.h"

RTTEstimationAgent::RTTEstimationAgent(GossipAgent *gossipAgentPtrIn,
				       VirtualRoutingTable *vrtPtrIn,
				       NbrTable *nbrTabPtrIn,
				       int srcFlgIn){
  gossipAgentPtr=gossipAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;
  rttEstimationTimer=new RTTEstimationTimer(this);
  srcFlg=srcFlgIn;
  if(srcFlg){
    SendRTTEstimator();
  }
}

RTTEstimationAgent::~RTTEstimationAgent(){
  delete rttEstimationTimer;
}


void RTTEstimationAgent::SendRTTEstimator(){
  RTTEstimationMsg *msgPtr = new RTTEstimationMsg(GetMyAddr(),
						  GetCurrTime()
						  );
  
  VRTEntryType *route=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
  int numChildren=route->GetNumChildren();
  int *childAddrListPtr = new int[numChildren];
  route->GetChildList(childAddrListPtr);
  
  for(int i=0; i < numChildren; i++){
    RTTEstimationMsg *msgCopyPtr = msgPtr->Duplicate();
    //    msgCopyPtr->Print();
    // fflush(stdout);
    gossipAgentPtr->SendToNetwork(childAddrListPtr[i],
				  GOSSIP_PORT,
				  RTT_ESTIMATOR,
				  msgCopyPtr);

  }
  delete [] childAddrListPtr;
  delete msgPtr;
  fflush(stdout);
  rttEstimationTimer->SetTimer(RTT_ESTIMATION_PKT_PERIOD);
}


void RTTEstimationAgent::SendUpstream(RTTEstimationMsg *msgPtr, 
				      int downStreamAddr
				      ){

  //printf("Entered SendUpstream");
  //fflush(stdout);
  
  int status=msgPtr->AppendReversePath(GetMyAddr(),
				       downStreamAddr,
				       GetCurrTime());
  if(status == 0){
    // Failure, because I didn't see the packet in the forward path
    // Or because this guy was not my successor
    return;
  }
  
  //Now forward to the predecessor
  int predecessor=msgPtr->GetPredecessor(GetMyAddr());
  RTTEstimationMsg *msgCopyPtr = msgPtr->Duplicate();
  // msgCopyPtr->Print();
  // fflush(stdout);
  gossipAgentPtr->SendToNetwork(predecessor,
				GOSSIP_PORT,
				RTT_ESTIMATOR,
				msgCopyPtr);


}

void RTTEstimationAgent::RecvRTTEstimator(int fromAddr,
					  int fromPort,
					  RTTEstimationMsg *msgPtr){
  if(
     !nbrTabPtr->IsNbr(fromAddr) ||
     (fromPort != GOSSIP_PORT)
     ){
    return;
  }
  
  // msgPtr->Print();
  // fflush(stdout);
  int srcAddr=msgPtr->GetSourceAddr();
  RTTMsgDirnType dirn=msgPtr->GetDirn();
  
  switch(dirn){
  case FORWARD:
    {
      VRTEntryType *route=vrtPtr->GetRoute(srcAddr);
      if(route == NULL){
	return; // No route for source
      }
      if(!route->IsNextHop(fromAddr)){
	return;
      }
      int status=msgPtr->AppendForwardPath(GetMyAddr(),GetCurrTime());
      if(status == 0){
	// Failure, because of a cycle.
	return;
      }
      
      // Successful
      int numChildren=route->GetNumChildren();
      if(numChildren > 0){
	int numChildren=route->GetNumChildren();
	int *childAddrListPtr = new int[numChildren];
	route->GetChildList(childAddrListPtr);
	
	for(int i=0; i < numChildren; i++){
	  RTTEstimationMsg *msgCopyPtr = msgPtr->Duplicate();
	  // msgCopyPtr->Print();
	  // fflush(stdout);
	  gossipAgentPtr->SendToNetwork(childAddrListPtr[i],
					GOSSIP_PORT,
					RTT_ESTIMATOR,
					msgCopyPtr);

	}
	delete [] childAddrListPtr;
      }
      else{
	RTTEstimationMsg *msgReplyPtr=msgPtr->Duplicate();
	msgReplyPtr->SetDirn(REVERSE);
	SendUpstream(msgReplyPtr,-1);
	delete msgReplyPtr;
      }
      
      break;
    }
    
  case REVERSE:
    {
      if(srcAddr == GetMyAddr()){
	EstimateRTT(msgPtr);
	return;
      }
      
      VRTEntryType *route=vrtPtr->GetRoute(srcAddr);
      if(route == NULL){
	return; // No route for source
      }
      
      SendUpstream(msgPtr,fromAddr);
      break;
    }

  default:
    MyError("Invalid direction");
    break;
  }
}

void RTTEstimationAgent::EstimateRTT(RTTEstimationMsg *msgPtr){
  int pathLen;
  int path[MyPath::MAX_PATH_LEN];
  long forwardTime[MyPath::MAX_PATH_LEN];
  long reverseTime[MyPath::MAX_PATH_LEN];
  
  pathLen = msgPtr->GetPathLen();
  assert(pathLen <= MyPath::MAX_PATH_LEN);

  msgPtr->GetPath(path);
  msgPtr->GetForwardTime(forwardTime);
  msgPtr->GetReverseTime(reverseTime);
  
  int srcRTT=GetCurrTime() - forwardTime[0];
  
  for(int i=1; i < pathLen; i++){
    int memberRTT= srcRTT - (reverseTime[i] - forwardTime[i]);
    if (verbosity > 0){
      cout << "\n" << GetCurrTime() << ":"
	   << "RTT to " << GetNameByAddr(path[i]) 
	   << " is " << memberRTT;
    }
  }
}

int RTTEstimationAgent::GetMyAddr(){
  return(gossipAgentPtr->GetMyAddr());
}
