#include <iostream.h>
#include <iomanip.h>
#include <stdio.h>
#include <fstream.h>
#include <assert.h>

#include <Util/inetMisc.h>
#include "global.h"
#include "vrtEntry.h"
#include "vrt.h"
#include "performanceHistory.h"


/* format 
 * my IP, remote IP, raw delay, smooth delay, raw bw, discretized bw
 */
PerformanceHistory::PerformanceHistory(VirtualRoutingTable *vrtPtr) {
  this->vrtPtr=vrtPtr;
  head = NULL;
}


PerformanceHistory::~PerformanceHistory() {
  PerformanceHistoryType *record = head;
  
  while (record != NULL) {
    PerformanceHistoryType *tmp;
    record = record->next;
    delete tmp;
  }
}


/* write performance info to a file 
 *
 * XXX: one potential problem is that the physical bandwidth info can
 * be outdated depending on when the last measurement was taken.
 * Currently we do not record the last bw measurement time
 */
int PerformanceHistory::WriteToFile(char *file) {  

  assert(file != NULL);

  ofstream ofs(PerformanceHistoryWriteFile, ios::out);
  if (ofs == NULL) MyError("Error opening packet loss file");
  
  cerr << "WriteToFile " << GetNameByAddr(vrtPtr->GetMyAddr()) << "\n";

  for(VRTEntryType *route=vrtPtr->GetNextRouteInit(); route != NULL; 
      route=vrtPtr->GetNextRoute()) {

    ofs << vrtPtr->GetMyAddr() << " " 
	<< GetNameByAddr(vrtPtr->GetMyAddr()) << " "
	<< route->GetAddr() << " " << GetNameByAddr(route->GetAddr()) << " ";

    int delay = route->GetPhysicalDelay();
    ofs << " " << delay;
    
    int bw = route->GetPhysicalBW();
    ofs << " " << bw;

    ofs << "\n";
  }
  
  ofs.close();
  return TRUE;
}


void PerformanceHistory::PrintRecord(PerformanceHistoryType *record) {
  cout << "\n" << GetNameByAddr(record->myIP) << "(" << record->myIP << ")"
       <<" -> "<< GetNameByAddr(record->remoteIP)<<"("<<record->remoteIP<<")"
       << " Delay " << record->delay 
       << " BW " << record->bw;
}


/* read performance info from a file */
int PerformanceHistory::ReadFromFile(char *file) {
  assert(PerformanceHistoryReadFile != NULL);

  FILE *fin = fopen(file, "r");
  if (fin == NULL) MyError("Error opening performance history");
  
  int maxline = 1000;
  char str[maxline];
  char tmp1[maxline];
  char tmp2[maxline];
  while (fgets(str, maxline, fin) != NULL) {
    PerformanceHistoryType *record = new PerformanceHistoryType;
    
    sscanf(str, "%d %s %d %s %d %d",
	   &(record->myIP), tmp1, &(record->remoteIP), tmp2, 
	   &(record->delay), &(record->bw));
    
    if (record->myIP != vrtPtr->GetMyAddr()) {
      PrintRecord(record);
      cout << " NOT MINE";

      delete record;
      continue;
    }

    PrintRecord(record);

    if (head == NULL) {head = record; record->next = NULL;}
    else {record->next = head; head = record;}
  }
  
  fclose(fin);
  
  return TRUE;
}


/* */
int PerformanceHistory::SetVrtEntry(VRTEntryType *route) {
  PerformanceHistoryType *record;

  for (record = head; record != NULL; record = record->next) {
    if (record->remoteIP != route->GetAddr()) 
      continue;

    /* record found */
    PrintRecord(record);
    
    switch(probeMethod) {
    case RANDOM: 
    case RANDOM_10K:
    case RANDOM_TFRC:
    case RTT:
    case RTT_10K:
    case RTT_TFRC:
    case RTT_10K_TFRC:
      cout << " No history";
      break;
    case HISTORY: 
      if (record->delay >= 0) {
	route->IntegrateDelayResult(record->delay*2);  /* rtt */
      }
      if (record->bw >= 0) {
	route->IntegrateBWResult(record->bw, BW_RESULT);
      }
      cout << " HISTORY";
      break;
    case GNP: 
    case GNP_10K:
    case GNP_10K_TFRC:
    case GNP_TFRC:
      if (record->delay >= 0) {
	route->IntegrateDelayResult(record->delay*2);  /* rtt */
      }
      cout << " GNP";
      break;
    default:
      MyError("ProbeMethodType %d not supported", probeMethod);
      break;
    }

    cout << " VrtEntry set";
    return TRUE;
  }

  return FALSE;  
}
