#include <iostream.h>
#include <stdlib.h>
#include <assert.h>
#include <stdio.h>

#include "partitionAgent.h"
#include "gossipAgent.h"
#include "connectivityAgent.h"
#include "neighborTable.h"
#include "vrt.h"
#include "vrtEntry.h"

#include "TIMERS/partitionTimer.h"
#include "TIMERS/rttProbeTimer.h"

/**********************
 
 1. Get a list  of guys to whom there is no route.
 2. Select who to probe, based on RTT/10K probes etc.
 3. Send ADD_NBR_REQUEST to this guy.
 4. If response, add link. Otherwise try someone else.
 5. If a member has not responded after several attempts,
    declare it dead.
************************/




PartitionAgent::PartitionAgent(GossipAgent *gossipAgentPtrIn, 
			       ConnectivityAgent *connectivityAgentPtrIn,
			       VirtualRoutingTable *vrtPtrIn, 
			       NbrTable *nbrTabPtrIn){
  gossipAgentPtr=gossipAgentPtrIn;
  connectivityAgentPtr=connectivityAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;
  
  partitionTimer = new PartitionTimer(this);
  
  numPartitionedNodes=0;
  partitionListPtr=NULL;
  
  rttProbeTimer=new RTTProbeTimer(this);

  seqNum=0;

}

PartitionAgent::~PartitionAgent(){
  PartitionList *ptr=partitionListPtr;
  
  while(ptr != NULL){
    PartitionList *tempPtr=ptr;
    ptr=ptr->next;
    delete tempPtr;
  }

  delete rttProbeTimer;
  delete partitionTimer;
}

void PartitionAgent::PartitionCheck(){
  RefreshPartitionList(); 
  if(partitionListPtr != NULL){
    DoRepair();
  }
  
  //  partitionTimer->SetTimer(PARTITION_TIMER);
  return;
}

void PartitionAgent::RecvAddNbrResponse(int fromAddr,
					int fromPort,
					AddNbrResponseMsg *msgPtr){

  RefreshPartitionList();

  // Go through list of partitioned nodes and 
  // check if response matches any particular request.
  PartitionList *ptr;
  for(ptr=partitionListPtr; (ptr != NULL) && (ptr->addr != fromAddr); ptr=ptr->next);
  
  if((ptr != NULL) &&
     (ptr->addr == fromAddr) &&
     (fromPort == GOSSIP_PORT) &&
     /******** 
	 (ptr->expectedResponseSeqNum == msgPtr->GetRemoteSeqNumOfPrevMsg()) && 
     *********/
     (
      (msgPtr->GetType() == PARTITION) ||
      (msgPtr->GetType() == PARTITION_PRIORITY) ||
      (msgPtr->GetType() == PARTITION_URGENT)
      )
     ){
    
    VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
    assert(route != NULL);
    
    //*** ptr->timer->DeleteTimer(); **** HAVE ?? ***//
    
    if(verbosity > 0){
      cout << "\n" << GetCurrTime() << ": ADD-NBR-PARTITION " 
	   << GetNameByAddr(GetMyAddr()) 
	   << " "  << GetNameByAddr(fromAddr) << " ";
    }
    
    connectivityAgentPtr->ProcessAddNbrResponse(fromAddr,
						fromPort,
						msgPtr);
  }
  else{
    cout << " REJECTED ";
  }
  
  RefreshPartitionList();
}


void PartitionAgent::RefreshPartitionList(){
  PartitionList *prevPtr,*ptr;

  // First prune off from the list nodes that have had an update

  ptr=partitionListPtr;
  prevPtr=NULL;
  
  while(ptr != NULL){
    VRTEntryType *route=vrtPtr->GetRoute(ptr->addr);
    if(route == NULL){
      MyError("No route for member in partitioned list !!");
    }
    
    if(route->IsValidRoute()){
      assert(
	     (GetCurrTime() - route->GetTimeOfLastUpdate()) <= 
	     MAX_CONTROL_SEQ_NUM_AGE
	     );
      
      // Must have had an update: Remove from queue.
      ptr=DeleteFromPartitionList(ptr,prevPtr);
    }
    else if(route->IsDead()){
      ptr=DeleteFromPartitionList(ptr,prevPtr);
    }
    else{
      if(ptr->numTries >= NUM_PROBES_TO_DETERMINE_DEAD){
	vrtPtr->MarkRouteDead(ptr->addr);
	if(verbosity > 0){
	  cout << " (PARTITION)";
	}
	ptr=DeleteFromPartitionList(ptr,prevPtr);
	
      }
      else{
	prevPtr=ptr;
	ptr=ptr->next;
      }
    }
  }
  
  // Next add fresh entries to the queue if required
  
  for(VRTEntryType *route=vrtPtr->GetNextRouteInit(); 
      route != NULL ; 
      route=vrtPtr->GetNextRoute()){
    
    /***************
    This guy is potentially partitioned if he does not have a route
    *****************/

    if(route->IsValidRoute()){
      assert(
	     (GetCurrTime() - route->GetTimeOfLastUpdate()) <
	     MAX_CONTROL_SEQ_NUM_AGE
	     );
    }

    if(route->IsDead()){
      assert(0);
    }
    
    if(!route->IsValidRoute()){
      int addr=route->GetAddr(); 
      assert(addr != GetMyAddr());
      if(nbrTabPtr->IsNbr(addr)){
	assert(nbrTabPtr->IsTempNbr(addr));
	continue;
      }
      if(IsInPartitionList(addr)) continue;
      InsertInPartitionList(addr);
      if(verbosity > 0){
	cout << "\n" << GetCurrTime() << ":" 
	     << GetNameByAddr(GetMyAddr()) 
	     << " has NO ROUTE to " 
	     << GetNameByAddr(addr);
      }
    }
  }
}

int PartitionAgent::GetMyAddr(){
  //return(15);
  return(gossipAgentPtr->GetMyAddr());
}



PartitionAgent::PartitionList *PartitionAgent::ProbeSelect() {
  PartitionList *nodeToTry = NULL;

  switch (probeMethod) {
  case RANDOM: 
  case RANDOM_10K:
  case RANDOM_TFRC:
    {
      do{
	nodeToTry=GetRandomElement();
      } while(IsInShortListArr(nodeToTry->addr));
      
      break;
    }
    
  case RTT:
  case RTT_10K:
  case RTT_TFRC:
  case GNP: 
  case GNP_10K:
  case GNP_TFRC:
  case GNP_10K_TFRC:
  case RTT_10K_TFRC:
  case HISTORY:
    {

      int numTries, delay, bw;
      int isFirst = TRUE;
      
      /* heuristic:
       * - select hosts only if they are currently not connected 
       * - bypass hosts that have tried previously
       * - prioritize hosts with minimum delay
       */

      PartitionList *ptr;

      for(ptr=partitionListPtr; ptr != NULL; ptr=ptr->next){
	if(ptr->numTries >= NUM_PROBES_TO_DETERMINE_DEAD) continue;
	if (IsInShortListArr(ptr->addr)) continue;
	
	VRTEntryType *route = 
	  vrtPtr->GetRoute(ptr->addr);
	
	int routeDelay = route->GetPhysicalDelay();
	int routeBw = route->GetPhysicalBW();
	
	/* if no delay info, max route delay */
	if (routeDelay < 0) routeDelay = 10000;
	
	/* priortize bw over delay */
	if ((isFirst == TRUE) ||
	    (ptr->numTries < numTries) ||
	    (ptr->numTries == numTries &&
	     (routeBw > bw || (routeBw == bw && routeDelay < delay)))) {  
	  isFirst = FALSE;
	  nodeToTry = ptr;
	  numTries = ptr->numTries;
	  delay = routeDelay;
	  bw = routeBw;
	}
      }
    }
  }
  return nodeToTry;
}
 


void PartitionAgent::SendAddNbrRequest(int addr){

  int numRoutes;
  UpdateRecord *addNbrRequestRecord = vrtPtr->CreateUpdateRecord(&numRoutes);
  AddNbrRequestMsg *msg;

  PartitionList *nodeToTryPtr=GetNodeGivenAddr(addr);
  
  AddNbrRequestType requestType=PARTITION;
  if(nodeToTryPtr->numTries > 2){
    requestType=PARTITION_PRIORITY;
  }
  if(nodeToTryPtr->numTries > 4){
    requestType=PARTITION_URGENT;
  }
  
  msg=new AddNbrRequestMsg(-1,
			   GetCurrTime(),
			   -1,
			   seqNum,
			   requestType,
			   numRoutes,
			   addNbrRequestRecord);
  
  
  //Make entry in table
  nodeToTryPtr->numTries++;
  nodeToTryPtr->seqNum=seqNum;
  seqNum++;
  
  if(!nbrTabPtr->IsNbr(nodeToTryPtr->addr)){
    gossipAgentPtr->SendToNetwork(nodeToTryPtr->addr,
				  GOSSIP_PORT,
				  ADD_NBR_REQUEST,
				  msg);
    int randJoinResponseTimer = (rand()%((PARTITION_TIMER-2000)*2))+2000;
    
    // backoff heuristic
    int numNbrs = nbrTabPtr->GetNumNbrs();
    assert(numNbrs >= 0);
    randJoinResponseTimer *= (numNbrs+1);

    partitionTimer->SetTimer(randJoinResponseTimer);
  }
  else{
    //partitionTimer->SetTimer(0);
    MyError("Should not send ADD_NBR_REQ to neighbor!!");
    assert(0);
  }
}

/* hack: wait for some time to initiate join: this is because
 * some hosts may start at different time
 */
void PartitionAgent::StartJoin() {
  partitionTimer->SetTimer(WAIT_TIME_BEFORE_JOIN);
}


void PartitionAgent::DoRepair(){
  switch(probeMethod){
  case RANDOM:
  case RANDOM_TFRC:
  case GNP:
  case GNP_TFRC:
  case HISTORY:
    {
      DoShortList(1);
      SendAddNbrRequest(shortListArr[0]);
    }
    
    break;

  case RANDOM_10K:
  case GNP_10K:
  case GNP_10K_TFRC:
    {

      DoShortList(NUM_MEMBERS_TO_SHORTLIST);
      connectivityAgentPtr->ScheduleSmallProbeTest(shortListArr,
						   numShortListedMembers,
						   this);
      
    }
				  
    break;
    
  case RTT:
  case RTT_10K:
  case RTT_TFRC:
  case RTT_10K_TFRC:
    {
      int send=SendRTTProbes();
      if(send){
	rttProbeTimer->SetTimer(RTT_PROBE_TIMER);
      }
      else{
	rttProbeTimer->SetTimer(0); // RTT estimates available for all
      }
    }
    break;
    
  default:
    MyError("Invalid Probe Method!!!");
  }
}

int PartitionAgent::SendRTTProbes(){
  int sendRTTProbes=0;

  PartitionList *ptr=partitionListPtr;
  for(; ptr != NULL; ptr=ptr->next){
    VRTEntryType *route=vrtPtr->GetRoute(ptr->addr);
    assert(route != NULL);
    int numDelayProbes=route->GetNumDelayProbes();
    if(numDelayProbes < 0){
      sendRTTProbes=1;
      JoinRequestMsg *msgPtr=
	new JoinRequestMsg(0,GetCurrTime());
      gossipAgentPtr->SendToNetwork(ptr->addr,
				    GOSSIP_PORT,
				    JOIN_REQUEST,
				    msgPtr);
    }
  }
  return(sendRTTProbes);
}

void PartitionAgent::RecvJoinResponse(int fromAddr,
				      int fromPort,
				      JoinResponseMsg *msgPtr){
  
  if(fromPort != GOSSIP_PORT){
    return;
  }
  
  // get the route 
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL){
    return;
  }
  
  int roundTripDelay=GetCurrTime() - msgPtr->GetRemoteTimeRequestSent();
  route->IntegrateDelayResult(roundTripDelay);
  
  if(fromAddr == timeSyncServerAddr){
    int offset= (msgPtr->GetRemoteTimeRequestSent() + roundTripDelay/2)
      -  (msgPtr->GetLocalTimeRequestRcvd());
    if(verbosity > 0){
      cout << "\n" << GetCurrTime() << ":"
	   << "Offset from " << GetNameByAddr(timeSyncServerAddr)
	   << "(" << GetNameByAddr(fromAddr) << ")"
	   << " is " << offset; 
    }
  }
}

void PartitionAgent::Timeout(){
  PartitionCheck();
}

void PartitionAgent::ReportFromSmallProbe(int candidateAddr) {

  RefreshPartitionList();
  if(partitionListPtr == NULL){
    return;
  }

  if(verbosity > 0){
    printf("\nSmall Probe Chooses JOIN: %s",GetNameByAddr(candidateAddr));
  }

  if(!IsInPartitionList(candidateAddr)){
    return;
  }
  
  switch(probeMethod){
  case RANDOM_10K:
  case GNP_10K:
  case RTT_10K:
  case GNP_10K_TFRC:
  case RTT_10K_TFRC:
    if(candidateAddr == -1){
      SendAddNbrRequest(shortListArr[0]);
    }
    else{
      SendAddNbrRequest(candidateAddr);
    }
    break;
    
  default:
    MyError("Should not come here!!! \n");
  }
}
  
  
void PartitionAgent::DoShortList(int maxDesiredShortListLen){
			    
  int i;
  PartitionList *ptr;

  numShortListedMembers=0;
  for(int i=0; i < NUM_MEMBERS_TO_SHORTLIST; i++){
    shortListArr[i]=-1;
  }
  
  for(ptr=partitionListPtr;ptr != NULL;ptr=ptr->next) {
    assert(ptr->numTries < NUM_PROBES_TO_DETERMINE_DEAD);
  }

  if(numPartitionedNodes > maxDesiredShortListLen){
    int i;
    for(i=0; i < maxDesiredShortListLen; i++){
      PartitionList *nodeToTry;
      nodeToTry =  ProbeSelect();  
      shortListArr[numShortListedMembers] = nodeToTry->addr;
      numShortListedMembers++;
    }
  }
  else{
    // Try every unprobed member
    PartitionList *node;
    for(node=partitionListPtr; node != NULL; node=node->next){
      shortListArr[numShortListedMembers]=node->addr;
      numShortListedMembers++;
    }
  }
  
  
  if(verbosity > 0){
    printf("\nNum Shortlisted JOIN: %d",numShortListedMembers);
  }
  
  for(i=0; i < numShortListedMembers; i++){
    VRTEntryType *route = 
      vrtPtr->GetRoute(shortListArr[i]);
    if(route == NULL){
      MyError("No route to short listed guy!");
    }
    
    if(verbosity > 0){
      printf ("\nShortListed JOIN: %s Delay: %d", 
	      GetNameByAddr(shortListArr[i]),
	      route->GetPhysicalDelay()
	      );
    }
  }
  assert(numShortListedMembers > 0);
}


int PartitionAgent::IsInShortListArr(int addr){ 

  int i;
  
  for(i=0; i < numShortListedMembers; i++){
    if(shortListArr[i] == addr){
      return(1);
    }
  }
  
  return(0);
}

void PartitionAgent::RTTProbeTimeout(){

  RefreshPartitionList();
  if(partitionListPtr == NULL){
    return;
  }

  switch(probeMethod){
  case RTT:
  case RTT_TFRC:
    {
      DoShortList(1);
      SendAddNbrRequest(shortListArr[0]);
    }

    break;
    
  case RTT_10K:
  case RTT_10K_TFRC: 
    {
      DoShortList(NUM_MEMBERS_TO_SHORTLIST);
      connectivityAgentPtr->ScheduleSmallProbeTest(shortListArr,
						   numShortListedMembers,
						   this);
      
    }
    
    break;
    
  default:
    MyError("Should not come here!!!");
  }
}

