#include <iostream.h>
#include <assert.h>

#include "neighborTable.h"
#include "global.h"
#include "gossipPayLoad.h"

NbrTable::NbrTable(){
  numNbrs=0;
  nbrList=NULL;

}

NbrTable::~NbrTable(){
  for(NbrRecord *ptr=nbrList; ptr != NULL;){
    NbrRecord *prevPtr=ptr;
    ptr=ptr->next;
    delete prevPtr->bwRcvdFromNbr;
    delete prevPtr->bwSentToNbr;
    if(prevPtr->lastControlUpdate != NULL){
      delete prevPtr->lastControlUpdate;
    }
    delete prevPtr;
  }
}
    
void NbrTable::AddNbr(int addr,long timeSinceNbr){
  NbrRecord *nbrRecord = new NbrRecord;
  nbrRecord->addr=addr;
  nbrRecord->tempFlag=0;
  nbrRecord->timeSinceNbr=timeSinceNbr;

  nbrRecord->timeOfLastControlUpdate=timeSinceNbr;
  nbrRecord->lastControlUpdate = NULL;

  nbrRecord->bwSentByNbr=UNDEFINED_RATE;
  nbrRecord->lastBWReportTime=NO_REPORT;
  nbrRecord->bwRcvdFromNbr 
    = new EstimateBandwidth(ESTIMATE_BW_HISTORY, ESTIMATE_BW_UNIT);
  nbrRecord->bwSentToNbr
    = new EstimateBandwidth(ESTIMATE_BW_HISTORY, ESTIMATE_BW_UNIT);
  nbrRecord->numDataPktsSentToNbr = 0;
  nbrRecord->timeLastDataPktSentToNbr = -1;
  nbrRecord->timeCurrSpurtStarts = -1;
  nbrRecord->next=nbrList;
  nbrList=nbrRecord;
  numNbrs++;
}

void NbrTable::DropNbr(int addr){
  if(nbrList == NULL) MyError("Dropping non-existent nbr!!");
  if(nbrList->addr == addr){
    NbrRecord *tempPtr=nbrList;
    nbrList=nbrList->next;
    
    delete tempPtr->bwRcvdFromNbr;
    delete tempPtr->bwSentToNbr;
    if(tempPtr->lastControlUpdate != NULL){
      delete tempPtr->lastControlUpdate;
    }
    delete tempPtr;
    numNbrs--;
    return;
  }
  
  NbrRecord *ptr;
  for(ptr=nbrList; ptr->next != NULL && ptr->next->addr != addr; ptr=ptr->next);
  if(ptr->next == NULL) MyError("Dropping non-existent nbr!!");
  NbrRecord *tempPtr=ptr->next;
  ptr->next=tempPtr->next;
  if(tempPtr->lastControlUpdate != NULL){
    delete tempPtr->lastControlUpdate;
  }
  delete tempPtr->bwRcvdFromNbr;
  delete tempPtr->bwSentToNbr;
  delete tempPtr;
  numNbrs--;
}

void NbrTable::MarkNbrTemp(int addr){
  NbrRecord *ptr;
  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  if(ptr == NULL){
    MyError("Marking a nbr temp when it does not exist");
  }
  ptr->tempFlag=1;
  ptr->timeSinceTempNbr=GetCurrTime();
}


int NbrTable::GetNumNbrs(){
  return(numNbrs);
}

int NbrTable::GetNumNonTempNbrs(){
  int numNonTempNbrs=0;
  
  NbrRecord *ptr;
  for(ptr=nbrList; ptr != NULL; ptr=ptr->next){
    if(!ptr->tempFlag){
      numNonTempNbrs++;
    }
  }
 return(numNonTempNbrs);
}    


void NbrTable::GetNbrList(int *nbrAddrList){
  NbrRecord *ptr;
  int i;
  
  for(i=0,ptr=nbrList; ptr != NULL; ptr=ptr->next,i++)
    nbrAddrList[i]=ptr->addr;
}

long NbrTable::GetTimeSinceNbr(int addr){
  NbrRecord *ptr;
  for(ptr=nbrList; ptr != NULL && ptr->addr != addr; ptr=ptr->next);
  if (ptr == NULL) MyError("Not nbr, yet asking time since nbr!!");
  return(ptr->timeSinceNbr);
  
}

long NbrTable::GetTimeSinceTempNbr(int addr){
  NbrRecord *ptr;
  for(ptr=nbrList; ptr != NULL && ptr->addr != addr; ptr=ptr->next);
  if ((ptr == NULL) || (!ptr->tempFlag) )
      MyError("Not temp nbr, yet asking time since temp nbr!!");
  return(ptr->timeSinceTempNbr);

}
int NbrTable::IsNbr(int addr){
  NbrRecord *ptr;
  for(ptr=nbrList; ptr != NULL && ptr->addr != addr; ptr=ptr->next);
  if(ptr != NULL) return(1);
  else return(0);
}

int NbrTable::IsTempNbr(int addr){
  NbrRecord *ptr;
  for(ptr=nbrList; ptr != NULL && ptr->addr != addr; ptr=ptr->next);
  if((ptr != NULL) && ptr->tempFlag) return(1);
  else return(0);
}

int NbrTable::DoExistTempNbrs(){
  NbrRecord *ptr;
  for(ptr=nbrList; (ptr != NULL) && (!ptr->tempFlag) ; ptr=ptr->next);
  if(ptr == NULL) return(0);
  else return(1);
}

void NbrTable::UpdateStatsRcvdFromNbr(int addr, int bytes){
  NbrRecord *ptr;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  ptr->bwRcvdFromNbr->Update(bytes);
}

void NbrTable::UpdateStatsSentToNbr(int addr, int bytes){
  NbrRecord *ptr;
  long currTime;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  ptr->bwSentToNbr->Update(bytes);
  
  currTime = GetCurrTime();
  if ( 
      (ptr->numDataPktsSentToNbr == 0) ||
      (
       (ptr->numDataPktsSentToNbr > 0) &&
       ((currTime - ptr->timeLastDataPktSentToNbr) > MAX_IDLE_TIME)
       )
      ){
    /**********
	       No packet has been sent to this neighbor for MAX_IDLE_TIME
	       This is a new spurt of traffic
    *************/
    ptr->timeCurrSpurtStarts=currTime;
  }
  ptr->timeLastDataPktSentToNbr = currTime;
  ptr->numDataPktsSentToNbr++;
}


void NbrTable::UpdateRateSentByNbr(int addr, long newSendRate){
  NbrRecord *ptr;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  ptr->bwSentByNbr=newSendRate;
}

long NbrTable::GetRateRcvdFromNbr(int addr){
  NbrRecord *ptr;
  long rate;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  rate = ptr->bwRcvdFromNbr->Report();
  if(rate == EstimateBandwidth::NO_REPORT_AVAILABLE){
    return(UNDEFINED_RATE);
  }
  else{
    return(rate);
  }
}

long NbrTable::GetRateSentToNbr(int addr){
  long rate;
  NbrRecord *ptr;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  rate = ptr->bwSentToNbr->Report();
  if(rate  == EstimateBandwidth::NO_REPORT_AVAILABLE){
    return(UNDEFINED_RATE);
  }
  if((ptr->numDataPktsSentToNbr > 0) &&
     (GetCurrTime() - ptr->timeCurrSpurtStarts) < MIN_SPURT_LENGTH){
    return(UNDEFINED_RATE);
  }
  else{
    return(rate);
  }
}

long NbrTable::GetRateSentByNbr(int addr){
  NbrRecord *ptr;
  
  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  return(ptr->bwSentByNbr);
}

void NbrTable::UpdateLastBWReportTime(int addr){
  NbrRecord *ptr;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  ptr->lastBWReportTime = GetCurrTime();
}

long NbrTable::GetLastBWReportTime(int addr){
  NbrRecord *ptr;

  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  assert(ptr != NULL);
  return(ptr->lastBWReportTime);
}

int NbrTable::PruneNbrList(long currTime,int timeout, int **madeTmpNbrArrayPtr){
  NbrRecord *ptr=nbrList;
  
  *madeTmpNbrArrayPtr = new int[numNbrs];
  int numMadeTmpNbrs=0;
  
  while(ptr != NULL){
    if((currTime - ptr->timeOfLastControlUpdate) > timeout){
      if(!ptr->tempFlag){
	ptr->tempFlag=1;
	ptr->timeSinceTempNbr=GetCurrTime();
	(*madeTmpNbrArrayPtr)[numMadeTmpNbrs]=ptr->addr;
	numMadeTmpNbrs++;
      }
    }
    ptr=ptr->next;
  }
  return(numMadeTmpNbrs);
}

  
int NbrTable::NbrStatusAndRefresh(int addr,
				  long currTime, 
				  ControlUpdateMsg *controlUpdate){
  NbrRecord *ptr;
  for(ptr=nbrList; (ptr != NULL) && (ptr->addr != addr); ptr=ptr->next);
  if(ptr == NULL){
    return(0);
  }
  ptr->timeOfLastControlUpdate=currTime;
  if(controlUpdate != NULL){
    if(ptr->lastControlUpdate != NULL){
      delete ptr->lastControlUpdate;
    }
    ptr->lastControlUpdate=controlUpdate;
  }
  if(!ptr->tempFlag) return(1);
  else return(2);
}

ControlUpdateMsg *NbrTable::GetLastControlUpdate(int addr){
  NbrRecord *ptr;
  for(ptr=nbrList; ptr != NULL && ptr->addr != addr; ptr=ptr->next);
  if (ptr == NULL) MyError("Not nbr, yet asking time since nbr!!");
  return(ptr->lastControlUpdate);
}
  
void NbrTable::Print(){
  cout << "\n Nbr Table BEGIN";
  for(NbrRecord *ptr=nbrList; ptr != NULL; ptr=ptr->next)
    cout << "\n Addr " << ptr->addr << " Name " << GetNameByAddr(ptr->addr) 
	 << " TimeSinceNeighbor " << ptr->timeSinceNbr << " timeoflastupdate " 
	 << ptr->timeOfLastControlUpdate;
  cout << "\n Nbr Table END";
}
