#include <stdio.h>
#include <assert.h>

#include "leaveAgent.h"
#include "gossipAgent.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "neighborTable.h"
#include "TIMERS/leaveTimer.h"

LeaveAgent::LeaveAgent(GossipAgent *gossipAgentPtrIn,
		       VirtualRoutingTable *vrtPtrIn,
		       NbrTable *nbrTabPtrIn){
  gossipAgentPtr=gossipAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;
  leaveTimer = new LeaveTimer(this);
}

LeaveAgent::~LeaveAgent(){
  delete leaveTimer;
}

void LeaveAgent::InitLeave(int stayOnLeaveTimeIn){

  stayOnLeaveTime=stayOnLeaveTimeIn;
  beginLeaveTime=GetCurrTime();

  int numNbrs=nbrTabPtr->GetNumNbrs();
  int *nbrAddrList= new int[numNbrs];
  nbrTabPtr->GetNbrList(nbrAddrList);
  
  // Send a INTENT_TO_CANCEL_NBR message to all neighbors. 
  // Mark each of them temporary

  for(int i=0; i < numNbrs; i++){
    VRTEntryType *route=vrtPtr->GetRoute(nbrAddrList[i]);
    if(route == NULL) MyError("Nbr absent or marked dead in VRT Table!");
    if(route->IsDead()) assert(nbrTabPtr->IsTempNbr(nbrAddrList[i]) == 1);
    
    nbrTabPtr->MarkNbrTemp(nbrAddrList[i]); // Mark neighbor temporary
    vrtPtr->UpdateOnNbrTemp(nbrAddrList[i]);

    // Send IntentToCancel Neighbor messages to each of these neighbors 
    IntentToCancelNbrMsg *msg=new IntentToCancelNbrMsg(1);
    gossipAgentPtr->SendToNetwork(nbrAddrList[i],
				  GOSSIP_PORT,
				  INTENT_TO_CANCEL_NBR,
				  msg);
  }

  // Set myself to be dead!
  vrtPtr->MarkRouteDead(gossipAgentPtr->GetMyAddr());
  delete [] nbrAddrList;
  CheckIfReadyToLeave();
}

void LeaveAgent::CheckIfReadyToLeave(){
  int numNbrs;
  
  // Check if no neighbors remain. If so, ready to leave.
  numNbrs=nbrTabPtr->GetNumNbrs();
  
  if(
     (numNbrs == 0) ||
     ((GetCurrTime() - beginLeaveTime) >= stayOnLeaveTime)
     ){
    
    if(numNbrs > 0){
      int *nbrAddrList= new int[numNbrs];
      nbrTabPtr->GetNbrList(nbrAddrList);
      
      for(int i=0; i < numNbrs; i++){
	CancelNbrMsg *msg=new CancelNbrMsg(1);
	gossipAgentPtr->SendToNetwork(nbrAddrList[i],
				      GOSSIP_PORT,
				      CANCEL_NBR,
				      msg);
      }
      delete [] nbrAddrList;
    }
    
    gossipAgentPtr->ReadyToLeave();
    return;
  }
  
  leaveTimer->SetTimer(PERIODIC_LEAVE_CHECK_TIMER);
}

int LeaveAgent::GetMyAddr(){
  return(gossipAgentPtr->GetMyAddr());
}



