#ifndef GOSSIP_PAYLOAD_H
#define GOSSIP_PAYLOAD_H

#include "global.h"
#include "payLoad.h"
#include <Util/serializableObject.h>
#include "path.h"

/* Description and use of messages

   RTT_ESTIMATOR: rtt measurement collection between the source and
   each receiver

   ESTIMATE_DELAY_REQUEST/ESTIMATE_DELAY_RESPONSE: keepalive between
   neighbors, also update latency info
 */


class InputSerializableStream;
class OutputSerializableStream;
class RoutingMetric;

enum GossipMsgType {ADD_NBR_REQUEST,ADD_NBR_RESPONSE,
		    NBR_CONFIRM_REQUEST,NBR_CONFIRM_RESPONSE,
		    CONTROL_UPDATE, 
		    PROBE_REQUEST, PROBE_RESPONSE,
		    JOIN_REQUEST,JOIN_RESPONSE,
		    INTENT_TO_CANCEL_NBR, CANCEL_NBR,
		    ESTIMATE_DELAY_REQUEST,ESTIMATE_DELAY_RESPONSE,
		    DATA,
		    POKE_REQUEST, POKE_DATA,
		    SMALL_PROBE_REQUEST, SMALL_PROBE_DATA,
                    RTT_ESTIMATOR};

enum AddNbrRequestType {PARTITION, PARTITION_PRIORITY, PARTITION_URGENT, PERFORMANCE};
const long UNDEFINED_RATE = -1;
typedef AddNbrRequestType AddNbrResponseType;

struct GossipMsgHdr{
  GossipMsgType code;
  int size;
};

class GossipPayLoad: public PayLoad{
 private:
  GossipMsgHdr hdr;
  SerializableObject *gossipMsgPtr;

 public:
  GossipPayLoad(GossipMsgType codeIn,SerializableObject *gossipMsgPtrIn);
  ~GossipPayLoad();

  GossipMsgType GetCode();
  SerializableObject *GetMsg();
  char *GetMsgCodeString();
  PacketType GetPacketType();

  GossipPayLoad *Duplicate();
  int Size();
  void Print();

  GossipPayLoad();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

class UpdateRecord: public SerializableObject{
 private:
  int addr;
  int lastControlSeqNum;
  int aliveFlag;
  RoutingMetric *routingMetricPtr;
  MyPath *path;
  
 public:
  UpdateRecord();
  ~UpdateRecord();

  void SetAddr(int addrIn);
  void SetLastControlSeqNum(int lastControlSeqNumIn);
  void SetAliveFlag(int aliveFlagIn);

  /*********
    Directly uses routingMetricPtrIn, and does not make
    a fresh copy!!
  *********/

  void SetRoutingMetric(RoutingMetric *routingMetricPtrIn);
  void SetPath(MyPath *pathIn);

  int GetAddr();
  int GetLastControlSeqNum();
  int GetAliveFlag();

  /********
       Returns the metric directly, and not a copy
       Deleting this, or changing this affects the message!!
  *********/
  RoutingMetric *GetRoutingMetric();
  MyPath *GetPath();

  void Print() ;
  
  UpdateRecord *Duplicate();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
  int Size();
};


class AddNbrResponseMsg:public SerializableObject{
 private:
  long remoteTimeOfPrevMsg;
  long localTimeOfThisMsg;
  int remoteSeqNumOfPrevMsg; // This message in response to what seq num
  int localSeqNumOfThisMsg;				  
  int numRecords;
  UpdateRecord *recordPtr;
  AddNbrRequestType type; // Not used for response, only for request

 public:
  AddNbrResponseMsg(long remoteTimeOfPrevMsgIn,
		    long localTimeOfThisMsgIn,
		    int remoteSeqNumOfPrevMsgIn,
		    int localSeqNumOfThisMsgIn,
		    AddNbrRequestType typeIn,
		    int numRecordsIn,
		    UpdateRecord *recordPtrIn);
  ~AddNbrResponseMsg();

  long GetRemoteTimeOfPrevMsg();
  long GetLocalTimeOfThisMsg();
  int GetRemoteSeqNumOfPrevMsg();
  int GetLocalSeqNumOfThisMsg();
  int GetNumRecords();
  UpdateRecord *GetRecord(int recordNum);
  UpdateRecord *GetRecordArray();

  int Size();
  AddNbrRequestType GetType();
  AddNbrResponseMsg *Duplicate();
  void Print();

  AddNbrResponseMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

typedef AddNbrResponseMsg AddNbrRequestMsg;
typedef AddNbrResponseMsg NbrConfirmRequestMsg;
typedef AddNbrResponseMsg NbrConfirmResponseMsg;

class ProbeRequestMsg:public SerializableObject{
 private:
  long time;
  int seqNum;
 public:
  ProbeRequestMsg(int seqNumIn,long timeIn);
  long GetTime();
  int GetSeqNum();
  ProbeRequestMsg *Duplicate();
  int Size();
  void Print();

  ProbeRequestMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};



class ProbeResponseRecord:public SerializableObject{
 private:
  int addr;
   int lastControlSeqNum;
  RoutingMetric *routingMetricPtr;


 public:
  
  ProbeResponseRecord();
  ~ProbeResponseRecord();

  void SetAddr(int addrIn);
  void SetLastControlSeqNum(int lastControlSeqNumIn);
  void SetRoutingMetric(RoutingMetric *routingMetricPtrIn);

  
  int GetAddr();
  int GetLastControlSeqNum();
  RoutingMetric *GetRoutingMetric();

  void Print();

  ProbeResponseRecord *Duplicate();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
  int Size();
};


enum SaturationCodeType {LOW,MEDIUM,HIGH};
class ProbeResponseMsg:public SerializableObject{
 private:
  int seqNum;
  long remoteTimeRequestSent;
  long localTimeRequestRcvd;
  int numRecords;
  SaturationCodeType saturationCode; 
  ProbeResponseRecord *recordPtr;
  
 public:
  ProbeResponseMsg(int seqNumIn,
		   long remoteTimeRequestSent,
		   long localTimeRequestRcvd,
		   SaturationCodeType saturationCodeIn, 
		   int numRecordsIn,
		   ProbeResponseRecord *recordPtrIn);

  ~ProbeResponseMsg();
  int GetNumRecords();
  int GetSeqNum();
  int GetRemoteTimeRequestSent();
  int GetLocalTimeRequestRcvd();
  SaturationCodeType GetSaturationCode();
  ProbeResponseRecord *GetRecord(int recordNum);
  ProbeResponseMsg *Duplicate();
  int Size();
  void Print();
  
  ProbeResponseMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

typedef ProbeRequestMsg JoinRequestMsg;
typedef ProbeResponseMsg JoinResponseMsg;  

class ControlUpdateMsg:public SerializableObject{
 private:
  int seqNum;
  int numRecords;
  UpdateRecord *recordPtr;

 public:
  ControlUpdateMsg(int seqNumIn,int numRecordsIn,UpdateRecord *recordPtrIn);
  ~ControlUpdateMsg();
  int GetNumRecords() ;
  int GetSeqNum() ;
  UpdateRecord *GetRecord(int recordNum);
  UpdateRecord *GetRecordArray();
  ControlUpdateMsg *Duplicate();
  int Size();
  void Print();

  ControlUpdateMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

class EstimateDelayRequestMsg:public SerializableObject{
 private:
  long time;
  int lastControlSeqNum;
 public:
  EstimateDelayRequestMsg(long timeIn,int lastControlSeqNumIn);
  ~EstimateDelayRequestMsg(void);
  long GetTime(void);
  int GetLastControlSeqNum(void);
  EstimateDelayRequestMsg *Duplicate(void);
  int Size();
  void Print();
  
  EstimateDelayRequestMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

class EstimateDelayResponseMsg:public SerializableObject{
 private:
  long time;
  int lastControlSeqNum;

 public:
  EstimateDelayResponseMsg(long timeIn,int lastControlSeqNumIn);
  ~EstimateDelayResponseMsg(void);
  long GetTime(void);
  int GetLastControlSeqNum(void);
  EstimateDelayResponseMsg *Duplicate(void);
  int Size();
  void Print();

  EstimateDelayResponseMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

class DataMsg:public SerializableObject{
 private:
  int sourceAddr;
  int seqNum;
  int dataSize;
  long time; // local time of source when it sent the packet.
  long lastHopSendRate; 
  // this field is modified at each hop and denotes the total send rate
  // from the parent to the child (independent of who the source is)
  int priority;
  char *dataPtr;
 
 public:
  DataMsg(int dataSizeIn, 
	  int sourceAddrIn, 
	  int seqNumIn, 
	  long timeIn, 
	  long lastHopSendRateIn,
	  int priority,
	  char *dataPtrIn);
  ~DataMsg();
  int GetSourceAddr();
  int GetDataSize();
  int GetSeqNum();
  long GetLastHopSendRate();
  int GetPriority();
  void ResetLastHopSendRate(long lastHopSendRateIn);

  int Size();
  void Print();
  DataMsg *Duplicate();
  char *GetData();
  long GetTime();
  DataMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

class IntentToCancelNbrMsg : public SerializableObject{
  int leaveFlag;
 public:
  int Size();
  void Print();
  IntentToCancelNbrMsg *Duplicate();

  IntentToCancelNbrMsg();
  IntentToCancelNbrMsg(int leaveFlagIn);
  long GetLeaveFlag();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};

class CancelNbrMsg : public SerializableObject{
  int leaveFlag;
 public:
  int Size();
  void Print();
  CancelNbrMsg *Duplicate();
  long GetLeaveFlag();
  CancelNbrMsg();
  CancelNbrMsg(int leaveFlagIn);
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};


class PokeRequestMsg : public SerializableObject{
 private:
  int nonce;

 public:
  int Size();
  void Print();
  PokeRequestMsg *Duplicate();

  PokeRequestMsg(int nonce);
  PokeRequestMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
  int GetNonce();
};

/*********
 Hack: Inheritance really should not be used here. "SessionNonce"
 is used instead of sourceAddr
***********/


class PokeDataMsg : public DataMsg {
 public:
  PokeDataMsg(int dataSizeIn, int sourceAddrIn, int seqNumIn, long timeIn, 
	      long lastHopSendRateIn, char *dataPtrIn);
  PokeDataMsg();
  int GetNonce();
};


class SmallProbeDataMsg : public DataMsg {
 public:
  SmallProbeDataMsg();
  SmallProbeDataMsg(int dataSizeIn, int sourceAddrIn, int seqNumIn, 
		    long timeIn, long lastHopSendRateIn, char *dataPtrIn);
};


class SmallProbeRequestMsg : public PokeRequestMsg {
 public:
  SmallProbeRequestMsg();
  SmallProbeRequestMsg(int nonce);
};



enum RTTMsgDirnType {FORWARD,REVERSE};
const int RTT_ESTIMATION_INIT = -1;

class RTTEstimationMsg : public SerializableObject{
  RTTMsgDirnType dirnFlg; 
  int pathLen; // includes the source
  int path[MyPath::MAX_PATH_LEN]; 
  long forwardTime[MyPath::MAX_PATH_LEN];
  long reverseTime[MyPath::MAX_PATH_LEN];
  
 public:
  int GetSourceAddr();
  RTTMsgDirnType GetDirn();
  int GetPathLen();
  void GetPath(int *pathOutPtr);
  void GetForwardTime(long *forwardTimeOutPtr);
  void GetReverseTime(long *reverseTimeOutPtr);

  // Get the address before me on the forward path.
  // return -1 if I don't occur on the path !
  // Function is not called by the source.
  int GetPredecessor(int addr);
  
  void SetDirn(RTTMsgDirnType dirn);
  
  // 1 if success, 0 if failure
  int AppendForwardPath(int addr,
			long forwardTime);

  // 1 if success, 0 if failure
  // downstreamAddr is the one from whom the reverse packet came
  // This is -1 for the last node.
  int AppendReversePath(int addr,
			int downstreamAddr,
			long time);
  
  int Size();
  void Print();
  RTTEstimationMsg *Duplicate();
  RTTEstimationMsg(RTTMsgDirnType dirnFlg,
		   int pathLen,
		   int *path,
		   long *forwardTime,
		   long *reverseTime);

  RTTEstimationMsg(int addr, int time);
  RTTEstimationMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};


/*******************
class ControlUpdateDemandMsg : public SerializableObject{
 public:
  int Size();
  void Print();
  ControlUpdateDemandMsg *Duplicate();

  ControlUpdateDemandMsg();
  void ReadFromStream(InputSerializableStream *isPtr); 
  void WriteToStream(OutputSerializableStream *osPtr);
};
**********************/

#endif
