#include <iostream.h>
#include <assert.h>

#include "global.h"
#include "neighborTable.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "gossipAgent.h"
#include "controlAgent.h"
#include "TIMERS/controlTimer.h"
#include "TIMERS/estimateDelayTimer.h"
#include "gossipPayLoad.h"
#include "METRICS/routingMetric.h"
#include "path.h"

ControlAgent::ControlAgent(GossipAgent *gossipAgentPtrIn,
			   VirtualRoutingTable *vrtPtrIn,
			   NbrTable *nbrTabPtrIn){
  gossipAgentPtr=gossipAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;
  periodicClock = new ControlTimer(this);
  periodicEstimateDelayClock = new EstimateDelayTimer(this);
  numPeriodicUpdatesSent=0;
  this->SendPeriodicUpdate();
  this->SendPeriodicEstimateDelay();
}

ControlAgent::~ControlAgent(){
  delete periodicClock;
  delete periodicEstimateDelayClock;
}

void ControlAgent::SendPeriodicUpdate(){

  //Prune nbrs who have not sent a control update recently.
  int *madeTmpNbrArray;
  int numMadeTmpNbrs=nbrTabPtr->PruneNbrList(GetCurrTime(),
					     NBR_TIMEOUT,
					     &madeTmpNbrArray);
  for(int i=0; i < numMadeTmpNbrs; i++){
    vrtPtr->UpdateOnNbrTemp(madeTmpNbrArray[i]);
    
    IntentToCancelNbrMsg *msg=new IntentToCancelNbrMsg(0);
    gossipAgentPtr->SendToNetwork(madeTmpNbrArray[i], 
				  GOSSIP_PORT,
				  INTENT_TO_CANCEL_NBR, 
				  msg);
    if(verbosity > 0){
      cout << "\n" << GetCurrTime() << ":" 
	   << GetNameByAddr(gossipAgentPtr->GetMyAddr())
	   << " Made temporary " << GetNameByAddr(madeTmpNbrArray[i]) 
	   << " as timeout on control messages";
    }
  }
  
  delete [] madeTmpNbrArray;
  
  //First get list of nbrs.

  int numNbrs=nbrTabPtr->GetNumNbrs();
  int *nbrAddrList= new int[numNbrs];  
  nbrTabPtr->GetNbrList(nbrAddrList);

  // Check if any nbr is temp, and if yes whether it is time to drop it
  for(int i=0; i < numNbrs; i++){
    if(nbrTabPtr->IsTempNbr(nbrAddrList[i])){
      VRTEntryType *route;
      int dropFlg=1;
      
      
      for(route=vrtPtr->GetNextRouteInit(); route != NULL; 
	  route=vrtPtr->GetNextRoute()){
	
	// Non-obvious fact: If I'm leaving, I mark myself dead. So, its 
	// alright if neighbors still think of me as the next hop to myself.
	// This is taken care off because the above loop skips dead people.

	if(route->IsInChildList(nbrAddrList[i])){
	  dropFlg=0;
	  break;
	}
	if((!gossipAgentPtr->LeaveMode()) && 
	   (route->IsNextHop(nbrAddrList[i]))){
	  dropFlg=0;
	  break;
	}
	
	if((gossipAgentPtr->LeaveMode()) && 
	   (route->IsNextHop(nbrAddrList[i])) && 
	   (route->GetNumChildren() != 0)){
	 /********
	   Even if I use this neighbor as a next hop, when I am in Leave Mode,
	   and I have no children, then this neighbor could be dropped. If I 
	   have children however, then, this temporary link needs to be 
	   retained for the children
	  **********/
	  dropFlg=0;                                                                               
	  break;
	}
      }
      
      if(dropFlg){
	if(verbosity > 0){
	  cout << "\n" << GetCurrTime() << ":" 
	       << GetNameByAddr(gossipAgentPtr->GetMyAddr())
	       << " Dropping " << GetNameByAddr(nbrAddrList[i]) 
	       << " as TF clear";
	}
      }
      if((GetCurrTime() - nbrTabPtr->GetTimeSinceTempNbr(nbrAddrList[i])) > 
	 MAX_TEMP_NBR_TIME){
	if(!dropFlg){
	  dropFlg=1; // Timeout! Temp nbr for too long.
	  if(verbosity > 0){
	    cout << "\n" << GetCurrTime() << ":" 
		 << GetNameByAddr(gossipAgentPtr->GetMyAddr())
		 << " Dropping " << GetNameByAddr(nbrAddrList[i]) 
		 << " as TF Timeout";
	  }
	}
      }

      if(dropFlg){
	// Time to drop the nbr
	if(verbosity > 0){
	  vrtPtr->Print();
	}
	nbrTabPtr->DropNbr(nbrAddrList[i]);

	CancelNbrMsg *msg=new CancelNbrMsg(0);
	gossipAgentPtr->SendToNetwork(nbrAddrList[i], 
				      GOSSIP_PORT,
				      CANCEL_NBR, 
				      msg);
	
	int routesDisconnected=
	  vrtPtr->UpdateOnNbrDeletion(nbrAddrList[i]);
	if(routesDisconnected){
	  SendControlUpdateToAllNbrs();
	}
      }
    }
  }
  
  
    
  // Prepare the control update to be sent
  // Control update is the same even to temporary neighbors !!!

  
  VRTEntryType *selfRoute=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
  selfRoute->IncrementMyControlSeqNum();
  int myNextControlSeqNum=selfRoute->GetLastControlSeqNum();
  
  int numRoutes;
  UpdateRecord *controlUpdateRecordPtr = vrtPtr->CreateUpdateRecord(&numRoutes);
  
  ControlUpdateMsg *msgPtr= new ControlUpdateMsg(myNextControlSeqNum,
						 numRoutes,
						 controlUpdateRecordPtr); 
  // Now send the update to everyone
  for(int i=0; i<numNbrs;i++){
    ControlUpdateMsg *msgCopyPtr=msgPtr->Duplicate();
    gossipAgentPtr->SendToNetwork(nbrAddrList[i],
				  GOSSIP_PORT,
				  CONTROL_UPDATE,
				  msgCopyPtr);
    }  	  
  delete msgPtr;
  delete [] nbrAddrList;

  if(verbosity > 0){
    vrtPtr->Print();
  }
  numPeriodicUpdatesSent++;
  if(numPeriodicUpdatesSent < NUM_UPDATES_STARTUP_PHASE){
    periodicClock->SetTimer(STARTUP_CONTROL_UPDATE_TIMER);
  }
  else{
    periodicClock->SetTimer(PERIODIC_CONTROL_UPDATE_TIMER);
  }
}

void ControlAgent::SendPeriodicEstimateDelay(){
  int numNbrs=nbrTabPtr->GetNumNbrs();
  int *nbrAddrList= new int[numNbrs];
  nbrTabPtr->GetNbrList(nbrAddrList);
  
  VRTEntryType *selfRoute=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
  int myLastControlSeqNum=selfRoute->GetLastControlSeqNum();
  
  for(int i=0; i < numNbrs; i++){
    EstimateDelayRequestMsg *msgPtr 
      = new EstimateDelayRequestMsg(GetCurrTime(),myLastControlSeqNum);
    // msgPtr->Print();
    gossipAgentPtr->SendToNetwork(nbrAddrList[i],
				  GOSSIP_PORT,
				  ESTIMATE_DELAY_REQUEST,
				  msgPtr);

  }
  delete [] nbrAddrList;
  periodicEstimateDelayClock->SetTimer(PERIODIC_ESTIMATE_DELAY_TIMER);
}

void ControlAgent::RecvControlUpdate(int fromAddr,
				     int fromPort,
		 		     ControlUpdateMsg *controlUpdatePtr){
  if(fromPort != GOSSIP_PORT){
    return;
  }

  int tempNbrFlag;
  
  ControlUpdateMsg *duplicateMsg=controlUpdatePtr->Duplicate();
  int nbrStatus=nbrTabPtr->NbrStatusAndRefresh(fromAddr,
					       GetCurrTime(),
					       duplicateMsg);
  if(nbrStatus == 0){
    delete duplicateMsg;
    return;
  }
  
  if(nbrStatus == 1){
    tempNbrFlag=0;
  }
  else {
    tempNbrFlag=1;
  }
  
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL)
    MyError("No routing information for a valid nbr !!");


  int status=vrtPtr->UpdateTable(controlUpdatePtr->GetNumRecords(),
				 controlUpdatePtr->GetRecordArray(),
				 fromAddr,
				 tempNbrFlag);
  if(
     (status == 1) 
     ){
    /* reply to control update with my own control update */
    if(verbosity > 0){
      vrtPtr->Print();
    }

    VRTEntryType *selfRoute=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
    selfRoute->IncrementMyControlSeqNum();
    int myNextControlSeqNum=selfRoute->GetLastControlSeqNum();

    int numRoutes;
    UpdateRecord *controlUpdateRecordPtr = vrtPtr->CreateUpdateRecord(&numRoutes);
    ControlUpdateMsg *msgPtr= new ControlUpdateMsg(myNextControlSeqNum,
						   numRoutes,
						   controlUpdateRecordPtr); 
    
    gossipAgentPtr->SendToNetwork(fromAddr,
				  GOSSIP_PORT,
				  CONTROL_UPDATE,
				  msgPtr);
  }

  else if (status > 1){
    SendControlUpdateToAllNbrs();
  }
}



void ControlAgent::RecvEstimateDelayRequest(int fromAddr,
					    int fromPort,
					    EstimateDelayRequestMsg *msgPtr){

  if(fromPort != GOSSIP_PORT){
    return;
  }

  int nbrStatus=nbrTabPtr->NbrStatusAndRefresh(fromAddr,
					       GetCurrTime(),
					       NULL);
  if(nbrStatus == 0) return;
    
  //  cout << "Receiving Estimate Delay Request ... \n";
  // msgPtr->Print();
  long time=msgPtr->GetTime();
  long nbrLcsn=msgPtr->GetLastControlSeqNum();
  
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No route to a nbr!!");
  if(nbrStatus == 2){
    route->UpdateControlInformation(nbrLcsn,GetCurrTime());
  }

  VRTEntryType *selfRoute=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
  int myLCSN=selfRoute->GetLastControlSeqNum();

  EstimateDelayResponseMsg *responsePtr=new EstimateDelayResponseMsg(time,myLCSN);
  //responsePtr->Print();
  gossipAgentPtr->SendToNetwork(fromAddr,
				GOSSIP_PORT,
				ESTIMATE_DELAY_RESPONSE,
				responsePtr);
}

void ControlAgent::RecvEstimateDelayResponse(int fromAddr,
					     int fromPort,
					     EstimateDelayResponseMsg *msgPtr){

  if(fromPort != GOSSIP_PORT){
    return;
  }

  int nbrStatus=nbrTabPtr->NbrStatusAndRefresh(fromAddr,
					       GetCurrTime(),
					       NULL);
  if(nbrStatus == 0) return;

  // cout << "Receiving ESTIMATE_DELAY_RESPONSE .... \n";
  // msgPtr->Print();
  long time=msgPtr->GetTime();
  long nbrLcsn=msgPtr->GetLastControlSeqNum();

  int roundTripDelay=GetCurrTime()-time;
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No route to a nbr!!");
  route->IntegrateDelayResult(roundTripDelay);
  if(nbrStatus  == 2){
    route->UpdateControlInformation(nbrLcsn,GetCurrTime());
  }
  
  int tempNbrFlg=nbrTabPtr->IsTempNbr(fromAddr);
  MyPath *pathFromNbrToNbr = new MyPath();
  pathFromNbrToNbr->SetZeroPath();

  RoutingMetric *linkMetric=route->GetLinkMetric();
  RoutingMetric *metricFromAddrToAddr=CreateNewRoutingMetric();
  metricFromAddrToAddr->SetBest();
  
  route->UpdateRoutingInformation(fromAddr,
				  tempNbrFlg,
				  linkMetric,
				  metricFromAddrToAddr,
				  GetMyAddr(),
				  pathFromNbrToNbr
				  );
  
  delete linkMetric;
  delete metricFromAddrToAddr;
  delete pathFromNbrToNbr;
}


int ControlAgent::GetMyAddr(){
  return(gossipAgentPtr->GetMyAddr());
}


void ControlAgent::RecvIntentToCancelNbrMsg(int fromAddr,
					    int fromPort,
					    IntentToCancelNbrMsg *msg){
  if (!nbrTabPtr->IsNbr(fromAddr)) return;
  if(fromPort != GOSSIP_PORT) return;

  nbrTabPtr->MarkNbrTemp(fromAddr);
  vrtPtr->UpdateOnNbrTemp(fromAddr);
  if(msg->GetLeaveFlag()){
    vrtPtr->MarkRouteDead(fromAddr);
  }
}

void ControlAgent::RecvCancelNbrMsg(int fromAddr,
				    int fromPort,
				    CancelNbrMsg *msg){
  if (!nbrTabPtr->IsNbr(fromAddr)) return;
  if(fromPort != GOSSIP_PORT) return;

  nbrTabPtr->MarkNbrTemp(fromAddr);
  vrtPtr->UpdateOnNbrTemp(fromAddr);
  if(msg->GetLeaveFlag()){
    vrtPtr->MarkRouteDead(fromAddr);
  }

  int numNbrs=nbrTabPtr->GetNumNbrs();
  int *nbrAddrList= new int[numNbrs];  
  nbrTabPtr->GetNbrList(nbrAddrList);

  if(verbosity > 0){
    vrtPtr->Print();
  }
  for(int i=0; i<numNbrs;i++){
    
    if(nbrAddrList[i] == fromAddr){
      continue;
    }
    
    ControlUpdateMsg *controlUpdate = 
      nbrTabPtr->GetLastControlUpdate(nbrAddrList[i]);
    
    if(controlUpdate == NULL){
      continue;
    }

    if(verbosity > 0){
      controlUpdate->Print();
    }
    
    int status = vrtPtr->UpdateTable(
				     controlUpdate->GetNumRecords(),
				     controlUpdate->GetRecordArray(),
				     nbrAddrList[i],
				     nbrTabPtr->IsTempNbr(nbrAddrList[i])
				     );
    
  }
  
  if(verbosity > 0){
    vrtPtr->Print();
  }
  delete [] nbrAddrList;
  
  nbrTabPtr->DropNbr(fromAddr);
  vrtPtr->UpdateOnNbrDeletion(fromAddr);
  SendControlUpdateToAllNbrs();

}

void ControlAgent::SendControlUpdateToAllNbrs(){
  // Message also sent to temporary neighbors!!

  int numNbrs=nbrTabPtr->GetNumNbrs();
  int *nbrAddrList= new int[numNbrs];  
  nbrTabPtr->GetNbrList(nbrAddrList);


  VRTEntryType *selfRoute=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
  selfRoute->IncrementMyControlSeqNum();
  int myNextControlSeqNum=selfRoute->GetLastControlSeqNum();
  
  int numRoutes;
  UpdateRecord *controlUpdateRecordPtr = vrtPtr->CreateUpdateRecord(&numRoutes);
  
  ControlUpdateMsg *msgPtr= new ControlUpdateMsg(myNextControlSeqNum,
						 numRoutes,
						 controlUpdateRecordPtr); 
  // Now send the update to everyone
  for(int i=0; i<numNbrs;i++){
    ControlUpdateMsg *msgCopyPtr=msgPtr->Duplicate();
    gossipAgentPtr->SendToNetwork(nbrAddrList[i],
				  GOSSIP_PORT,
				  CONTROL_UPDATE,
				  msgCopyPtr);
  }  	  
  delete msgPtr;
  delete [] nbrAddrList;
  
  if(verbosity > 0){
    vrtPtr->Print();
  }
}





