#include <iostream.h>
#include <stdio.h>
#include <assert.h>

#include "connectivityAgent.h"
#include "performanceAgent.h"
#include "partitionAgent.h"
#include "neighborTable.h"
#include "gossipAgent.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "smallProbe.h"

ConnectivityAgent::ConnectivityAgent(GossipAgent *gossipAgentPtrIn, 
				     VirtualRoutingTable *vrtPtrIn, 
				     NbrTable *nbrTabPtrIn){
  gossipAgentPtr=gossipAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;

  partitionMode=0;
  partitionAgentPtr=NULL;


  partitionAgentPtr= new PartitionAgent(
					gossipAgentPtrIn, 
					this, 
					vrtPtrIn, 
					nbrTabPtrIn
					);


  switch(protocolMetric){
  case STATIC:
    performanceAgentPtr=NULL;
    break;
    
  case BW_ONLY:
  case BW_LATENCY:
  case LATENCY_ONLY:
    performanceAgentPtr= new PerformanceAgent(
					      gossipAgentPtrIn, 
					      this, 
					      vrtPtrIn, 
					      nbrTabPtrIn);
    break;
    
  default:
    MyError("Invalid Protocol Metric !!!!");
  }

  /* XXX: revisit, not compatible with simulator */

  switch(protocolMetric){
  case STATIC:
  case BW_ONLY:
  case LATENCY_ONLY:
    smallProbePtr = NULL;
    break;
    
  case BW_LATENCY:
    smallProbePtr = new SmallProbe(gossipAgentPtr);
    break;
    
  default:
    MyError("Invalid Protocol Metric !!!!");
  }
}

ConnectivityAgent::~ConnectivityAgent(){
  if(performanceAgentPtr != NULL){
    delete performanceAgentPtr;
  }

  if(partitionAgentPtr != NULL){
    delete partitionAgentPtr;
  }
  
}

SaturationCodeType ConnectivityAgent::DetermineSaturationLevel(){
  /*****************Pseudocode used currently is as follows.
   Let UpperDegreeBound be the max. number of nbrs I am willing to have
   Let LowerDegreeBound be the min number of nbrs I am willing to have with no complaints.
   Let degree be current number of nbrs.
   Then, if degree < LowerDegreeBound : SaturationCode = LOW
         if LowerDegreeBound < degree < UpperDegreeBound : SaturationCode = MEDIUM
	 if degree > UpperDegreeBound : SaturationCode = HIGH
	 
   More realistically, bandwidth utilities may be used
 *******************************************************/
  
  int degree = nbrTabPtr->GetNumNonTempNbrs();

  if(degree >= gossipAgentPtr->GetUpperDegreeBound()) return(HIGH);
  if(degree <= gossipAgentPtr->GetLowerDegreeBound()) return(LOW);
  return(MEDIUM);  
}

void ConnectivityAgent::RecvAddNbrRequest(int fromAddr,
					  int fromPort,
					  AddNbrRequestMsg *msgPtr
					  ){



  /***********************************************************
     In our design, we leave the decision to the sender of the 
    add-nbr-request message to make an honest decision regarding adding 
    a new nbr. The saturation level of this node was already sent in a probe 
    response and would have been considered. The only issue is if several 
    add-nbr-request messages arrive at the same time. We consider this by 
    doing a check to ensure the node's saturation level is not high before 
    responding.  We might want to build further complexity into this.
  **************************************************************/

  if(!nbrTabPtr->IsNbr(fromAddr)){
    
    AddNbrRequestType requestType=msgPtr->GetType();
    SaturationCodeType saturationLevel=DetermineSaturationLevel();

    int addNbr=0;

    /***************
     Currently Static trees are constructed entirely at Join.
     If this is disconnected due to death of members etc.
     there is no repair process. This may be modified in the
     future to have random connected overlays.
    *****************/
	   
    switch(protocolMetric){

    case STATIC:
      addNbr=1;
      break;
      
    case BW_LATENCY:
    case BW_ONLY:
    case LATENCY_ONLY:
      
      if
	(
	 ((requestType == PARTITION) && (saturationLevel == LOW)) ||
	 ((requestType == PERFORMANCE) && (saturationLevel != HIGH)) ||
	 ((requestType == PARTITION_PRIORITY) && (saturationLevel != HIGH)) ||
	 (requestType == PARTITION_URGENT) 
	 )
	{
	  addNbr=1;
	}
      else{
	if(verbosity > 0){
	cout << "\n Reject ADD_NBR_REQUEST at " 
	     << GetNameByAddr(gossipAgentPtr->GetMyAddr())
	     << " from " << GetNameByAddr(fromAddr)
	     << " <" <<  nbrTabPtr->GetNumNbrs()
	     << " " << (int) saturationLevel << " " << (int) requestType << ">";
	}
      }
      
      break;
    }

    if(addNbr){

      // Add entry as new nbr. No other updates needed.

      if(verbosity > 0){
	vrtPtr->Print();
      }
      
      nbrTabPtr->AddNbr(fromAddr,GetCurrTime());
      //Add entry in VRT if not present
      VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
      if(route == NULL){
	route=vrtPtr->AddRoute(fromAddr);
      }
      vrtPtr->UpdateTable(msgPtr->GetNumRecords(),
			  msgPtr->GetRecordArray(),
			  fromAddr,
			  0);
    }
    else{
      return;
    }
  }
  
  if(nbrTabPtr->IsNbr(fromAddr)){
    // Either already was a neighbor, or became a 
    //neighbor after this request
    //Send an ADD_NBR_RESPONSE
    
    int numRoutes;
    UpdateRecord *addNbrResponseRecord = vrtPtr->CreateUpdateRecord(&numRoutes);
    
    AddNbrResponseMsg *msg=
      new AddNbrResponseMsg(msgPtr->GetLocalTimeOfThisMsg(), 
			    GetCurrTime(),
			    msgPtr->GetLocalSeqNumOfThisMsg(), -1,
			    msgPtr->GetType(),
			    numRoutes, 
			    addNbrResponseRecord);
    
    gossipAgentPtr->SendToNetwork(fromAddr,fromPort,ADD_NBR_RESPONSE,msg);
    
    VRTEntryType *selfRoute=vrtPtr->GetRoute(gossipAgentPtr->GetMyAddr());
    int myLastControlSeqNum=selfRoute->GetLastControlSeqNum();
    
    EstimateDelayRequestMsg *estDelayMsgPtr=
      new EstimateDelayRequestMsg(GetCurrTime(),myLastControlSeqNum);
    
    gossipAgentPtr->SendToNetwork(fromAddr,
				  fromPort,
				  ESTIMATE_DELAY_REQUEST,
				  estDelayMsgPtr);
    
    if(verbosity > 0){
      vrtPtr->Print();
    }
  }
}



void ConnectivityAgent::SendAddNbrRequest(int addrToRequest,
					  int seqNum,
					  AddNbrRequestType type){
  int numRoutes;
  UpdateRecord *addNbrRequestRecord = vrtPtr->CreateUpdateRecord(&numRoutes);

  AddNbrRequestMsg *msg=new AddNbrRequestMsg(-1,GetCurrTime(),
					     -1,seqNum,
					     type,
					     numRoutes, addNbrRequestRecord);
  gossipAgentPtr->SendToNetwork(addrToRequest,GOSSIP_PORT,ADD_NBR_REQUEST,msg);
}

void ConnectivityAgent::SendNbrConfirmRequest(int addr, 
					      AddNbrResponseMsg *msgPtr){
  int numRoutes;
  UpdateRecord *nbrConfirmRequestRecord = vrtPtr->CreateUpdateRecord(&numRoutes);

  if(verbosity > 0){
    vrtPtr->Print();
  }
  NbrConfirmRequestMsg *newMsgPtr=
    new NbrConfirmRequestMsg(msgPtr->GetLocalTimeOfThisMsg(), 
			     GetCurrTime(),
			     msgPtr->GetLocalSeqNumOfThisMsg(), -1,
			     msgPtr->GetType(),
			     numRoutes,nbrConfirmRequestRecord);
  gossipAgentPtr->SendToNetwork(addr,GOSSIP_PORT,NBR_CONFIRM_REQUEST,newMsgPtr);
}

int ConnectivityAgent::RecvNbrConfirmRequest(int fromAddr, 
					     int fromPort, 
					     NbrConfirmRequestMsg *msgPtr){
  if(fromPort != GOSSIP_PORT) return(0);
  if(!nbrTabPtr->IsNbr(fromAddr)) return(0);
  
  //Estimate delay
  int roundTripDelay=GetCurrTime() - msgPtr->GetRemoteTimeOfPrevMsg();
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No Route to neighbor !!");
  route->IntegrateDelayResult(roundTripDelay);

  vrtPtr->UpdateTable(msgPtr->GetNumRecords(),
		      msgPtr->GetRecordArray(),
		      fromAddr,
		      0);
  
  int numRoutes;
  UpdateRecord *nbrConfirmResponseRecord = vrtPtr->CreateUpdateRecord(&numRoutes);

  if (verbosity > 0){
    vrtPtr->Print();
  }
  NbrConfirmResponseMsg *newMsgPtr=new 
    NbrConfirmResponseMsg(msgPtr->GetLocalTimeOfThisMsg(),
			  GetCurrTime(),
			  msgPtr->GetLocalSeqNumOfThisMsg(), 
			  -1,
			  msgPtr->GetType(),
			  numRoutes, nbrConfirmResponseRecord);
  gossipAgentPtr->SendToNetwork(fromAddr,GOSSIP_PORT,NBR_CONFIRM_RESPONSE,newMsgPtr);
  return(1);
}
  
int ConnectivityAgent::RecvNbrConfirmResponse(int fromAddr, 
					       int fromPort, 
					       NbrConfirmResponseMsg *msgPtr){
  if(fromPort != GOSSIP_PORT)
    return(0);
  if(!nbrTabPtr->IsNbr(fromAddr)) return(0);

  //Estimate delay
  int roundTripDelay=GetCurrTime() - msgPtr->GetRemoteTimeOfPrevMsg();
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No Route to a node to which I sent an ADD_NBR_REQUEST!!");
  route->IntegrateDelayResult(roundTripDelay);

  vrtPtr->UpdateTable(msgPtr->GetNumRecords(),
		      msgPtr->GetRecordArray(),
		      fromAddr,
		      0);
  return(1);
}
  

  

void ConnectivityAgent::RecvAddNbrResponse(int fromAddr,
					   int fromPort,
					   AddNbrResponseMsg *msgPtr){


  if( 
     (nbrTabPtr->IsNbr(fromAddr)) ||
     (fromPort != GOSSIP_PORT)
     ){
    if(verbosity > 0){
      cout << " REJECTED ";
    }
    return;
  }
  
  AddNbrResponseType responseType=msgPtr->GetType();
  
  if(responseType == PERFORMANCE){
    assert(performanceAgentPtr != NULL);
    performanceAgentPtr->RecvAddNbrResponse(fromAddr,fromPort,msgPtr);
  }
  else if(
	  (responseType == PARTITION) ||
	  (responseType == PARTITION_PRIORITY) ||
	  (responseType == PARTITION_URGENT)
	  ){
    assert(partitionAgentPtr != NULL);
    partitionAgentPtr->RecvAddNbrResponse(fromAddr,fromPort,msgPtr);
  }
  else
    cout << " REJECTED ";
}


void ConnectivityAgent::ProcessAddNbrResponse(int fromAddr,
					      int fromPort,
					      AddNbrResponseMsg *msgPtr){

  //Estimate delay
  int roundTripDelay=GetCurrTime() - msgPtr->GetRemoteTimeOfPrevMsg();
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL){
    MyError("No Route to a node to which I sent an ADD_NBR_REQUEST!!");
  }
  route->IntegrateDelayResult(roundTripDelay);
  int delayToNewNbr=route->GetPhysicalDelay();
  
  if(verbosity > 0){
    cout << delayToNewNbr; // XXX fixme
  }

  //Add node as new nbr
  nbrTabPtr->AddNbr(fromAddr,GetCurrTime());
  vrtPtr->UpdateTable(msgPtr->GetNumRecords(),
		      msgPtr->GetRecordArray(),
		      fromAddr,
		      0);
  
  if(verbosity > 0){
    vrtPtr->Print();
  }
  
  SendNbrConfirmRequest(fromAddr,msgPtr);
}

void ConnectivityAgent::RecvProbeRequest(int fromAddr, 
					 int fromPort, 
					 ProbeRequestMsg *msg){
  assert(performanceAgentPtr != NULL);
  performanceAgentPtr->RecvProbeRequest(fromAddr,fromPort,msg);
}

void ConnectivityAgent::RecvProbeResponse(int fromAddr,
					  int fromPort,
					  ProbeResponseMsg *msg){ 
  assert(performanceAgentPtr != NULL);
  performanceAgentPtr->RecvProbeResponse(fromAddr,fromPort,msg);
}

void ConnectivityAgent::RecvJoinRequest(int fromAddr, 
					int fromPort, 
					JoinRequestMsg *msg){
  assert(performanceAgentPtr != NULL);
  performanceAgentPtr->RecvJoinRequest(fromAddr,fromPort,msg);
}

void ConnectivityAgent::RecvJoinResponse(int fromAddr,
					 int fromPort,
					 JoinResponseMsg *msg){ 
  if(partitionAgentPtr != NULL){
    partitionAgentPtr->RecvJoinResponse(fromAddr,fromPort,msg);
  }
}

void ConnectivityAgent::RecvProbeMsg(int fromAddr, int fromPort, 
				     GossipPayLoad *msg){
  assert(performanceAgentPtr != NULL);
  performanceAgentPtr->RecvProbeMsg(fromAddr, fromPort, msg);
}

void ConnectivityAgent::RecvSmallProbeMsg(int fromAddr, int fromPort, 
					  GossipPayLoad *msg){
  assert(smallProbePtr != NULL);
  smallProbePtr->RecvMsg(fromAddr, fromPort, msg);
}

void ConnectivityAgent::SetPartitionMode(int partitionModeIn){
  partitionMode=partitionModeIn;
}

int ConnectivityAgent::IsPartitionMode(){
  return(partitionMode);
}

void ConnectivityAgent::StartJoin(){
  partitionAgentPtr->StartJoin();
  if(performanceAgentPtr != NULL){
    performanceAgentPtr->StartAgent();
  }
}

void ConnectivityAgent::ScheduleSmallProbeTest(int *member, int size,
					      PerformanceAgent *ptr) {
  smallProbePtr->ScheduleTest(member, size, ptr);
}

void ConnectivityAgent::ScheduleSmallProbeTest(int *member, int size,
					      PartitionAgent *ptr) {
  smallProbePtr->ScheduleTest(member, size, ptr);
}

