;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; Implementtion of the Agent
;;;; Programmers: Luiza da Silva and Lisa Anthony
;;;; Created: May 10 2001
;;;; Last revised: May 26 2001
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; FILE: agent_2.lisp
;; The Agent plays the game of Acquire interacting with the Engine by sending 
;; commands addressed to it though the lisp command line (output). Each Agent has 
;; an id, which helps the Engine discern who it is "talking" to, and who to send
;; commands to.
;; At first, the agent is initialized and requests to be attached to the engine.
;; After this request is accepted by the Engine, the agent starts playing.
;; When is this agent's turn, the engine will send a command to it, requesting a move
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;; AGENT INITIALIZATION ;;;;;;;;;;;;;;;;
;;; This part of the code initializes the agent.

;; Function SETUP-AGENT
;; Initializes global parameters for the Agent
(defun setup-agent (?id)
  ;; PRE: takes the agent's id
  ;; POST: delcares global parameters for the agent 

  ;; id - a unique identification for the agent, given by the
  ;; controller when the agent subscribes to it
  (defparameter *id*                 ?id)

  ;; heuristic - the name of the Heuristic which this Agent uses
  ;; (defparameter *heuristic*       NIL)

  ;; Agent state components
  ;; items money, tiles and stocks are going to be properties of an agent.
  
  (defparameter *stocks*       NIL)
  (defparameter *money*        0)
  (defparameter *tiles*        NIL)

  ;; Agent state - global variable with all components

  (defparameter *agent-state* NIL)

  ;; World state - global variable with all components

  (defparameter *world-state* NIL)

  ;; World state components
  
  ;; the agent player who is making a move
  (defparameter *current-player*    NIL)

  ;; a list of all tiles that have not yet been chosen
  (defparameter *avail-tiles*       NIL)

  ;; a list of hotel chains which have not yet been defined in the board
  (defparameter *avail-chains*      NIL)

  ;; a list of the amounts of stock shares not yet sold for each hotel chain
  (defparameter *avail-stocks*      NIL)
  
  ;; a list of all chain and respective tiles which have been placed on the board
  (defparameter *played-tiles*      NIL)

  ;; a list of all hotel chains which have been placed on the board
  (defparameter *existing-chains*    NIL)

  ;; Necessary for Game playing

  (defparameter *last-place-tile-action*    NIL)
  (defparameter *rem-tiles*                 NIL)  
  (defparameter *last-buy-stock-action*     NIL)
  (defparameter *rem-stocks*                NIL)

  (defparameter *last-move*                 NIL)
 
  ;; Setting parameter to print
  (setf *print-right-margin* 1000)

  ;; Added by Luiza, 05/18 - Seeding the random function
  (setf *random-state* (make-random-state t))

NIL

)

;; Function MAIN
;; Generates the agent; executable method to allow this 
;; Agent to be instantiated and run on a separate machine from the 
;; "referee"; it connects to the "controller" and waits for the 
;; game to begin
(defun main (?id)
  ;; PRE: takes agent id
  ;; POST: agent setup done, agent attached to engine

  ;; (if (eql (id-check ?id) 0) (wrong-id ?id))

  (setup-agent ?id)

  (show-agent)
  
  ;;added by Luiza(05/25): agent picks heurisitic
  (load "../agent/GenHeur.lisp")
  (setup-gen-heuristic)

  (attach-agent)
)


;; Function RESET-AGENT
;; ***Added by Lisa: need to be able to reset agent attributes when a new 
;; game within same session is begun.
;; Is actually called by engine at the end of each game because the engine 
;; still knows who/where the players
;; are then, not at the beginning of the games.

(defun reset-agent (?id)
  ;; PRE: takes the agent id
  ;; POST: agent reset

  (main ?id)
  ;; same seq of events occurs: init attrib, set ?id, attach to server
)

;;Function SHOW_AGENT
;; display information about agent, facilitator
(defun show-agent( )
  ;; PRE: agent should be set and attached to engine
  ;; POST: no changes

  (pprint *id*)
  (format t "~%*tiles*=~s~%" *tiles*)
  (format t "*money*=~d~%" *money*)
  (dolist (temp *stocks*)
	  (format t "~% ~s" temp)
	  (format t "~%")
	  )
)

;;;;;;;;;;;; GAME PLAYING ;;;;;;;;;;;;;;;;;;;;;;;
;; This part of the code deals with game playing.

;; Function ID-CHECK
;; checks agent id
(defun id-check (?x)
  ;; PRE: takes id
  ;; POST: 1 if it is actually this agent's id, 0 if it is not

  (if (eql *id* ?x) 1 0)
)

;; Function WRONG-ID
;; handles if a different id Was received
(defun wrong-id(?id)
  ;; PRE: takes ids
  ;; POST: let the engine know the id sent to this agent is incorrect.

  (format t "~%INVALID ID: ~s~%" ?id)
)

;; Function ATTACH_AGENT
;; requests attachment to server
(defun attach-agent()
  ;; PRE: 
  ;; POST: agent attached to the engine
  (format t "~%COMMAND:REFEREE:~A:(attach-agent ~A):END~%" *id* *id*)
)

;; Function SET-INITIAL_WORLD_STATE
;; receives initial world state from server
(defun set-initial-world-state(?id ?world-state)
  ;; PRE: takes agent's id and the initial world state
  ;; POST: sets internal global variables accordingly

   (if (eql (id-check ?id) 0) (wrong-id ?id))

  ;;Added by Luiza, 05/25 - updates global variable *world-state*
  (setf *world-state* ?world-state)

  ;;sets current player's info
  (setf *current-player* (nth 0 ?world-state)) 

  ;;sets tiles available
  (setf *avail-tiles* (nth 1 ?world-state))

  ;;sets list of available stocks and the quantities
  (setf *avail-stocks* (nth 2 ?world-state))

  ;;sets list of available chains
  (setf *avail-chains* (nth 3 ?world-state))

  ;;sets list of playes tiles (tiles on the board)
  (setf *played-tiles* (nth 4 ?world-state))

  ;;sets list of existing chains
  (setf *existing-chains* (nth 5 ?world-state))

  ;;sets list of remaining stocks, added by Luiza, 05/18
  (setf *rem-stocks* (mapcar #'car *existing-chains*))
)

;; Function SET_INITIAL_STATE
;; receives initial state for agent if server if connection was accepted, 
;; otherwise resends attach-agent
(defun set-initial-state(?id ?initial-state)
  ;; PRE: takes agent's id and agent's initial state
  ;; POST: sets internal global variables accordingly

  ;;Added by Luiza, 05/25 - updates global variable *agent-state*
  (setf *agent-state* ?initial-state)

  (cond ((eql (id-check ?id) 0)(attach-agent)) 
        (t (setf *money* (nth 0 ?initial-state))
           (setf *tiles* (nth 1 ?initial-state))
	   (setf *stocks* (nth 2 ?initial-state))
	   (setf *rem-tiles* (copy-list *tiles*))))
)

;; Function AGENT-GENERAL-MOVE
;; server asks for agent's move
(defun agent-general-move(?id)
  ;; PRE: takes agent's id
  ;; POST: sends move to engine

  ;; Modified by Luiza (05/25)
  ;; now this funtion will be responsible only for calling
  ;; the calc-move function in the heuristic file
  ;; and receive back from it a move, which is going to be sent 
  ;; to the engine

  (format t "~%Inside agent-general-move~%")

   (if (eql (id-check ?id) 0) (wrong-id ?id))

  (setf actions (calc-move *world-state* *agent-state*))

  ;; This code will remain here - sends the move to the engine

  (format t "~%COMMAND:REFEREE:~A:(accept-move ~A '~A):END~%" *id* *id* actions)
  
  ;; save actions for future reference, for the case the place-tile or buy-stock 
  ;; action ends up being invalid

  (setf *last-place-tile-action* (nth 0 actions))
  (setf *last-buy-stock-action* (nth 1 actions))
  (setf *last-move* actions)

)

;; Function UPDATE_AGENT-STATE
;; updates agent state
(defun update-agent-state(?id ?current-state)
  ;; PRE: takes agent id, and the agent's current state
  ;; POST: sets global variables accordingly

  ;;Added by Luiza, 05/25 - updates global variable *agent-state*
  (setf *agent-state* ?current-state)

  (cond ((eql (id-check ?id) 0))
        (t (setf *money* (nth 0 ?current-state))
           (setf *tiles* (nth 1 ?current-state))
           (setf *stocks* (nth 2 ?current-state))
	   (setf *rem-tiles* (copy-list *tiles*))))
)

;; Function UPDATE-WORLD-STATE
;; updates world state

(defun update-world-state(?id ?current-state)
  ;; PRE: takes agent's id and current state of the world
  ;; POST: updates world-state global variables accordingly

  ;;Added by Luiza, 05/25 - updates global variable *world-state*
  (setf *world-state* ?current-state)

  ;;update current player's info
  (setf *current-player* (nth 0 ?current-state)) 

  ;;updates tiles available
  (setf *avail-tiles* (nth 1 ?current-state))

  ;;updates list of available stocks and the quantities
  (setf *avail-stocks* (nth 2 ?current-state))

  ;;updates list of available chains
  (setf *avail-chains* (nth 3 ?current-state))

  ;;updates list of playes tiles (tiles on the board)
  (setf *played-tiles* (nth 4 ?current-state))

  ;;updates list of existing chains
  (setf *existing-chains* (nth 5 ?current-state))

  ;;update list of remaining stocks, modified by Luiza, 05/18
  (setf *rem-stocks* (mapcar #'car *existing-chains*))

)

;; Function PICK-NEW-CHAIN-NAME
;; when a chain is formed, agent is asked to pick the new chain

(defun pick-new-chain-name(?id ?avail-chains)
  ;; PRE: takes agent's id and list of available chains
  ;; POST: a chain name is picked and sent to the engine

  ;;for now, pick a chain at random
  ;;(setf ?x (random(length ?avail-chains)))
  ;;(setf ?chain (nth ?x ?avail-chains))

  ;;** Added bu Luiza, 05/28
  ;; it does not pick a name at random anymore, it calls a function
  ;; in the heuristic to get the chain name

   (if (eql (id-check ?id) 0) (wrong-id ?id))

  (setq ?chain nil)
  
  (setf ?chain (get-new-chain-name))
  
  ;;send command to server to call (set-new-chain-name(?id ?chain))  
  (format t "~%COMMAND:REFEREE:~A:(set-new-chain-name ~A '~A):END~%" *id* *id* ?chain)
)

;; Function PICK-DOMINANT-CHAIN-IN-MERGER
;; in a merger with >2 chains, agent must pick one among dominant chain in merger

(defun pick-dominant-chain-in-merger(?id ?chain-choices)
  ;; PRE: takes agent's id and a list of chain choices
  ;; POST: a chain is picked to become the dominant chain in the merger

  ;;for now, pick a chain at random
  ;;(setf ?x (random(length ?chain-choices)))
  ;;(setf ?chain (nth ?x ?chain-choices))

   (if (eql (id-check ?id) 0) (wrong-id ?id))

  ;;** Added by Luiza, 05/28:
  ;; does not pick a chain at random anymore
  ;; calls function in heuristic file that will determine what chain to
  ;; pick

  (setq ?chain nil)

  (setf ?chain (calc-pick-dominant-chain-in-merger ?chain-choices *agent-state*))

  ;;send command to server to call (set-dominant-chain-in-merger(?id ?chain))

  (format t "~%COMMAND:REFEREE:~A:(set-dominant-chain-in-merger ~A '~A):END~%" *id* *id* ?chain)
)

;; Function CHOOSE-STOCK-ACTION
;; agent chooses stock action

(defun choose-stock-action(?id ?dominant-chain ?merging-chains)
  ;; PRE: takes id, the name of the dominant chain, and a list of the merging chains
  ;; POST: an action regarding the stocks the agent has from a chain that has been
  ;; absorbed in a merger is taken and sent to the engine

  (format t "~%Enter choose-stock-action~%")
  (setq actions nil)

  ;; ** Added by Luiza, 05/28/2001
  ;; The stock actions are not picked randomly anymore;
  ;; now function pick-stock-action is called from the heuristic file


;;  (dolist (chain ?merging-chains)
;;	(format t "~A~%" chain)
;;        (setf pair (assoc chain *stocks*))
;;	(format t "~A~%" pair)
;;        (setf a (nth 1 pair))
;;	(format t "~d~%" a)
;;	(if (eql a nil) (setf a 0))
;;        (setf b (random (+ a 1)))
;;	(format t "~d~%" b)
;;        (setf action1 (list 'TRADE-STOCK b chain ?dominant-chain))
;;        (setf a (- a b))
;;	(format t "~d~%" a)
;;	(setf b (random (+ a 1)))
;;	(format t "~d~%" b)
;;        (setf action2 (list 'HOLD-STOCK b chain))
;;        (setf a (- a b))
;;	(format t "~d~%" a)
;;        (setf action3 (list 'SELL-STOCK a chain))
;;	(setf actions (append actions (list (list action1 action2 action3)))))

  (setf actions (pick-stock-action ?dominant-chain ?merging-chains *world-state* *agent-state*))

  ;;send command to engine
  (format t "~%COMMAND:REFEREE:~A:(agent-stock-action ~A '~A '~A '~A):END~%" *id* *id* ?dominant-chain ?merging-chains actions)

)

;; Function DECLARE_GAME_OVER
;; server offers agent to declare the game over, if it is a valid action in the 
;; agent's move

(defun declare-game-over (?id)
  ;; PRE: takes agent's id
  ;; POST: agent declares game over

  ;; for now, decides if wants to declare game over at random too
  (setf ?answer (random 2))
  
  (if (eql ?answer 0)(setf ?answer t)(setf ?answer nil))

  ;; then it sends answer to engine
  (format t "~%COMMAND:REFEREE:~A:(declare-game-over ~A ~A):END~%" *id* *id* ?answer)
  
)

;; Function INVALID-PLACE-TILE
;; server checks PLACE-TILE action from agent's move and realizes it is invalid.
;; server then asks for another tile-placing action

(defun invalid-place-tile(?id)
  ;; PRE: takes agent's id
  ;; POST: returns a new move with a modified place-tile action

  ;;**Modified my Luiza, 05/25
  ;; Now selection of another tile is done inside the Heuristic

  ;; ** Added by Luiza:
  ;; Took parameters away since now the ranking of tiles happen at the heuristic file
  
  ;; here a new move has to be calculated, since the tile choice could have been tied up to 
  ;; the stock action chosen

  ;; so calc-new-buy-stock is called too

  (setf tile-action (calc-new-place-tile))

  (setf stock-action (calc-new-buy-stock))
 
  (setf actions (list tile-action stock-action))

  ;; saves place-tile action and entire new move
  (setf *last-place-tile-action* tile-action)
  (setf *last-buy-stock-action* stock-action)
  
  (setf *last-move* actions)

  ;;Added by Luiza, 05/18: necessary to contain the remaining tiles
  ;;(setf *rem-tiles* tiles)

  ;;calls in server accept-new-place-tile(?id ?move)
  (format t "~%COMMAND:REFEREE:~A:(accept-new-place-tile ~A '~A):END~%" *id* *id* actions)

)

;; Function INVALID-BUY-STOCK
;; server checks BUY-STOCK action from agent's move and realizes it is invalid.
;; server then asks for another stock buying action

(defun invalid-buy-stock(?id ?qty ?chain)
  ;; PRE: takes agent's id, the amount of stocks the agent tried to buy, and
  ;; the chain
  ;; POST: returns a new move with a modified buy-stock action to the engine
 
  ;;***Code modified by Luiza, 05/25
  ;; Now this function calls calc-new-buy-stock in the heuristic

  (setf action (calc-new-buy-stock))

  (setf actions (list *last-place-tile-action* action))

  ;; saves place-tile action and entire new move
  (setf *last-buy-stock-action* action)
  (setf *last-move* actions)

  ;;calls in server accept-new-buy-stock(?id ?move)

  (format t "~%COMMAND:REFEREE:~A:(accept-new-buy-stock ~A '~A):END~%" *id* *id* actions)

)

;; Function INVALID-MOVE-FORMAT
;; server checks agent's move and realizes there is something wrong with the format
;; server then calls this function in agent so agent can check move format and 
;; resend move

(defun invalid-move-format(?id)

  ;;calls accept-move(?id ?revised-move) 

  ;;for now calls accept move again
  (format t "~%COMMAND:REFEREE:~A:(accept-move ~A '~A):END~%" *id* *id* *last-move*)  
) 

;; Function ENGINE_NOT_READY
;; resends attachment request if engine is not ready

(defun engine-not-ready (?id)
  ;; PRE: takes agent's id
  ;; POST: resends attachment request to engine
  (format t "~%COMMAND:REFEREE:~A:(attach-agent ~A):END~%" *id* *id*)
)








