;;; Necessary functions for Acquire
;;; Lisa Anthony
;;; created: Apr. 24, 2000

;;; Adapted to be part of Software Engineering Workshop Team 2 Project
;;; last revised: 05-07-2001

;; FUNCTIONS

;; GAME INITIALIZATION
;; Must be initialized via the Java server connector
(defun init-engine ()
  ;; miscellaneous maintenance thingamabob to force printing to occur
  ;; all on one line with no linebreaks in the middle of a state
  (setf *print-right-margin* 1000)
  ;; signifies the game has been set-up and initialized via the IO module
  (defparameter *go* NIL)
  T)

;; Takes initialization of game world.
(defun io-start-game (?numagents ?agents)
  ;; set up agents with their appropriate heuristics
  ;;(setf *numplayers* ?numagents)
  ;;  (start-game)
  ;; set up global constants/data structures.
  ;;(defun start-game ()
  (setup-game ?numagents)
  (load "boardgen.lisp")
  (gen-board)
  ;; set *go* to T, signaling engine is ready to accept player connections
  (setf *go* T)
  ;; nothing further happens until correct number of players have connected
  )

;; Initializes ?x number of players.
(defun setup-game (?x)
  ;; game play constants
  (defparameter *hand-size*            6)
  (defparameter *largest-merge-length* 10)
  (defparameter *max-chain-length*     41)
  (defparameter *max-stock-avail*      20)
  (defparameter *max-stock-to-buy*     3)
  (defparameter *init-money*           6000)
  (defparameter *init-stock-qty*       0)
  
  ;; global data structures
  ;; hotel chains and stocks in the stock market for each chain
  (defparameter *all-chains* '(TOWER LUXOR AMERICAN WORLDWIDE FESTIVAL IMPERIAL CONTINENTAL))
  (defparameter *avail-chains* '(TOWER LUXOR AMERICAN WORLDWIDE FESTIVAL IMPERIAL CONTINENTAL))
  (setup-chains *avail-chains*)
  (defparameter *played-chains* NIL)
  (defparameter *avail-stocks* NIL)
  (setup-stocks *avail-chains*)

  ;; the number of players
  (defparameter *numplayers*     ?x)
  ;; the list of players in this game
  (defparameter *players*        NIL)
  ;; the player whose turn it is currently to make a move
  (defparameter *current_player* NIL)
  ;; the player who is actively making a choice right now (as in a merger loop)
  (defparameter *active-player*  NIL)
  
  ;; tiles in the pot and on the board
  ;; *avail-tiles* is defined in boardgen.lisp
  (defparameter *played-tiles* NIL)
  (defparameter *current-tile* NIL)

  ;; move in the process of being made by *current-player*
  (defparameter *current-move* NIL)

  ;; location of all history information in game
  (defparameter *history* NIL)
  T)

(defun setup-chains (x)
  (cond ((eq x NIL) NIL)
	(t          (setf (get (car x) 'tiles) NIL)
		    (setup-chains (cdr x))))
  )

;; Sets up *avail-stocks* based on the chains listed in *avail-chains*.
(defun setup-stocks (x)
  (dolist (temp x)
	  (eval (read-from-string (format nil "(setf *avail-stocks* (append *avail-stocks* (cons '~s nil)))~%" temp)))
	  (eval (read-from-string (format nil "(setf (get '~s 'qty) ~s)~%" temp *max-stock-avail*))))
  )

;; Randomly deals *hand-size* tiles from the *avail-tiles* list.
(defun deal-tiles ()
  (let ((ret-value) (*random-state* (make-random-state t)))
    (dotimes (temp *hand-size* temp)
	     (setf r (random (length *avail-tiles*)))
	     (setf ret-value (append (cons (nth r *avail-tiles*) nil) ret-value))
	     (setf *avail-tiles* (remove (nth r *avail-tiles*) *avail-tiles*))) ;; dotimes
    ret-value)
  )

(defun attach-agent (?x)
  ;; if io hasn't initialized game, send back message agent::engine-not-ready (?id)
  (cond ((not *go*) (format t "~%COMMAND:~d:REFEREE:(engine-not-ready ~d):END~%" ?x ?x))
	(t (progn
	     ;; add a new player to *players*, with id ?x
	     (eval (read-from-string (format nil "(setf *players* (append *players* (cons 'player~s nil)))~%" ?x)))
	     (eval (read-from-string (format nil "(setf (get 'player~s 'id) ~d)~%" ?x ?x)))
	     
	     (if (eq (length *players*) *numplayers*) (begin-play))
	     ;; otherwise wait some more
	     )))
  T)


;; GAME PLAY
;; Determines play order and loops through turn sequence until end of game.
(defun begin-play ()
  (det-turn-order)
  (announce-init-world-state-to-all)
  (init-players)

  (get-move-from-player *current-player*)
  )

;; Randomly draws a tile for each player to determine who goes first.
;; After the first player, play proceeds in order of connection to Engine.
(defun det-turn-order ()
  (setq temp-tiles NIL)
  (let ((*random-state* (make-random-state t)))
    (dotimes (temp (length *players*))
	     (setf tile (nth (random (length *avail-tiles*)) *avail-tiles*))
	     (setf *avail-tiles* (remove tile *avail-tiles*))
	     (setf temp-tiles (append temp-tiles (cons tile NIL)))
	     ) ;;dotimes
    )
  
  ;; find lowest-valued tile - make this its own function eventually
  (let ((minimum 108) (first-player NIL))
    (dolist (temp temp-tiles)
	  (cond ((= minimum (get temp 'value))
		 (format t "~%==  in det-turn-order() setting 'minimum' to ~s (~s)~%" minimum temp)
		 (setf first-player (append (cons temp nil) first-player)))
		((> minimum (get temp 'value))
		 (setf minimum (get temp 'value))
		 (format t "~%>  in det-turn-order() setting 'minimum' to ~s (~s)~%" minimum temp)
		 (setf first-player temp)
		 (format t "~%in det-turn-order() tile=~s 1 player=~s~%" temp first-player))))
    (setf *current-player* (nth (position first-player temp-tiles) *players*))
    (format t "~%in det-turn-order() current-player=~s~%" *current-player*)
    )

  ;; place drawn tiles on board, no chains form
  (dolist (temp temp-tiles)
	  (setf *played-tiles* (cons temp *played-tiles*))
	  (setf (get temp 'played) T))
  )

;; Sets up initial state of each player and sends info to agent.
(defun init-players ()
  (dolist (temp *players*)
	  (setq tiles (deal-tiles))
	  (format t "~%returned from deal-tiles ok~%")
	  (eval (read-from-string (format nil "(setf (get '~s 'money) ~s)~%" temp *init-money*)))
	  (eval (read-from-string (format nil "(setf (get '~s 'tiles) '~s)~%" temp tiles)))
	  (dolist (temp2 *all-chains*)
		  (eval (read-from-string (format nil "(setf (get '~s '~s) ~d)~%" temp temp2 *init-stock-qty*))))

	  ;; construct initial player state to send to ?player
	  (setq init-stock-qty '((TOWER 0) (LUXOR 0) (AMERICAN 0) (WORLDWIDE 0) (FESTIVAL 0) (IMPERIAL 0) (CONTINENTAL 0)))
	  (setq player-state (list *init-money* tiles init-stock-qty))
	  ;; send message agent::set-initial-state (?player ?player-state) (needs no response)
	  (format t "~%COMMAND:~d:REFEREE:(set-initial-state ~d '~s):END~%" (get temp 'id) (get temp 'id) player-state)
	  )
  )

;; Sends last completed move to the IO module.
(defun output-move (?player ?move)
  ;; send message io::write-move (?id ?move)
  (format t "~%COMMAND:IO:REFEREE:(write-move ~d '~s):END~%" (get ?player 'id) ?move)
  )

;; Sends end-of-game signal to IO module.
(defun send-endgame (?standings)
  ;; send message io::end-of-game (?standings)
  (format t "~%COMMAND:IO:REFEREE:(end-of-game '~s):END~%" ?standings)
  )

;; Evaluates state of world & players at end of game and determines winner(s). -- no longer needed since sort can do all this for me.
;;(defun evaluate-winner ()
;;  (format t "~%made it to evaluate winner~%")
;;  (sell-out-stocks)
;;  (let ((maximum 0) (winners NIL))
;;    (dolist (temp *players*)
;;	    (cond ((= maximum (get temp 'money))
;;		   (format t "  = setting 'maximum' to ~s (~s)~%" maximum temp)
;;		   (setf winners (append winners (cons temp nil))))
;;		  ((< maximum (get temp 'money))
;;		   (setf maximum (get temp 'money))
;;		   (format t "  < setting 'maximum' to ~s (~s)~%" maximum temp)
;;		   (setf winners (append winners (cons temp nil))))))
;;   (if (eq (length winners) 1) (format t "~%... AND THE WINNER IS: ~s!!~%" winners)
;;      (progn
;;	(format t "~%... A TIE BETWEEN ")
;;	(dolist (temp winners)
;;		(format t "~s " temp))
;;	(format t "~%")))
;;   winners)
;;  )

;; Checks validity of (PLACE-TILE ?x) within ?move from ?player;
;; if valid, continues by applying ?move to world.
;; if invalid, asks player for a new (PLACE-TILE ?x) segment.
(defun make-move (?player ?move)
  ;; once received a move from the player, apply to world state
  (format t "~%make-move(~s ~s)~%" ?player ?move)
  (setf *current-move* ?move)
  (cond ((validate-move (nth 1 (car ?move)))
	 (start-turn ?player ?move))
	(t
	 ;; send message agent::invalid-place-tile ()
	 (format t "~%COMMAND:~d:REFEREE:(invalid-place-tile ~d):END~%" (get ?player 'id) (get ?player 'id))
	 ))
  )

;; Applies (PLACE-TILE ?x) action; tries to apply (BUY-STOCK ?qty ?chain) actions.
(defun start-turn (?player ?move)
  ;; place tile
  (place-tile ?player (nth 1 (car ?move)))
;;  (buy-stocks ?player ?move)
  )

;; Tile ?x is placed on the board as long as it is a valid move.
(defun place-tile (?player ?x)
  (format t "~%Place-Tile~%")
  ;; if tile is NIL, as when player's tiles are all unplayable, just go to buy-stock portion
  (cond ((eq ?x NIL) (buy-stocks ?player *current-move*))
	(t 
	 (setf (get ?player 'tiles) (remove ?x (get ?player 'tiles)))
	 (setf *played-tiles* (cons ?x *played-tiles*))
	 (setf (get ?x 'played) T)
	 (setf *current-tile* ?x)
	 (update-chains ?x ?player)
	 (format t "~%leaving Place-tile~%")))
  )

;; Buy stocks - will have to loop for length of ?move but for now assume only 1 BUY-STOCK
(defun buy-stocks (?player ?move)
;;  (setf *current-move* ?move)
  (buy-stock ?player (nth 1 (nth 1 ?move)) (nth 2 (nth 1 ?move)))
  )

;; Allows ?player to buy stocks in any hotel chain currently on the board.
(defun buy-stock (?player ?qty ?chain)
  ;; resource constraints - money, availability of stocks (*avail-stocks*)
  ;;  (setf stock (get-stock-choice-from-player ?player))
  (format t "~%  Buy-Stock(~s ~d ~s)~%" ?player ?qty ?chain)
  (cond ((eq ?chain NIL) (end-turn ?player))
	(t 
	;; (setf qty (get-stock-qty-from-player ?player))
	 (format t "~%in buy-stock() ~s has ~d, trying to buy ~d~%" ?chain (get ?chain 'qty) ?qty)
	 (format t "~%in buy-stock() ~s has $~d ~%" ?player (get ?player 'money))
	 (if (or (< (get ?chain 'qty) ?qty)
		  (<= (get ?player 'money) (* (get-stock-price ?chain) ?qty)))
	     ;; send message agent::invalid-buy-stock (?player ?qty ?chain)
	     (format t "~%COMMAND:~d:REFEREE:(invalid-buy-stock ~d ~d '~s):END~%" (get ?player 'id) (get ?player 'id) ?qty ?chain)
;;	   (format t "~%NOT ENOUGH STOCK/MONEY~%")
	   (add-stocks-to-player-state ?player ?qty ?chain))))
  )

;; Performs the actual stock transaction, where ?player buys ?qty in ?chain.
(defun add-stocks-to-player-state (?player ?qty ?chain)
  (format t "~%in add-stocks-to-player-state() ~s BUYING ~d stock in chain ~s at $~d~%" ?player ?qty ?chain (get-stock-price ?chain))
  (setf (get ?player 'money) (- (get ?player 'money) (* (get-stock-price ?chain) ?qty)))
  (format t "~%in add-stocks-to-player-state() ~s now has $~d left~%" ?player (get ?player 'money))
  (setf (get ?player ?chain)  (+ (get ?player ?chain) ?qty))
  (setf (get ?chain 'qty) (- (get ?chain 'qty) ?qty))
  (end-turn ?player)
  )

;; Ends ?player's turn by asking if he wants to end the game, otherwise draw ?player a new tile.
(defun end-turn (?player)
  ;; if the player has played his last tile, game must end
  (if (last-tile-played *current-player*) (end-the-game) NIL)
  ;; declare end of game - here ask agent if he wants to end the game if it is legal
  (if (eq (legal-end-of-game) T) (player-declare-game-over ?player) (continue-game ?player))
  )

;; Continue playing (used when end-game is not valid or when ?player chose not to declare an end of game).
(defun continue-game (?player)
  ;; send message agent::update-agent-state (?id ?player-state) (needs no response)
  (format t "~%COMMAND:~d:REFEREE:(update-agent-state ~d '~s):END~%" (get ?player 'id) (get ?player 'id) (construct-player-state ?player))
    
  (update-history ?player *current-move*)
  (output-move ?player *current-move*)
  (announce-new-world-state-to-all)
  ;; advance to next player
  (setf *current-player* (get-next-player *current-player*))
  (get-move-from-player *current-player*)
  )

;; Takes a ?player and creates a list which can be used in message passing.
(defun construct-player-state (?player)
  ;; construct ?player-state
  (draw-tile ?player)
  (setq player-stocks NIL)
  (dolist (temp *all-chains*)
	  (format t "~%in construct-player-state() chain=~s ~s has ~s~%" temp ?player (get ?player temp))
	  (setq player-stocks (append player-stocks (cons (list temp (get ?player temp)) nil))))
  (list (get ?player 'money) (get ?player 'tiles) player-stocks)
  )

;; Update the chains affected by placing tile ?x on the board.
(defun update-chains (?x ?player)
  (format t "~%Update-Chains~%")
  ;; assumes caller already checked (valid-move ?x)
  (cond
   ;; if tile ?x is next to ONE chain
   ((= (list-length (adj-chains ?x)) 1) (add-adj-tiles-to-chain ?x (car (adj-chains ?x))) (buy-stocks ?player *current-move*))
   ;; if tile ?x is next to MORE THAN ONE chain, merge chains
   ((> (list-length (adj-chains ?x)) 1) (format t "  TRYING TO ADD MERGER~%") (form-merger ?x ?player))
   ;; if tile ?x is next to no chain, but is next to a tile, create a new chain
   ((> (list-length (adj-tiles ?x)) 0)  (format t "  CREATING NEW CHAIN~%") (get-new-chain-from-player ?player))
   ;; else return NIL
   (t                                  (format t "  PLACING SINGULAR TILE~%") (buy-stocks ?player *current-move*)))
  )

;; Randomly deals one tile to ?player from the *avail-tiles* list.
(defun draw-tile (?player)
  (format t "~%Drawing a Tile...~%")
  (let ((tile) (*random-state* (make-random-state t)))
    (cond ((= 0 (length *avail-tiles*)) NIL)
	  (t
	   (cond ((= (length *avail-tiles*) 1)
		  (setf tile (car *avail-tiles*)))
		 (t
		  (setf tile (nth (random (length *avail-tiles*)) *avail-tiles*))))
	   (setf *avail-tiles* (remove tile *avail-tiles*))
	   ;; add to player's list
	   (setf (get ?player 'tiles) (append (cons tile nil) (get ?player 'tiles)))
	   T)))
  )

;; Adds ?move made by ?player to *history*.
(defun update-history (?player ?move)
  (setf *history* (append *history* (cons (list ?player ?move) nil)))
  )

;; Recursively adds all adjacent tiles to tile ?x to chain ?y.
(defun add-adj-tiles-to-chain (?x ?y)
  (format t "~%ADDING ADJACENT TILES TO ~s to CHAIN ~s~%" ?x ?y)
  (cond ((eq ?x NIL) NIL)
	(t           (add-tile-to-chain ?x ?y)
		     (format t "~%all adjacent tiles to ~s: " ?x)
		     (pprint (adj-tiles ?x))
		     (dolist (temp (adj-tiles ?x))
			     (if (and (get temp 'played) (eq (member temp (get ?y 'tiles)) NIL) (eq (get ?y 'chain) NIL))
				 (add-adj-tiles-to-chain temp ?y)))))
  )
	
;; Adds tile ?x to chain ?y.
(defun add-tile-to-chain (?x ?y)
  (format t "  ADDING TILE ~s TO CHAIN ~s~%" ?x ?y)
  (if (get ?x 'played)
      (progn
	(setf (get ?x 'chain) ?y)
	(setf (get ?y 'tiles) (append (cons ?x nil) (get ?y 'tiles)))))
  )

;; Responsible for handling marging on the board when tile ?x is played by ?player.
(defun form-merger (?x ?player)
  ;; when chains are same size, player picks
  (format t "~%in form-merger(): how many adj chains to ~s: " ?x)
  (pprint (list-length (remove-duplicates (mapcar #'chain-size (adj-chains ?x)))))
  (cond ((not (atom (get-biggest-chain ?x)))
	 (get-chain-choice-from-player ?x ?player))
	(t
	 (format t "~%biggest chain next to ~s = " ?x)
	 (pprint (get-biggest-chain ?x))
	 (consume-chains ?x (get-biggest-chain ?x))
	 (add-adj-tiles-to-chain ?x (get-biggest-chain ?x))))
  )


;; Chain ?y takes over all other chains adjacent to tile ?x.
(defun consume-chains (?x ?y)
  (format t "~%Entering consume-chains(~s ~s)~%" ?x ?y)
  (setq merging-chains NIL)
  (mapcar #'(lambda (z)
	      (distrib-bonuses z)
	      (format t "~%  Eating up chain ~s~%" z)
	      (mapcar #'(lambda (v)
			  (add-tile-to-chain v ?y))
		      (get z 'tiles))
	      (setf (get z 'tiles) NIL)
	      (setf *avail-chains* (append (cons z nil) *avail-chains*))
	      (setf *played-chains* (remove z *played-chains*))
	      (setf merging-chains (append (cons z nil) merging-chains)))
	  (other-adj-chains ?x ?y))
  (dispose-of-stocks *current-player* ?y merging-chains)
  )

;; Allow ?player to dispose of each chain within ?merging-chains.
(defun dispose-of-stocks (?player ?dominant-chain ?merging-chains)
  (setf *active-player* ?player)
  ;; send message agent::choose-stock-action (?player ?chains)
  (format t "~%COMMAND:~d:REFEREE:(choose-stock-action ~d '~s '~s):END~%" (get ?player 'id) (get ?player 'id) ?dominant-chain ?merging-chains)
  )

;; Determine if placing tile ?x would be a valid move.  Returns NIL if
;; not, T if yes.
(defun validate-move (?x)
  (format t "~%validating move ~s~%" ?x)
  (cond ((eq ?x NIL)                                                          (format t "~%VALID MOVE~%") T)
	((and (member ?x (get *current-player* 'tiles))
	      (eq (get ?x 'played) NIL)
	      (< (list-length (prune (mapcar #'chain-too-big (adj-chains ?x)))) 2)
	      (not (and (= (list-length *avail-chains*) 0)
			(>= (list-length (adj-tiles ?x)) 1)
			(= (list-length (adj-chains ?x)) 0))))
	 (format t "~%VALID MOVE~%") T)
	(t                                                                    (format t "~%INVALID MOVE~%")  NIL))
  )

;; Gives ?x stock in chain ?y to ?player.
(defun give-stock-to-player (?x ?y ?player)
  (format t "~%  giving ~d stock in chain ~s to ~s~%" ?x ?y ?player)
  (if (< (get ?y 'qty) ?x)
      (format t "~%NOT ENOUGH STOCK IN CHAIN~%")
    (progn
      (setf (get ?player ?y) (+ ?x (get ?player ?y)))
      (setf (get ?y 'qty) (- (get ?y 'qty) ?x))))
  )

;; Distributes bonuses to the majority and secondary stockholder's bonus when chain ?x is bought out.
(defun distrib-bonuses (?x)
  (format t "~%Entering distrib-bonuses(~s)~%" ?x)
  (mapcar #'(lambda (z)
	      (format t "~%giving $~d to maj stockholder ~s~%" (get-maj-bonus ?x) (get-maj-stockholder ?x))
	      (setq bonus (/ (get-maj-bonus ?x) (list-length (get-maj-stockholder ?x))))
	      (setf (get z 'money) bonus)
	      ;; send message agent::accept-shareholder-bonus (?id ?money) (loop if necessary) (no response needed)
	      ;; taken out for now - agent didn't have it
	      ;;	      (format t "~%COMMAND:~d:REFEREE:(accept-shareholder-bonus ~d ~d):END~%" (get z 'id) (get z 'id) bonus)
	      )
	  (get-maj-stockholder ?x))
  (mapcar #'(lambda (z)
	      (format t "~%giving $~d to sec stockholder ~s~%" (/ (get-maj-bonus ?x) 2) (get-sec-stockholder ?x))
	      (setq bonus2 (/ (/ (get-maj-bonus ?x) 2) (list-length (get-sec-stockholder ?x))))
	      (setf (get z 'money) bonus2)
	      ;; send message agent::accept-shareholder-bonus (?id ?money) (loop if necessary) (no response needed)
	      ;; taken out for now - agent didn't have it
	      ;;	      (format t "~%COMMAND:~d:REFEREE:(accept-shareholder-bonus ~d ~d):END~%" (get z 'id) (get z 'id) bonus2)
	      )
	  (get-sec-stockholder ?x))
  )

;; At end of game, assign bonuses to majority and secondary stockholders of remaining chains on board.
(defun sell-out-stocks ()
  (format t "~%made it to sell-out-stocks~%")
  (mapcar #'distrib-bonuses *played-chains*)
  ;; now sell all players' stocks
  (dolist (temp *players*)
	  (dolist (temp2 *all-chains*)
		  (format t "~%sell-stock ~s ~d ~s~%" temp (get temp temp2) temp2)
		  (sell-stock temp (get temp temp2) temp2)))
  )


;; PLAYER COMMUNICATION
;; Asks ?player for a move.
(defun get-move-from-player (?player)
  ;; send message to player asking for move
  ;; send message agent::agent-general-move (?id) 
  (format t "~%COMMAND:~d:REFEREE:(agent-general-move ~d):END~%" (get ?player 'id) (get ?player 'id))
  )

;; Accepts new ?move from ?player.  Validates its format.
(defun accept-move (?id ?move)
  ;; make sure ?id belongs to *current-player*
  (if (not (eq (get *current-player* 'id) ?id)) (wrong-id-in-response (get *current-player* 'id) ?id)
    (progn
      ;; check form of ?move
      (if (or (not (eq (car (car ?move))   'PLACE-TILE))
	      (not (eq (car (nth 1 ?move)) 'BUY-STOCK))
	      (not (eq (car (nth 2 ?move)) 'DRAW-TILE))) (invalid-move-format *current-player*)
	(progn (setf *current-move* ?move) (make-move *current-player* ?move)))))
  )

;; Asks ?player for a new move if the one previously sent is in invalid format.
(defun invalid-move-format (?player)
  ;; send message agent::invalid-move-format (?id)
  (format t "~%COMMAND:~d:REFEREE:(invalid-move-format ~d):END~%" (get ?player 'id) (get ?player 'id))
  )

;; Takes new (PLACE-TILE ?x) action from ?player.
(defun accept-new-place-tile (?id ?move)
  ;; make sure ?id belongs to *current-player*
  (if (not (eq (get *current-player* 'id) ?id)) (wrong-id-in-response (get *current-player* 'id) ?id)
    (progn (setf *current-move* ?move) (make-move *current-player* ?move)))
  )

;; Takes new (BUY-STOCK ?qty ?chain) action from ?player.
(defun accept-new-buy-stock (?id ?move)
  ;; make sure ?id belongs to *current-player*
  (if (not (eq (get *current-player* 'id) ?id)) (wrong-id-in-response (get *current-player* 'id) ?id)
    (progn (setf *current-move* ?move) (buy-stocks *current-player* ?move)))
  )

;; Asks ?player for what to name a newly created chain, when *current-tile* is placed next to a singular tile.
(defun get-new-chain-from-player (?player)
  ;; send message agent::pick-new-chain-name (?id *avail-chains*)
  (format t "~%COMMAND:~d:REFEREE:(pick-new-chain-name ~d '~s):END~%" (get ?player 'id) (get ?player 'id) *avail-chains*)
  )

;; Takes ?chain choice (when a new chain has been formed) from ?player.
(defun set-new-chain-name (?id ?chain)
  ;; make sure ?id belongs to *current-player*
  (if (not (eq (get *current-player* 'id) ?id)) (wrong-id-in-response (get *current-player* 'id) ?id)
    (progn (format t "  CHAIN PICKED: ~s~%" ?chain)
	   (setf *avail-chains* (remove ?chain *avail-chains*))
	   (setf *played-chains* (append (cons ?chain nil) *played-chains*))
	   (add-adj-tiles-to-chain *current-tile* ?chain)
	   ;; player gets one free stock when creating a new chain
	   (give-stock-to-player 1 ?chain *current-player*)
	   (buy-stocks *current-player* *current-move*)))
  )

;; Returns the chain chosen (of the chains adjacent to ?x)
;; by ?player when there is a tie in multiple mergers.
(defun get-chain-choice-from-player (?x ?player)
  (setf ?chain-choices (get-biggest-chain ?x))
  ;; send message agent::pick-dominant-chain-in-merger (?id ?chain-choices)
  (format t "~%COMMAND:~d:REFEREE:(pick-dominant-chain-in-merger ~d '~s):END~%" (get ?player 'id) (get ?player 'id) ?chain-choices)
  )

;; Takes ?chain choice from ?player to dominate (take over) when there is a tie in multiple mergers.
(defun set-dominant-chain-in-merger (?id ?chain)
  ;; make sure ?id belongs to *current-player*
  (if (not (eq (get *current-player* 'id) ?id)) (wrong-id-in-response (get *current-player* 'id) ?id)
    (progn (consume-chains *current-tile* ?chain)
	   (add-adj-tiles-to-chain *current-tile* ?chain)))
  )

;; Takes the ?action (TRADE/SELL/HOLD triple) ?player has made regarding his stock in certain chains (as in a merger).
;; Applies stock decisions returned from ?player to world state.
(defun agent-stock-action (?id ?dominant-chain ?merging-chains ?actions)
  ;; make sure ?id belongs to *active-player*
  (if (not (eq (get *active-player* 'id) ?id)) (wrong-id-in-response (get *active-player* 'id) ?id)

    (progn
      ;; validate form of response
      ;; ((x ((TRADE-STOCK qty x z) (SELL-STOCK qty x) (HOLD-STOCK qty x))) (y ((TRADE-STOCK qty y z) (SELL-STOCK qty y) (HOLD-STOCK qty y))))
      
      ;; get each TRADE/SELL/HOLD triple for each chain
      (dolist (temp ?merging-chains)
	      (format t "~%what to do with chain ~s~%" temp)
	      ;; trade
	      (format t "~%trade = (car (nth (position temp ?merging-chains) ?actions)) = ~s~%" (car (nth (position temp ?merging-chains) ?actions)))
	      (setq trade (car (nth (position temp ?merging-chains) ?actions)))
	      (trade-stock *active-player* (nth 1 trade) (nth 2 trade) (nth 3 trade))
	      ;; sell
	      (format t "~%sell = (nth 1 (nth (position temp ?merging-chains) ?actions)) = ~s~%" (nth 1 (nth (position temp ?merging-chains) ?actions)))
	      (setq sell (nth 1 (nth (position temp ?merging-chains) ?actions)))
	      (sell-stock *active-player* (nth 1 sell) (nth 2 sell))
	      ;; hold - do nothing
	      )
      ;; get next player's decisions
      (format t "~%getting next player's decision~%")
      (if (eq (get-next-player *active-player*) *current-player*) (buy-stocks *current-player* *current-move*)
	(dispose-of-stocks (get-next-player *active-player*) ?dominant-chain ?merging-chains))))
  )

;; Trades ?qty of stock in chain ?x 2:1 in stock in chain ?y.
(defun trade-stock (?player ?qty ?x ?y)
  ;; if even, divide evenly
  (format t "~%qty of chain ~s=~d~%" ?x ?qty)
  (format t "~%is qty even=~s~%" (evenp ?qty))
  (if (evenp ?qty)
      (progn
	;; remove ?qty stock from ?player's ?x stock holdings
	(setf (get ?player ?x) (- (get ?player ?x) ?qty))
	;; add ?qty/2 stock to ?player's ?y stock holdings
	(setf (get ?player ?y) (+ (get ?player ?y) (/ ?qty 2))))
    (progn
      ;; else if odd, subtract 1 from ?qty, then divide evenly
      ;; remove ?qty-1 stock from ?player's ?x stock holdings
      (setf (get ?player ?x) (- (get ?player ?x) (1- ?qty)))
      ;; add (?qty-1)/2 stock to ?player's ?y stock holdings
      (setf (get ?player ?y) (+ (get ?player ?y) (/ (1- ?qty) 2)))))
  )

;; Sells ?qty of stock in chain ?x at proper market price.
(defun sell-stock (?player ?qty ?x)
  ;; remove ?qty stock from ?player's stock holdings
  (setf (get ?player ?x) (- (get ?player ?x) ?qty))
  ;; add money back to ?player's money
  (setf (get ?player 'money) (+ (get ?player 'money) (* (get-stock-price ?x) ?qty)))
  )

;; Returns T if the conditions are met for valid end of game, NIL if not.
(defun legal-end-of-game ()
  ;; does any chain have >41 members?
  (if (or (largest-chain-reached)
	  ;; are all chains too big to merge?
	  (all-chains-safe))
      t NIL)
  )

;; Give ?player choice of declaring the game is over, assumes conditions are met.
(defun player-declare-game-over (?player)
  ;; send message agent::declare-game-over (?id)
  (format t "~%COMMAND:~d:REFEREE:(declare-game-over ~d):END~%" (get ?player 'id) (get ?player 'id))
  )

;; Takes ?player's choice to end game or not (?bool = 1 if true, 0 if false).
(defun declare-game-over (?id ?bool)
  ;; make sure ?id belongs to *current-player*
  (if (not (eq (get *current-player* 'id) ?id)) (wrong-id-in-response (get *current-player* 'id) ?id)
    (progn (cond ((eq ?bool T) (format t "~%ENDING GAME~%")       (announce-winner))
		 ((eq ?bool NIL) (format t "~%CONTINUING GAME~%") (continue-game *current-player*))
		 (t NIL))))
  )

;; Announces new state of world to *players*.
(defun announce-init-world-state-to-all ()
  ;; construct current state of world, minus *players* info
  (setq init-state nil)
  (setf init-state (append init-state (cons (get *current-player* 'id) nil)))
  (setf init-state (append init-state (cons *avail-tiles* nil)))
  ;; construct list of *avail-stocks* (chain stock-left)
  (setq avail-stocks nil)
  (dolist (temp *avail-stocks*)
	  (setf avail-stocks (append avail-stocks (cons (list temp (get temp 'qty)) nil))))
  (setf init-state (append init-state (cons avail-stocks nil)))
  (setf init-state (append init-state (cons *avail-chains* nil)))
  (setf init-state (append init-state (cons *played-tiles* nil)))
  ;; construct list of *played-chains* (chain (tiles))
  (setq played-chains nil)
  (dolist (temp *played-chains*)
	  (setf played-chains (append played-chains (cons (list temp (get temp 'tiles)) nil))))
  (setf init-state (append init-state (cons played-chains nil)))
  
  ;; iterate through each player
  (dolist (temp *players*)
	  ;; send message agent::update-world-state (?temp ?init-state) (needs no response)
	  (format t "~%COMMAND:~d:REFEREE:(set-initial-world-state ~d '~s):END~%" (get temp 'id) (get temp 'id) init-state)
	  )
  )

;; Announces new state of world to *players*.
(defun announce-new-world-state-to-all ()
  ;; construct current state of world, minus *players* info
  (setq current-state nil)
  (setf current-state (append current-state (cons (get *current-player* 'id) nil)))
  (setf current-state (append current-state (cons *avail-tiles* nil)))
  ;; construct list of *avail-stocks* (chain stock-left)
  (setq avail-stocks nil)
  (dolist (temp *avail-stocks*)
	  (setf avail-stocks (append avail-stocks (cons (list temp (get temp 'qty)) nil))))
  (setf current-state (append current-state (cons avail-stocks nil)))
  (setf current-state (append current-state (cons *avail-chains* nil)))
  (setf current-state (append current-state (cons *played-tiles* nil)))
  ;; construct list of *played-chains* (chain (tiles))
  (setq played-chains nil)
  (dolist (temp *played-chains*)
	  (setf played-chains (append played-chains (cons (list temp (get temp 'tiles)) nil))))
  (setf current-state (append current-state (cons played-chains nil)))
  
  ;; iterate through each player
  (dolist (temp *players*)
	  ;; send message agent::update-world-state (?temp ?current-state) (needs no response)
	  (format t "~%COMMAND:~d:REFEREE:(update-world-state ~d '~s):END~%" (get temp 'id) (get temp 'id) current-state)
	  )
  )

;; Announces winner of game to IO module.
(defun announce-winner ()
  ;; construct string of winner/id/total money triplets
  (setq standings nil)
  (dolist (temp *players*)
	  (setq standings (append standings (cons (list temp (get temp 'id) (get temp 'money)) nil))))
  (setq standings (stable-sort standings #'> :key #'caddr))

  ;; engine is not ready to play (until the IO reinitializes it)
  (setf *go* NIL)
  ;; tell IO module game is over
  (send-endgame standings)
  
  ;; send message to *players* saying to reset themselves for the next game;
  ;; put here instead of at beginning because agents must reattach themselves
  ;; and engine doesn't keep track of who the players are going to be
  (dolist (temp *players*)
	  ;; send message agent::reset-agent (?id)
	  (format t "~%COMMAND:~d:REFEREE:(reset-agent ~d):END~%" (get temp 'id) (get temp 'id)))
  )

;; How to handle receiving a message from the wrong agent??
(defun wrong-id-in-response (?id ?bad-id)
  ;; send message agent::bad-id (?id ?bad-id)
  ;;  (format t "~%COMMAND:~d:REFEREE:(bad-id ~d ~d):END~%" ?id ?id ?bad-id)
  )


;; STATE OF WORLD
;; Determine whether the two tiles ?x and ?y are adjacent to each
;; other.  Returns NIL if not, T if yes.
(defun adjacent (?x ?y)
  ;; check left, right, up, and down pointers
  (cond ((eq (get ?x 'left)  ?y) T)
	((eq (get ?x 'right) ?y) T)
	((eq (get ?x 'up)    ?y) T)
	((eq (get ?x 'down)  ?y) T)
	(t                       NIL))
  )

;; Determine whether tile ?x is in chain ?y.  Returns NIL if not, T if
;; yes.
(defun in-chain (?x ?y)
  (cond ((eq (get ?x 'chain) ?y) T)
	(t                       NIL))
  )

;; Determine if tile ?x is in a chain.  Returns NIL if not, T if yes.
(defun in-a-chain (?x)
  (cond ((eq (get ?x 'chain) NIL) NIL)
	(t                        T))
  )

;; Determine the tiles which are adjacent to tile ?x and have been
;; played.  Returns them as a list.
(defun adj-tiles (?x)
  (let (ret-value)
    (if (not (eq (get (get ?x 'left)  'played) NIL)) (setq ret-value (append ret-value (list (get ?x 'left)))))
    (if (not (eq (get (get ?x 'right) 'played) NIL)) (setq ret-value (append ret-value (list (get ?x 'right)))))
    (if (not (eq (get (get ?x 'up)    'played) NIL)) (setq ret-value (append ret-value (list (get ?x 'up)))))
    (if (not (eq (get (get ?x 'down)  'played) NIL)) (setq ret-value (append ret-value (list (get ?x 'down)))))
    (prune ret-value)
    )
  )

;; Determines if a tile ?x is still in *avail-tiles*, i.e., can it still be played or dealt.
(defun is-tile-avail (?x)
  (cond ((member ?x *avail-tiles*) T)
	(t                         NIL))
  )

;; Determine the name of the chain ?x is in.  Returns the chain name,
;; or NIL if undefined.
(defun get-chain (?x)
  (get ?x 'chain)
  )

;; Returns T if any chain in *played-tiles* has *max-chain-length* members, NIL if not.
(defun largest-chain-reached ()
  (format t "~%ok, entering largest-chain-reached()~%")
  (show-chains)
  (cond ((eq *played-chains* NIL) NIL)
	((< 0 (list-length (prune (mapcar #'(lambda (x)
					      (cond ((>= (chain-size x) *max-chain-length*) T)
						    (t NIL)))
					  *played-chains*))))
	 (format t "~%returning T~%") T)
	(t NIL))
  )

;; Returns T if all chains in *played-chains* have more than
;; *largest-merge-length* members, NIL if not.
(defun all-chains-safe ()
  (format t "~%ok, entering largest-chain-reached()~%")
  (show-chains)
  (cond ((eq *played-chains* NIL) NIL)
	((eq (list-length *played-chains*)
	     (list-length (prune (mapcar #'(lambda (x)
					     (cond ((> (chain-size x) *largest-merge-length*) T)
						   (t NIL)))
					 *played-chains*))))
	 (format t "~%returning T~%") T)
	(t NIL))
  )

;; Determine the size of chain ?x.  Returns the size, or NIL if
;; undefined.
(defun chain-size (?x)
  (format t "~%chain-size ~s = " ?x)
  (pprint (list-length (get ?x 'tiles)))
  (list-length (get ?x 'tiles))
  )

;; Determine whether the chain ?x has reached the
;; *largest-merge-length* for mergers.  Returns NIL if not, T if yes.
(defun chain-too-big (?x)
  (cond ((>= (list-length (get ?x 'tiles)) *largest-merge-length*) T)
	(t                                                         NIL))
  )

;; Returns the list of all adjacent chains to tile ?x, with NIL's and duplicates pruned out.
(defun adj-chains (?x)
  (format t "~%ADJ-CHAINS to ~s: " ?x)
  (pprint (remove-duplicates (prune (mapcar #'get-chain (adj-tiles ?x)))))
  (remove-duplicates (prune (mapcar #'get-chain (adj-tiles ?x))))
  )

;; Returns the list of all adjacent chains to tile ?x, excluding chain ?y, with NIL's and duplicates pruned out.
(defun other-adj-chains (?x ?y)
  (remove ?y (adj-chains ?x))
  )

;; Returns the list of tiles in chain ?x from *played-chains*.
(defun get-chain-list (?x)
  (get (find ?x *played-chains*) 'tiles)
  )

;; Returns the largest chain(s) that border tile ?x.
(defun get-biggest-chain (?x)
  (let ((maximum 0) (chain NIL))
    (mapcar #'(lambda (z)
		(cond ((eq maximum (chain-size z))
		       (format t "~%  in get-biggest-chain() appending ~s to 'maximum' ~s~%" z maximum)
		       (setf chain (append chain (cons z nil))))
		      ((< maximum (chain-size z))
		       (setf maximum (chain-size z))
		       (format t "~%  in get-biggest-chain() setting 'maximum' to ~s (size of chain ~s)~%" maximum z)
		       (setf chain (append chain (cons z nil))))))
	    (adj-chains ?x))
    (format t "~%the biggest chain next to ~s = " ?x)
    (pprint chain)
    chain)
  )

;; Has last tile been played for ?player ?
(defun last-tile-played (?player)
  (cond ((= 0 (list-length (get ?player 'tiles))) T)
	(t NIL))
  )

;; Returns next player after ?player in *players*.
(defun get-next-player (?player)
  (cond ((eq (position ?player *players*) (1- (length *players*))) (nth 0 *players*))
	(t (nth (1+ (position ?player *players*)) *players*)))
  )

;; Returns 'bucket' chain ?x belongs to based on its size.
(defun get-size-bucket (?x)
  (cond ((eq (chain-size ?x) 2)  2)
	((eq (chain-size ?x) 3)  3)
	((eq (chain-size ?x) 4)  4)
	((eq (chain-size ?x) 5)  5)
	((<= (chain-size ?x) 10) 6)
	((<= (chain-size ?x) 20) 7)
	((<= (chain-size ?x) 30) 8)
	((<= (chain-size ?x) 40) 9)
	((>= (chain-size ?x) 41) 10))
  )

;; Returns the price of stock in chain ?x.
(defun get-stock-price (?x)
  ;; first determine which size bucket the chain belongs to
  (let ((bucket (get-size-bucket ?x)))
    (cond ((or (eq ?x 'TOWER) (eq ?x 'LUXOR))
	   (* (+ bucket 0) 100))
	  ((or (eq ?x 'AMERICAN) (eq ?x 'WORLDWIDE) (eq ?x 'FESTIVAL))
	   (* (+ bucket 1) 100))
	  ((or (eq ?x 'IMPERIAL) (eq ?x 'CONTINENTAL))
	   (* (+ bucket 2) 100))))
  )

;; Returns the majority stockholder's bonus for chain ?x.
(defun get-maj-bonus (?x)
  (let ((bucket (get-size-bucket ?x)))
    (cond ((or (eq ?x 'TOWER) (eq ?x 'LUXOR))
	   (* (+ bucket 0) 1000))
	  ((or (eq ?x 'AMERICAN) (eq ?x 'WORLDWIDE) (eq ?x 'FESTIVAL))
	   (* (+ bucket 1) 1000))
	  ((or (eq ?x 'IMPERIAL) (eq ?x 'CONTINENTAL))
	   (* (+ bucket 2) 1000))))
  )

;; Returns player who holds the majority of stock in chain ?x.
(defun get-maj-stockholder (?x)
  (format t "~%Entering get-maj-stockholder(~s)~%" ?x)
  (let ((maximum 0) (maj NIL))
    (mapcar #'(lambda (z)
		(cond ((eq maximum (get z ?x))
		       (format t "~%  in get-maj-SH() appending ~s to 'maximum' ~s~%" z maximum)
		       (setf maj (append (cons z nil) maj)))
		      ((< maximum (get z ?x))
		       (setf maximum (get z ?x))
		       (format t "~%  in get-maj-SH() setting 'maximum' to ~s (size of maj ~s)~%" maximum z)
		       (setf maj (append (cons z nil) nil)))))
	    *players*)
    maj)
  )

;; Returns player who holds the second majority of stock in chain ?x.
(defun get-sec-stockholder (?x)
  (format t "~%Entering get-sec-stockholder(~s)~%" ?x)
  (let ((maximum (get (car (get-maj-stockholder ?x)) ?x)) (maximum2 0) (sec NIL))
    (mapcar #'(lambda (z)
		(cond ((eq maximum2 (get z ?x))
		       (format t "~%  in get-sec-SH() appending ~s to 'maximum2' ~s~%" z maximum2)
		       (setf sec (append (cons z nil) sec)))
		      ((and (< maximum2 (get z ?x))
			    (< (get z ?x) maximum))
		       (setf maximum2 (get z ?x))
		       (format t "~%  in get-sec-SH() setting 'maximum2' to ~s (size of sec ~s)~%" maximum2 z)
		       (setf sec (append (cons z nil) nil)))))
	    *players*)
    sec)
  )


;; OUTPUT TO TERMINAL
(defun show-played-tiles ()
  (pprint *played-tiles*)
  )

(defun show-avail-tiles ()
  (pprint *avail-tiles*)
  )

(defun show-chains ()
  (dolist (temp *played-chains*)
	  (format t "~s: " temp)
	  (pprint (get temp 'tiles))
	  (format t "~%"))
  )

(defun show-players ()
  (pprint *players*)
  (dolist (temp *players*)
	  (eval (read-from-string (format nil "(pprint (get '~s 'tiles))~%" temp)))
	  (eval (read-from-string (format nil "(pprint (get '~s 'money))~%" temp)))
	  (dolist (temp2 *avail-stocks*)
		  (format t "~% ~s" temp2)
		  (eval (read-from-string (format nil "(pprint (get '~s '~s))~%" temp temp2))))
	  (format t "~%"))
  )

;; MISC LISP
;; Prune out the NIL elements of the list x.  Return altered list.
(defun prune (x)
  (cond ((eq x NIL)       NIL)
	((eq (car x) NIL) (prune (cdr x)))
	((atom (car x))   (cons (car x) (prune (cdr x))))
	(t                NIL))
  )