/*
  Connector.java
  - An object which controls a socket connection interface,
  and has a process pipe to a LISP program.
  28 April 2001
  Programmer: Michael Czajkowski
*/

// Package

// Imports

import java.io.*;
import java.net.*;
import java.util.Vector;

// <Connector>

public abstract class Connector{

    // ------ Private Data Fields ------

    // LISP FIELDS ::
    private Process lisp_process_ = null;   
    private BufferedReader from_lisp_process_ = null;
    private BufferedWriter to_lisp_process_ = null;
    private LispReader lisp_reader_ = null;
    private LispWriter lisp_writer_ = null;

    // ------ Constructors ------

    public Connector(String LOCATION){
	// PRE: Takes (String:+);
	// POST: Returns type Connector.

	// Start Running Lisp.
	try{
	    lisp_process_ = Runtime.getRuntime().exec(LOCATION);
	    	   
	    // Get the input stream from the process.
	    InputStream i = lisp_process_.getInputStream();
	    from_lisp_process_ = new BufferedReader(new InputStreamReader(i));
	
	    // Get the output stream from the process.
	    OutputStream o = lisp_process_.getOutputStream();
	    to_lisp_process_ = new BufferedWriter(new OutputStreamWriter(o));

	    // Now we set up the LispWriter and LispReader classes.

	    lisp_writer_ = new LispWriter(this, to_lisp_process_);
	    lisp_reader_ = new LispReader(this, from_lisp_process_);

	    lisp_reader_.start();

	} // try
	catch(Exception e){
	    // Error in making the process.
	    System.err.println("Error. Could not make LISP process. Restart.");
	    System.exit(1);
	} // catch
	
    } // constructor()
	
    // ------ Public Methods ------

    public abstract void handleLispCommand(String COMMAND);
       // This function will handle commands from Lisp,
       // when implemented by the subclasses.
    public abstract void handleNetCommand(SocketHandler SOCKET, String COMMAND);
       // This function will handle commands from a network socket.
       // Since Agents, Servers and other have many sockets, we specify
       // a name string to understand who has sent a system something.

    public void sendLispCommand(String COMMAND){
	// PRE: Takes a (String:+).
	// POST: Returns nothing.

	// Sends the COMMAND to the LISP process.

	lisp_writer_.sendCommand(COMMAND);
	
    } // sendLispCommand();

    public void killLispProcess(){
	// PRE: Takes nothing.
	// POST: Returns nothing.
	
	System.out.println("Destroying Lisp Process.");
	lisp_process_.destroy();
	System.out.println("Destroying Lisp Reader.");
	lisp_reader_.killLispReader();

    } // killLispProcess

    // ------ Protected Methods ------

    protected void finalize(){
	// PRE: Takes nothing.
	// POST: Returns nothing.

	killLispProcess();

    } // finalize()

    // ------ Private Methods ------

    public Vector parseCommand(String COMMAND){
	// PRE: Takes (String:+);
	// POST: Returns (java.util.Vector:+)

	// This function creates a Vector which contains:
	// 1. To whom this message is going to.
	// 2. From whom this message is sent from.
	// 3. The command itself, remote procedure.
	// .... all Strings.

	/* 
	   The most important part of parsing the Lisp Command 
	   is to know whether it was a complete command or not.
	   If COMMAND does not start with "COMMAND:" and end
	   with :END" we will return the Vector above with
	   "INVALID", "INVALID", "INVALID" in it.
	*/

	Vector command = new Vector();
	Vector returnvector = new Vector();

	int placeholder = 0;

	for(int i=0;i<COMMAND.length();i++){
	    // Go through each character and look for the :
	    // when you get to it then you add that piece into
	    // the command Vector.

	    // System.out.println("CHAR AT: "+i+" is "+COMMAND.charAt(i));

	    if(COMMAND.charAt(i) == ':'){
		// We have a semi colon. We take from placeholder
		// up to i-1 and put into a substring.
		
		String sub = COMMAND.substring(placeholder,i);

		// Update place holder
		
		placeholder = i+1;

		// Put Sub in the command vector.

		// System.out.println("Put in substring: "+sub);
		command.addElement(sub);

	    } // if
	} // for

	// Put the rest of the string into command.

	command.addElement(COMMAND.substring(placeholder));

	for(int i=0;i<command.size();i++){
	    // Print each element.
	    // System.out.println("COMMAND SUB "+i+" is "+(String)command.elementAt(i));
	} // for

	String c = null;
	String e = null;

	try{
	    c = (String)command.elementAt(0);
	    e = (String)command.elementAt(4);
	}
	catch(Exception ex){
	    // Basically we have an invalid string here.
	    returnvector.addElement("INVALID");
	    returnvector.addElement("INVALID");
	    returnvector.addElement("INVALID");
	    return returnvector;
	} // catch

	if(c.equals("COMMAND")&&e.equals("END")){
	    // Valid command.
	    System.out.println("VALID COMMAND: "+COMMAND);
	    returnvector.addElement(command.elementAt(1));
	    returnvector.addElement(command.elementAt(2));
	    returnvector.addElement(command.elementAt(3));
	} // if
	else{
	    // Invalid command.
	    System.out.println("INVALID COMMAND: "+COMMAND);
	    returnvector.addElement("INVALID");
	    returnvector.addElement("INVALID");
	    returnvector.addElement("INVALID");
	} // else

	return returnvector;
	
    } // parseCommand(String)

    // ------ Public Static Void Main ------

    // ------ Test Method ------

    public void test(){
	// PRE: Takes nothing.
	// POST: Returns nothing.
	
	// Description:
	// This method outputs information about itself to System.out.

	System.out.println("Testing method of class: "+toString());
    
    } // test()

} // class xxx



