///////////////////////////////
// qjgpu.C
// Keenan Crane (kcrane@uiuc.edu)
//
// I/i, J/j, K/k, L/l: move through parameter space
// :/; : move slice (k)
// space: toggle morph between current parameter and (0, 0, 0, 0)
// 'r': toggle rotation
// 5, 6, 7, 8, 9, 0: change resolution to 2^n (0 is 2^10)
// 'f': fullscreen
// 'g': windowed
// escape: exit
// any other key: update buffer if not morphing or rotating
//

#ifdef MAC_CONFIG

#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#include <OpenGL/glext.h>
#include <GLUT/glut.h>

#else

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <GL/glext.h>

#endif

#include <Cg/cg.h>
#include <Cg/cgGL.h>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#define CONST_INC 0.05 //increment speed of parameters

CGcontext cgContext;
CGprogram cgFragmentProgram, cgVertexProgram;
CGprofile cgFragmentProfile, cgVertexProfile;
int refresh, animateOn, rotateOn;

GLuint texture; //buffer for scaling
int res; //size of buffer (2^res*2^res)

static float t = 0;  //rotation, morph timers
static float u = 0; 

//parameters passed to shader
float light[3], eye[3], lookAt[3], up[3], fov;
float N[3], T[3], B[3]; //projected subspace
float mu[4], slice; //set parameter, k-slice

void enable(int flag)
{
   ;
}

void disable(int flag)
{
   ;
}

void calculateView()
   //convert eye, lookAt, up to basis vectors
{
   float mag, dot;
   int i;

   eye[0] = 3 * cos(u);
   eye[1] = 0.0;
   eye[2] = 3 * sin(u);
   light[0] = 20 * cos(u);
   light[1] = 0.0;
   light[2] = 20 * sin(u);

   for(i=0; i<3; i++)
      N[i] = lookAt[i] - eye[i];

   mag = sqrt(N[0]*N[0] + N[1]*N[1] + N[2]*N[2]);
   for(i=0; i<3; i++)
      N[i] /= mag;

   dot = N[0]*up[0] + N[1]*up[1] + N[2]*up[2];
   for(i=0; i<3; i++)
      T[i] = up[i] - N[i]*dot;
   mag = sqrt(T[0]*T[0] + T[1]*T[1] + T[2]*T[2]);
   for(i=0; i<3; i++)
      T[i] /= mag;

   B[0] = N[1]*T[2] - N[2]*T[1];
   B[1] = N[0]*T[2] - N[2]*T[0];
   B[2] = N[0]*T[1] - N[1]*T[0];
}

void resizeTexture(int newRes)
{
   res = newRes;
   glGenTextures(1, &texture);
   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
   glBindTexture(GL_TEXTURE_2D, texture);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   unsigned char *im = new unsigned char[res*res*3];
   for(int i=0; i<res*res*3; i++) im[i] = 0;
   glTexImage2D(GL_TEXTURE_2D,
                0,
                GL_RGB,
                res,
                res,
                0,
                GL_RGB,
                GL_UNSIGNED_BYTE,
                im);
   delete[] im;
}

bool Initialize()
{
   glClearColor (0.0f, 0.0f, 0.0f, 0.5f);
   glClearDepth (1.0f);

   resizeTexture(128);

   //initialize Cg
   cgContext = cgCreateContext();
   if(!cgContext) { printf("Could not create Cg context.\n"); exit(1); }
   cgFragmentProfile = cgGLGetLatestProfile(CG_GL_FRAGMENT);
   cgVertexProfile = cgGLGetLatestProfile(CG_GL_VERTEX);
   if(cgFragmentProfile == CG_PROFILE_UNKNOWN || cgVertexProfile == CG_PROFILE_UNKNOWN)
   { printf("Could not create Cg profile.\n"); exit(1); }
   cgGLSetOptimalOptions(cgFragmentProfile);
   cgGLSetOptimalOptions(cgVertexProfile);
   cgFragmentProgram = cgCreateProgramFromFile(cgContext, CG_SOURCE, "QJuliaFragment.cg", cgFragmentProfile, "main", 0);
   if(!cgFragmentProgram)
   { CGerror error=cgGetError(); printf("%s (fragment)\n", cgGetErrorString(error)); exit(1); }
   cgVertexProgram = cgCreateProgramFromFile(cgContext, CG_SOURCE, "QJuliaVertex.cg", cgVertexProfile, "main", 0);
   if(!cgVertexProgram)
   { CGerror error=cgGetError(); printf("%s (vertex)\n", cgGetErrorString(error)); exit(1); }
   cgGLLoadProgram(cgFragmentProgram);
   cgGLLoadProgram(cgVertexProgram);

   //setup view
   eye[0] = 0;
   eye[1] = 0;
   eye[2] = -3;

   lookAt[0] = 0;
   lookAt[1] = 0;
   lookAt[2] = 0;

   up[0] = 0;
   up[1] = 1;
   up[2] = 0;

   fov = 60.0;
   calculateView();
   
   slice = 0.0;

   //set to the unit sphere
   mu[0] = mu[1] = mu[2] = mu[3] = 0.0;

   //set flag to draw an initial frame
   refresh = 1;

   return true;
}

void Deinitialize (void)
{
   ;
}

void displayConstant()
   //display current parameter
{
   printf("[ %f %f %f %f ]\n", mu[0], mu[1], mu[2], mu[3]);
}

void Update (int key,int x,int y)
{
   if ( key==27 )
   {
      Deinitialize();
      exit(0);
   };

   if(key == ' ')
      animateOn = !animateOn;
   else if(key == 'r')
      rotateOn = !rotateOn;
   else if(key == 'l')
      mu[0] += CONST_INC; 
   else if(key == 'L')
      mu[0] -= CONST_INC; 
   else if(key == ';')
      slice += CONST_INC;
   else if(key == ':')
      slice -= CONST_INC;
   else if(key == 'i')
      mu[1] += CONST_INC; 
   else if(key == 'I')
      mu[1] -= CONST_INC; 
   else if(key == 'j')
      mu[2] += CONST_INC; 
   else if(key == 'J')
      mu[2] -= CONST_INC; 
   else if(key == 'k')
      mu[3] += CONST_INC; 
   else if(key == 'K')
      mu[3] -= CONST_INC; 
   else if(key=='f')
      glutFullScreen();	
   else if(key=='5')
      resizeTexture(32);
   else if(key=='6')
      resizeTexture(64);
   else if(key=='7')
      resizeTexture(128);
   else if(key=='8')
      resizeTexture(256);
   else if(key=='9')
      resizeTexture(512);
   else if(key=='0')
      resizeTexture(1024);
   else if(key=='g')
   {
      glutPositionWindow(50, 50);
      glutReshapeWindow(128, 128);
   }

   refresh = 1;

   glutPostRedisplay();
}


void Draw (void)
{
   //don't always update (it can take a while to render at high res)
   if(refresh || animateOn || rotateOn)
   {
      displayConstant();

      glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity ();

      //save the current viewport information and set the viewport to the
      //desired rendering size
      int origViewport[4];
      glGetIntegerv(GL_VIEWPORT, origViewport);
      glViewport(0, 0, res, res);
      
      cgGLEnableProfile(cgVertexProfile);
      cgGLBindProgram(cgVertexProgram);
      cgGLEnableProfile(cgFragmentProfile);
      cgGLBindProgram(cgFragmentProgram);

      CGparameter constant = cgGetNamedParameter(cgFragmentProgram, "mu");
      CGparameter eyeP = cgGetNamedParameter(cgFragmentProgram, "eye");
      CGparameter lightP = cgGetNamedParameter(cgFragmentProgram, "light");

      if(animateOn) //morph between current parameter and target parameter
      {
         t += 0.1;
         cgGLSetParameter4f(constant, 
               mu[0]*(0.5*cos(t)+0.5),
               mu[1]*(0.5*cos(t)+0.5),
               mu[2]*(0.5*cos(t)+0.5),
               mu[3]*(0.5*cos(t)+0.5));
      }
      else
         cgGLSetParameter4f(constant, mu[0], mu[1], mu[2], mu[3]);

      if(rotateOn) //increment rotation angle
         u += 0.06;


      calculateView(); //find the updated basis

      //pass eye and light to shader for lighting
      cgGLSetParameter3f(eyeP, eye[0], eye[1], eye[2]);
      cgGLSetParameter3f(lightP, light[0], light[1], light[2]);

      CGparameter modelViewProjection = cgGetNamedParameter(cgVertexProgram, "modelViewProj");
      cgGLSetStateMatrixParameter(modelViewProjection, CG_GL_MODELVIEW_PROJECTION_MATRIX, CG_GL_MATRIX_IDENTITY);

      glBegin(GL_QUADS);

      float alpha = tan((fov*M_PI/180.0)/2.0);

      //send ray information in as texture coordinates and colors
      //the corners of the texture specify the position and direction of the
      //rays being shot from the corner of the screen (which are then
      //interpolated)
      
      glTexCoord4f(eye[0], eye[1], eye[2], slice);

      glColor4f(-alpha*T[0] - alpha*B[0] + N[0],
            -alpha*T[1] - alpha*B[1] + N[1],
            -alpha*T[2] - alpha*B[2] + N[2],
            slice);
      glVertex2f(-1, -1);

      glColor4f( alpha*T[0] - alpha*B[0] + N[0],
            alpha*T[1] - alpha*B[1] + N[1],
            alpha*T[2] - alpha*B[2] + N[2],
            slice);
      glVertex2f( 1, -1);

      glColor4f( alpha*T[0] + alpha*B[0] + N[0],
            alpha*T[1] + alpha*B[1] + N[1],
            alpha*T[2] + alpha*B[2] + N[2],
            slice);
      glVertex2f( 1,  1);

      glColor4f(-alpha*T[0] + alpha*B[0] + N[0],
            -alpha*T[1] + alpha*B[1] + N[1],
            -alpha*T[2] + alpha*B[2] + N[2],
            slice);
      glVertex2f(-1,  1);

      glEnd();

      cgGLDisableProfile(cgFragmentProfile);
      cgGLDisableProfile(cgVertexProfile);

      //copy the rendered image into a texture so it can be scaled to the size
      //of the window
      glBindTexture(GL_TEXTURE_2D, texture);
      glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, res, res);

      //restore the viewport
      glViewport(origViewport[0],
            origViewport[1],
            origViewport[2],
            origViewport[3]);

      refresh = 0;
   }

   //draw the scaled image
   glColor4f(1.0, 1.0, 1.0, 1.0);
   glEnable(GL_TEXTURE_2D);
   glBegin(GL_QUADS);

   glTexCoord2f(0, 0);
   glVertex2f(-1, -1);

   glTexCoord2f(1, 0);
   glVertex2f( 1, -1);

   glTexCoord2f(1, 1);
   glVertex2f( 1,  1);

   glTexCoord2f(0, 1);
   glVertex2f(-1,  1);

   glEnd();
   glDisable(GL_TEXTURE_2D);

   glFlush ();
   glutSwapBuffers();
}

void ReshapeGL (int width, int height)
{
   glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   glViewport (0, 0, (GLsizei)(width), (GLsizei)(height));

   glMatrixMode (GL_PROJECTION);
   glLoadIdentity();
   glOrtho(-1, 1, -1, 1, 0, 100);

   glMatrixMode (GL_MODELVIEW);
   glLoadIdentity ();

   glutPostRedisplay();
}

void Key(unsigned char key,int x,int y)
{
   Update(key,x,y);
   return ;
};

void OnIdle()
{
   glutPostRedisplay();
};

int main( int argc, char *argv[] )
{
   glutInit( &argc, argv );
   glutInitWindowPosition( 30, 30 );
   glutInitWindowSize( 256, 256 );
   glutInitDisplayMode( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );
   glutCreateWindow("QJulia");
   glutReshapeFunc( ReshapeGL );
   glutKeyboardFunc( Key );
   glutSpecialFunc( Update );
   glutDisplayFunc( Draw );
   glutIdleFunc( OnIdle );
   Initialize();
   glutMainLoop();
   return 0;
}
