package automata;

import automata.actions.*;

public class Element {
  private static int ID_COUNTER = 0;
  private int ID;
  
  private Location l;
  private State s;
  private double nextTransition;
  
  public Element(State s, Location l){
    this.s = s; this.l = l; nextTransition = 0;
    
    ID = ID_COUNTER++;
  }
  
  // Deep copy constructor
  public Element(Element other){
    this.s = new State(other.s);
    this.nextTransition = other.nextTransition;
    this.l = other.l;
    this.ID = other.ID;
  }
  
  public Location getLocation() { return l; }
  public void setLocation(Location l) { this.l = l; }

  public State getState() { return s; }
  public void setState(State s) { this.s = s; }
  
  public double getNextTransition() { return nextTransition; }
  //public void setNextTransition(double nextTransition) { this.nextTransition = nextTransition; }
  
  /**
   * Decide the next transition and possibly perform it.
   * @return null if the element should disappear (die action) <br />
   *         itself (this) if the element carries on as usual (null action)
   *         Element != this if a new element is created for the return (new action)
   */
  public Element transition(double time, MessageLayer ml){
    if(nextTransition != 0)
      System.err.println("[Warning] Element transitioning out of sync: " + nextTransition);
    
    int tr = l.transition(s, ml);
    if(tr < 0) // Continuous transition
      nextTransition = l.cTransition(s);
    else { // Discrete transition      
      Edge e = l.getEdge(tr);
      // Action based on previous state
      if(e.hasAction()){
        State previousState = s;
        switch(e.getAction().getType()){
          case IAction.NEW_ACTION:
            s = e.transition(s);
            l = e.getDst();
            return ((NewAction) e.getAction()).execute(previousState);
            
          case IAction.DIE_ACTION:
            return null;
            
          case IAction.SEND_ACTION:{
            s = e.transition(s);
            l = e.getDst();
            SendAction a = ((SendAction) e.getAction());
            Message m = a.produceMessage(previousState);
            m.setTime(time);
            ml.insertMessage(a.getChannel(previousState), m);
            return this;
          }
          case IAction.RECEIVE_ACTION: {
            ReceiveAction a = ((ReceiveAction) e.getAction());
            Message m = ml.receiveMessage(a.getChannel(previousState));
            s = a.transition(previousState, m.getPayload());
            l = e.getDst();
            return this;
          }
          default:
            System.out.println("[WARNING] Unknown action to be executed.");
            break;
        }
      }
      
      
      s = e.transition(s);
      l = e.getDst();
    }
    
    return this;
  }
  
  public void transition(double t) { // Continuous transition
    if(nextTransition - t < 0)
      System.err.println("[Warning] Element evolving continuously for longer than desired.");
    
    nextTransition = nextTransition - t;
    s = l.cTransition(t, s);
  }
  
  public String toString(){ return "E["+ID+"]{" +l+ ", " + ((int)(100*nextTransition))/100f + ", " + s + "}"; }
}
