/** @file rt_trimesh.cpp
 *
 *  @brief Implementation of a raytracer triangle mesh
 *
 *  You should add your code here
 **/

#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>
#include <map>

#include "rt_trimesh.h"
#include "rt_triangle.h"
#include "rt_vectors.h"

#define MAX_LINE_LEN 1000

using namespace std;

/** The code below reads in a triangle mesh and creates all the triangles
 *  defined in the mesh.  No optimization is done.
 **/
rt_TriMesh::rt_TriMesh(rt_parse *parser, const char * objfile, int mesh_mat)
  : rt_Object(parser)
{
  ifstream infile (objfile); assert (infile!=NULL);
  
  char line[MAX_LINE_LEN];
  char command[MAX_LINE_LEN];
  int position;
  vector<Vec3f*> vertices;
  vector<Vec3f*> normals;
  
  while (infile.good ())
  {
    infile.getline (line, MAX_LINE_LEN);
    parse (line, MAX_LINE_LEN, command, position);

    if (strcmp (command,"v")==0)
    {
      Vec3f *pos = getVec (&(line[position]));
      vertices.push_back (pos);
    }
    else if (strcmp (command,"vn")==0)
    {
      Vec3f *norm = getVec (&(line[position]));
      normals.push_back (norm);
    }
    else if (strcmp (command,"f")==0)
    {
      int num = 0; // number of edges
      int v_index[3]; // vertex index
      int n_index[3]; // normal index
      
      getCoords (&(line[position]), num, v_index, n_index);
      assert (num==3);
      rt_Triangle * tri =
	new rt_Triangle(parser,*(vertices[v_index[0]]),*(vertices[v_index[1]]),
			*(vertices[v_index[2]]),*(normals[n_index[0]]),
			*(normals[n_index[1]]),*(normals[n_index[2]]),
			mesh_mat,mesh_mat,mesh_mat,0,0,0,0,0,0);
      triangles.push_back (tri);
    }
    else
    {
      //cout << "Ignoring command <"<<command<<"> in obj file"<<endl;
    }
  }
  infile.close ();
}


void rt_TriMesh::parse (char * line, int max, char * command, int & position)
{
  int i = 0;
  int j = 0;

  while (i<max && line[i]==' ') i++;
  while (i<max && line[i]!='\0' && line[i]!=' ')
  {
    command[j] = line[i];
    j++; i++;
  }
  
  command[j] = '\0';
  position = i;
}

Vec3f* rt_TriMesh::getVec (char * str)
{
  int i = 0;
  float x,y,z;
  while (str[i]!='\0' && str[i]==' ') i++;
  x = atof (&(str[i]));
  while (str[i]!='\0' && str[i]!=' ') i++;
  y = atof (&(str[i]));
  while (str[i]!='\0' && str[i]==' ') i++;
  while (str[i]!='\0' && str[i]!=' ') i++;
  z = atof (&(str[i]));
  return new Vec3f(x,y,z);
}

void rt_TriMesh::getCoords (char *str, int &num, int v_index[3], int n_index[3])
{
  int i = 0;
  num = 0;
  
  while (str[i]!='\0' && str[i]==' ') i++;
  while (str[i]!='\0')
  {
    while (str[i]!='\0' && str[i]!=' ') i++;
    while (str[i]!='\0' && str[i]==' ') i++;
    num++;
  }
  assert (num==3);
  
  i = 0;
  for (int j=0; j<num; j++)
  {
    while (str[i]==' ') i++;
    v_index[j] = atoi (&(str[i])) - 1;
    while (str[i]!='/' && str[i]!=' ') i++;
    assert (str[i]=='/');
    i++;
    while (str[i]!='/' && str[i]!=' ') i++;
    assert (str[i]=='/');
    i++;
    n_index[j] = atoi (&(str[i])) - 1;
    while(str[i]!='\0' && str[i]!=' ') i++;
  }
}
