/** @file rt_parse.h
 *  
 *  @brief Definitions and interface for the Scene Description Language parser
 *
 *  The SDL parser is a C++ class that can convert an SDL file to internal
 *  structures for your program. It assumes specially-formatted constructors
 *  exist for the following classes:
 * 
 *  <ul>
 *  <li>rt_Light
 *  <li>rt_Object
 *  <li>rt_Group
 *  <li>rt_Sphere
 *  <li>rt_Triangle
 *  <li>rt_Material
 *  </ul>
 *
 *  @author Created by Sriram Vaidhyanathan on Thu Feb 26 2004.
 *  @author Modified by Mark T. Tomczak, October 2004.
 *  @author Modified by Miklos Bergou, March 2005.
 *
 *  15-462 Computer Graphics Spring 2004 Programming Assignment 3
 *
 */

#ifndef _RT_PARSE_H_
#define _RT_PARSE_H_

#include "rt_vectors.h"
#include <vector>

/**********************************************/
/* You need to define and write these classes */
/**********************************************/
class rt_Light;
class rt_Object;
class rt_Group;
class rt_Sphere;
class rt_Triangle;
class rt_TriMesh;
class rt_Material;

/* For parsing purposes */
#define MAX_PARSER_TOKEN_LENGTH 100

class rt_parse
{
public:

    /** @brief Constructor for class rt_parse 
     */
    rt_parse();
    
    /** @brief Constructor for class rt_parse
     *
     *  @param filename Name of file to parse
     **/
    rt_parse(const char *filename);

    /** @brief Destructor 
     */
    ~rt_parse();

    /** @brief Returns a 3-vector with the background color attributes 
     *
     *  Returns a vector describing color attributes in the three
     *  array elements (organized red-green-blue).
     */
    Vec3f getBackgroundColor() { return background_color; }

    /** @brief Returns a 3-vector with ambient light attributes 
     *
     *  Returns a vector describing color attributes in the three
     *  array elements (organized red-green-blue).
     */
    Vec3f getAmbientLight() { return ambient_light; }

    /** @brief Returns the number of lights in the scene 
     */
    int getNumLights() { return (int)(lights.size()); }

    /** @brief Returns the number of materials in the scene 
     */
    int getNumMaterials() { return (int)(materials.size()); }

    /** @brief Given an integer i, returns the ith light in the scene 
     *
     *  Lights are numbered 0 to (number_of_lights - 1)
     */
    rt_Light* getLight(unsigned int i)
    {
        if((i < 0) || (i >= lights.size()))
        {
          cout<<"ERROR: request for light "<<i<<", which is outside "<<
                "acceptable indices. There are "<<lights.size()<<" lights.\n";
          exit(-1);
        }
        rt_Light *result = lights[i];
        assert(result != NULL);
        return result;
    }

    /** @brief Given an integer i, returns the ith material in the scene 
     *
     *  Materials are numbered 0 to (number_of_materials - 1)
     */
    rt_Material* getMaterial(unsigned int i)
    {
        if((i < 0) || (i >= materials.size()))
        {
          cout<<"ERROR: request for material "<<i<<", which is outside "<<
                "acceptable indices. There are "<<lights.size()<<" lights.\n";
          exit(-1);
        }
        rt_Material *result = materials[i];
        assert (result != NULL);
        return result;
    }

    /** @brief Returns the group 
     *  The SDL organizes all objects under one root group. This function 
     *  returns a pointer to the root group
     */
    rt_Group* getGroup()
    {
        return group;
    }
    
    /** @brief Returns the camera position
     *
     *  Camera position is specified by (x,y,z) coordinates.
     */
    Vec3f getEye() { return eye; }
        
    /** @brief Returns the camera look-at vector
     *
     *  Camera look-at vector is specified by (x,y,z) coordinates.
     */
    Vec3f getCenter() { return center; }
        
    /** @brief Returns the camera up vector
     *
     *  Camera camera up vector is specified by (x,y,z) coordinates.
     */
    Vec3f getUp() { return up; }
        
    /** @brief Returns the camera's vertical field of view
     *
     *  Camera's vertical field of view (in degrees)
     */
    float getFovy() { return fovy; }

private:

    /** @brief Initializes the parsing process 
     *
     *  Internal function that inits the parser.
     */
    void rt_parse_initialize();

    /** @brief munges whitespace and comments out of the file 
     *
     *  Used by the parser to consume whitespace
     */
    void eatWhitespace(void);

    /** @brief Returns a single token 
     */
    int getToken(char token[MAX_PARSER_TOKEN_LENGTH]);

    /** @brief Reads in a 3-vector 
     */
    Vec3f readVec3f();

    /** @brief Reads in a single float 
     */
    float readFloat();

    /** @brief Reads in a single int 
     */
    int readInt();

    /** @brief Parses the file
     */
    void parseFile();
    
    /** @brief Called by parseFile to parse background information
     */
    void parseBackground();
    
    /** @brief Called by parseFile to parse lighting information
     */
    void parseLights();
    
    /** @brief Called by parseFile to parse camera information
     */
    void parseCamera();
    
    /** @brief Called by parseFile to parse material information 
     */
    void parseMaterials();
    
    /** @brief Called by parseMaterials to parse a specific material
     */
    void parseMaterialIndex();
    
    /** @brief Parses and returns a group. Also parses the objects
     *         within the group
     */
    rt_Group* parseGroup();
    
    /** @brief Parses and returns a sphere
     */
    rt_Sphere* parseSphere();
    
    /** @brief Parses and returns a triangle
     */
    rt_Triangle* parseTriangle();
    
    /** @brief Parses and returns a triangle mesh (obj file)
     */
    rt_TriMesh* parseTriMesh();

    /* Private data members for class rt_parse */
    rt_Group *group; /**< The group information */

    Vec3f background_color; /**< Background "sky" color (red, green, blue) */
    Vec3f ambient_light;    /**< Ambient light (red, green, blue */
    Vec3f eye;              /**< Camera location (x, y, z) */
    Vec3f center;           /**< Camera looking at (x, y, z) */
    Vec3f up;               /**< Camera up vector (x, y, z) */
    float fovy;             /**< Camera vertical field of view */

    FILE *file;     /**< reference to the open file during parsing */
    
    int parse_char; /**< current character read from file */
    int curline;    /**< current line in the file */

    std::vector<rt_Light *> lights; /**< List of lights in file */
    std::vector<rt_Material *> materials; /**< List of materials in file */
    rt_Material *current_material; /**< The material currently being parsed */
};

#endif
