% -*- SLang -*-		bufed.sl
%
% Simple JED `bufed' mode by Mark Olesen <olesen@me.QueensU.CA>
%
% Bufed is a simple buffer manager -- patterned somewhat after dired.
% Provides easy, interactive switching, saving and killing of buffers.
%
% To invoke Bufed, do `M-x bufed'.
% Or re-bind to the key sequence which is normally bound to the
% `list_buffers' function `C-x C-b' (emacs)
%
% ------------------------------------------------------------------------
% TO USE THIS MODE: add the line
%
%	autoload ("bufed", "bufed");		to ~/.jedrc
%
% and optionally re-bind to the `C-x C-b' (emacs) key sequence
%
%	setkey ("bufed", "^X^B");
% ------------------------------------------------------------------------

variable Bufed_buf = "*BufferList*";	% as used by `list_buffers' (buf.sl)

% save the buffer
define bufed_savebuffer (buf)
{
   variable file, dir, flags, ch, this_buf;

   ch = int (buf);
   if ((ch == 32) or (ch == '*')) return;	% internal buffer or special

   this_buf = whatbuf ();
   setbuf (buf);
   (file,dir,,flags) = getbuf_info ();

   ERROR_BLOCK { setbuf (this_buf); }
   if (strlen (file) and (flags & 1))	% file assciated with it
     {
	!if (write_buffer (strcat (dir, file)))
	  error (strcat ("Error writing buffer ", buf));
     }
   EXECUTE_ERROR_BLOCK;
}

% extract the buffer name associated with the current line
define bufed_get ()
{
   push_spot_bol ();
   EXIT_BLOCK { pop_spot (); }

   () = ffind_char ('\t');
   if (bolp ()) return Null_String;

   go_right_1 ();
   push_mark ();
   () = ffind_char ('\t');
   bufsubstr ();		% leave on the stack
}

define bufed_list ()
{
   list_buffers ();
   pop2buf (Bufed_buf);
   go_down (4);
}

% kill a buffer, if it has been modified then pop to it so it's obvious
define bufed_kill ()
{
   variable file, dir, flags, buf = bufed_get ();
   !if (int (buf)) return;

   setbuf (buf);
   (file,dir,,flags) = getbuf_info ();

   if (flags & 1)		% modified
     {
	pop2buf (buf);
	update (1);
     }
   delbuf (buf);
   if (strcmp (buf, Bufed_buf))
     bufed_list ();
}

define bufed_save ()
{
   variable buf = bufed_get ();
   !if (int (buf)) return;
   bufed_savebuffer (buf);
}

% try to re-load the file from disk
define bufed_update ()
{
   variable file, dir, flags;
   (file,dir,,flags) = getbuf_info ();
   if (flags & 2)		% file on disk modified?
     {
	!if (find_file (strcat (dir, file)))
	  error ("Error reading file");
     }
}


define bufed_pop2buf ()
{
   variable buf = bufed_get ();

   !if (int (buf)) return;

   % if the buffer is already visible, scroll down
   buffer_visible (buf);	% leave on the stack
   pop2buf (buf);
   if (()) call ("page_down");

   bufed_update ();
   pop2buf (Bufed_buf);
}

define bufed_sw2buf ()
{
   variable buf = bufed_get ();
   !if (int (buf)) return;
   sw2buf (buf);
   bufed_update ();
}

define bufed_exit ()
{
   delbuf (whatbuf ());
}

variable Bufed_help;
Bufed_help = "k:kill, s:save, g:refresh, SPC,f:pop2buf, CR:sw2buf, q:quit, h:help, ?:this help";

define bufed_help
{
   message (Bufed_help);
}

$1 = "bufed";
!if (keymap_p ($1)) make_keymap ($1);
definekey ("bufed_list",	"g",	$1);
definekey ("describe_mode",	"h",	$1);
definekey ("bufed_kill",	"k",	$1);
definekey ("bufed_save",	"s",	$1);
definekey ("bufed_pop2buf",	"f",	$1);
definekey ("bufed_pop2buf",	" ",	$1);
definekey ("bufed_sw2buf",	"\r",	$1);
definekey ("bufed_exit",	"q",	$1);
definekey ("bufed_help",	"?",	$1);

% Also possible,
%  U	toggle_undo
%  O	toggle_overwrite
%  R	toggle_readonly
%  C	toggle_crmode

%!%Mode designed to aid in navigating through multiple buffers
%!%patterned somewhat after dired.
%!%
%!%To invoke Bufed, do @M-x bufed@ or bind to @C-x C-b@ (emacs)
%!%
%!% @g@	Update the buffer listing.
%!%
%!% @k@	Kill the buffer described on the current line, like typing
%!%	@M-x kill_buffer@ and supplying that buffer name.
%!%
%!% @s@	Save the buffer described on the current line.
%!%
%!% @f@, @CR@, @SPC@
%!%	Visit the buffer described on the current line. Using @SPC@
%!%	will create a new window if required, the others will use
%!%	the current window.
%!%
%!% @Q@	Quit bufed mode.
define bufed ()
{
   variable mode = "bufed";

   bufed_list ();
   bufed_help ();
   use_keymap (mode);
   set_mode (mode, 0);
   runhooks ("bufed_hook");
}
