/*
	A demonstration of (primarily) the floating point type "double"
	and mixed-type arithmetic
	
*/

public class RealDemo {

	public static void main(String args[]) {
		
		/* Example 1:
		    A "complicated" arithmetic expression
		    
		    Issue 1: does an intermediate computation "ruin" the
		    result because it exceeds the range of int?
		         Yes. the computation " a * a * b " has a value
		         beyond the range of int
		      One solution: use longs
		      A second possible solution that will sometimess work: 
		         change the order of computation; instead of 
		         " a*a*b/c ", use " a*a/c*b " which is mathematically
		         the same but which avoids going beyond the range
		    
		*/
		int a = 100,
			b = 5000,
			c = 4;
			
		int result = a * a * b / c  + b * b - c + 17;
		
		
		/*
			Example 2: Use of doubles
		*/
		
		double x = 3.199995,
			   y = 4.0;
			   
		System.out.println("x = " + x + " and y = " + y);
		System.out.println("x - y = " + (x - y) );
		System.out.println("x / y = " + x / y);
		
		/*
			Example 3: obtaining "real" division when the types of
			    the numerator doesn't work
		*/
		
		// this doesn't work
		double z = a / b;
		System.out.println ("double z = a / b yields " + z );
		
		
		// Mixed type arithmetic: if one of the operands is a richer
		// type the other type is implicitly "promoted" to the richer
		// type. 
		// In other words, since the denominator is a double, "real"
		// division is used.
		z =  a / 5000.0;
		System.out.println ("double z = a / 5000.0 yields " + z);



		// Casting: the general solution
		// Casting explicitly "promotes" a type to a richer type
		// Here, the numerator is treated as a double, so "real" division
		// gets used.
		z = (double)a / b;
		System.out.println ("double z = (double) a / b yields " + z );
		
		
		// Casting should not be abused in Java
		// In a language like C or C++, casting is used a lot but
		// Java is "strongly typed"	which means casting is generally
		// not required.
		// C programmers might use casting to "demote" a double to an
		// int but, in Java, you should the Math class
		z = Math.round(x);
		System.out.println("The int nearest to " + x + " is " + z);
		System.out.println("The smallest int greater than " + x 
							+ " is " + Math.ceil(x) );
		System.out.println("The largest int less than " + x 
							+ " is " + Math.floor(x) );
							
		/*
			Example 4: When is integer division used and 
				when is real division used?
		*/
		System.out.println ();										
		System.out.println ("Example 4:");

		double w = z * x +  a / b * y - 4;
		System.out.println ("The double obtained from the first expression: " + w);
		
		w = z * x +  y * a / b  - 4;
		System.out.println ("The double obtained from the second expression: " + w);
		/* In the first case, integer division is used and, in the second
		   case, real division is used (even though the two statements
		   are mathematically equivalent.
		
		   Terms are evaluated left-to-right
		   
		   In the first case ("a/b*y"), the compiler sees an int 
		   numerator and an int denominator, so it uses int division.
		   
		   In the second case ("y*a/b"), the numerator is a double ("y*a" is
		   a double since a is implicitly cast as a double), so
		   real division is used.
		*/
		System.out.println ("a/b*y = " + a/b*y + " while y*a/b = " + y*a/b);
	}
}
