/*
	This main program CDLibrary handles the commands
	necessary to manipulate a CDCollection (a collection class for CDs)
	
	Written by Jeff Carroll 3/25/05

*/
import java.io.*;
import java.util.*;
public class CDLibrary {

	public static BufferedReader kb = new BufferedReader(
										new InputStreamReader(System.in));
	
	public static void main(String args[]) throws IOException {

		System.out.println("This is my CD collection updater\n");	
		CDCollection  cdLibrary = new CDCollection();

		// Stage 1: Initializing
		if (getYesResponse("Do you have an existing collection? ")) {
			loadCollection(cdLibrary);
			for (int i = 0; i < cdLibrary.getSize(); i++)
				System.out.println(cdLibrary.get(i));
		}
		else 
			System.out.println("Fine, we'll start one.");
		
		String title =null;
		String artist = null;
		try {
			System.out.print  ("What artist are you searching for? ");
			artist = kb.readLine();
			System.out.print  ("What title are you searching for? ");
			title = kb.readLine();			
		}
		catch (IOException e){}
			

		CD theAnswer = cdLibrary.search(title, artist);
		if (theAnswer == null)
			System.out.println("Sorry, couldn't find the CD");
		else
			System.out.println("Here it is: " + theAnswer);

		CDCollection subcollection = cdLibrary.filterByArtist (artist);
		System.out.println("Here is a list of all CDs by " + artist);
		for (int i = 0; i < subcollection.getSize(); i++)
			System.out.println(subcollection.get(i));


		// Stage 2: Adding to the array
		boolean willContinue = getYesResponse ("Do you wish to add a CD to the collection? ");
		while (willContinue) {
			CD newCD = addCD();
			if (newCD != null) {
				cdLibrary.append(newCD);
				willContinue = getYesResponse ("Do you wish to add another CD to the collection? ");				
			}
			else
				System.out.println("Please enter the information for this CD again");
		}
				
		// Stage 3: Finishing  with file update
		fileUpdate(cdLibrary);

	}
	
	
	/* ******************************** *
	 * 		Stage 1 File Input Methods	*
	 * ******************************** */
	 
	public static void loadCollection (CDCollection c) {
	// This method opens a file containing CD info for input
	// it then loads the contents into the array arr.
	// The final logical size of the array is returned
		BufferedReader cdFile = 
			getInputStream("What is the file name for the collection? ");
	
		String nextLine = null;
		try {
			nextLine = cdFile.readLine();
		} catch (IOException e) 
			{System.out.println("General I/O Error: " + e);}
		
		while (nextLine != null) {
		
			CD next = getCD(nextLine);
			if (next != null) {
				c.append(next);
			}
			try {
				nextLine = cdFile.readLine();
			} catch (IOException e) 
				{System.out.println("General I/O Error: " + e);}
		}
	}

	
	public static CD getCD (String text) {
	// This method creates a CD object from the file encoded
	// in the string,
	// If the encoding is wrong, null is returned
		StringTokenizer st = new StringTokenizer(text,",");
		try {
			String name = st.nextToken();
			String title = st.nextToken();
			if (!st.hasMoreTokens())
				return new CD(name, title);
			int tracks = Integer.parseInt(st.nextToken());
			int timing = Integer.parseInt(st.nextToken());
			return new CD(name,title,tracks,timing/60,timing%60);
		}
		catch (Exception e) {
			System.out.println("I/O Error: unable to parse CD info: \"" +
						text + "\"");
			return null;
		}
	}


	public static BufferedReader getInputStream(String prompt) {
	// This method continually attempts to obtain a file that can
	// be used for input. It returns a new buffered reader object
	// that reads the file
	
		FileInputStream fis = null;
		// the creation of the file input stream is where the 
		// the i/o exception error can occur. The loop below
		// uses the fact that assigning fis a value has not
		// been successful.
		
		do {
			System.out.print (prompt);

			try {
				String fileName = kb.readLine();
				fis = new FileInputStream(new File(fileName));
			} 
			catch (FileNotFoundException fnne) {
				System.out.println(fnne);
				System.out.println("The file name you entered doesn't appear to exist");
			}
			 catch (IOException ioe) { // this exception is unlikely
				System.out.println("General I/O Error: " + ioe);
			}
			
		} while (fis == null);
		
		return new BufferedReader (new InputStreamReader(fis));
	}
	

	/* **************************************** *
	 * 		Stage 2 Interactive Typist Methods	*
	 * **************************************** */
	
	public static CD addCD() {
	// this method prompts the typist for information about a CD
	// once the information is obtained, a new CD object is constructed
	// and returned.
	// If any unexpected I/O error occurs and the CD cannot be created,
	// null is returned
		try {		
			System.out.print("What is the title of the CD? ");
			String cdTitle = kb.readLine();
			System.out.print("Who is the artist of the CD? ");
			String cdArtist = kb.readLine();
			if (!getYesResponse("Do you have additional timing info? "))
				return new CD(cdTitle,cdArtist);
				
			// we may assume that the remaining info will be added
			int tracks = getTracks();
			int numMinutes = getMinutes();
			int numSeconds = getSeconds();
			
			return new CD (cdTitle,cdArtist,tracks,numMinutes,numSeconds);
		
		} catch (IOException e) {
			System.out.println("An unhandled I/O exception occurred: " + e);
			System.out.println("No CD was created");
			return null;
		}
	}
	
	public static int getTracks() {
	// Continually asks the typist for the number of tracks
	// until the typist enters an integer greater than zero
	// That integer is returned
		int tracks = -1;
		do {
			System.out.print("How many tracks on the CD? ");
			try {
				tracks = Integer.parseInt(kb.readLine());
				if (tracks <= 0) 
					System.out.println("The number of tracks must be at least 1");
			}
			catch (IOException e) {
				System.out.println("General I/O error: " + e);
			}
			catch (NumberFormatException nfe) {
				System.out.println("The number of tracks must be an int");
			}
		} while (tracks <= 0);
		return tracks;
	}
	
	public static int getMinutes() {
	// Continually asks the typist for the number of run time minutes
	// until the typist enters an integer greater than zero
	// That integer is returned
		int min = -1;
		do {
			System.out.print("What is the total of runtime minutes on the CD? ");
			try {
				min = Integer.parseInt(kb.readLine());
				if (min <= 0) 
					System.out.println("The total runtime minutes must be at least 1");
			}
			catch (IOException e) {
				System.out.println("General I/O error: " + e);
			}
			catch (NumberFormatException nfe) {
				System.out.println("The number of minutes must be an int");
			}
		} while (min <= 0);
		return min;
	}
	
	public static int getSeconds() {
	// Continually asks the typist for the number of run time minutes
	// until the typist enters an integer from 0 up to but not including 60
	// That integer is returned
		int sec = -1;
		do {
			System.out.print("What is the additional number of runtime seconds on the CD? ");
			try {
				sec = Integer.parseInt(kb.readLine());
				if (sec < 0 || sec >= CD.SECONDS_PER_MINUTE) 
					System.out.println("The number of runtime seconds must be between 0 and 60");
			}
			catch (IOException e) {
				System.out.println("General I/O error: " + e);
			}
			catch (NumberFormatException nfe) {
				System.out.println("The number of seconds must be an int");
			}
		} while (sec <= 0);
		return sec;
	}		
	
	/* ************************************ *
	 * 		Stage 3 File Output Methods		*
	 * ************************************ */
	
	public static void fileUpdate(CDCollection coll) {
	// This method opens a file for output and then writes
	// the contents of thea array arr to that file
	
		PrintStream outStream = 
			getOutputStream("To what file do you want to output the collection? ");
		for (int i = 0; i < coll.getSize(); i++)
			outStream.println(coll.get(i));
	}
	
	public static PrintStream getOutputStream(String prompt) {
	// This method continually attempts to obtain a file that can
	// be used for output. It returns a new print stream object
	// that can overwrite the file.
	// The method does not verify that the user wants to overwrite an existing file
	
		FileOutputStream fos = null;
		// the creation of the file output stream is where the 
		// the i/o exception error can occur. The loop below
		// uses the fact that assigning fis a value has not
		// been successful.
		
		do {
			System.out.print (prompt);

			try {
				String fileName = kb.readLine();
				fos = new FileOutputStream(new File(fileName));
			} 
			catch (FileNotFoundException fnne) {
				System.out.println(fnne);
				System.out.println("Either the file you entered already exists and cannot be overwritten");
				System.out.println("or you cannot create a file in the specified location.");
			}
			 catch (IOException ioe) {
				System.out.println("General I/O Error: " + ioe);
			}
			
		} while (fos == null);
		
		return new PrintStream (fos);	
	}
	
	/* **************************** *
	 * 		Other Methods			*
	 * **************************** */
	public static boolean getYesResponse (String prompt) {
	// This method continually asks the typist for a Yes/No response
	// prompting with the string provided until the typist enters a string
	// beginning with the letter y or n (case doesn't matter)
	// The method returns true iff the result was y
		String response = "";
		boolean isValid = false;
		
		do { 
			System.out.print(prompt);
			try {
				response = kb.readLine();
				
				if (response.toUpperCase().charAt(0) != 'Y' &&
					 	response.toUpperCase().charAt(0) != 'N')
					System.out.println("You must type a Y or an N"); 
				else
					isValid = true;
			}
			catch (IOException e) {
				System.out.println("General I/O error: " + e);
			}
			catch (StringIndexOutOfBoundsException e) {
				System.out.println("You must type something");
			}
		}while (!isValid);
		
		return response.toUpperCase().charAt(0) == 'Y';
	}	
}
