/*
	This class represents a "grid" game object

	Written by Jeff Carroll 11/14/03
	Modified 11/15/04
	
*/

import java.util.*;
public class GridGame {

	//default gameboard size
	private static final int DEFAULT_ROWS = 5,
							 DEFAULT_COLS = 5,
							 DEFAULT_TOKENS = 7;
	
	// Maximum gameboard size:
	// my game display method is not general enough to
	// correctly display information about a game grid
	// with multi-digit rows and columns						 
	private static final int MAX_ROWS = 10,
							 MAX_COLS = 10;
							 
							 
	// This constant to limit the maximum number of "treasure"
	// tokens per grid
	private static final double MAX_FRACTION_TOKENS = 0.25;


	public static final char EMPTY = ' ',
							  EXAMINED = '.',
					          HIDDEN = 'H',
					          EXPOSED = '$';

	/* *********************************
		Instance Variables
	   ********************************* */

	private char[][] board;
	private int numRows = DEFAULT_ROWS,
			    numCols = DEFAULT_COLS,
			    numTokens = DEFAULT_TOKENS,
				score = 0,
				numFound = 0;


	/* *********************************
		Constructors
		(this game class has tons of constructors)
	   ********************************* */

	public GridGame () {
	// The default constructor calls the general constructor
	// using the default game size.
	// However,  a fixed game will be played;
	// 		Testing the correctness of games is more difficult to do 
	// 		when the game outcomes are not known ahead of time.
	// 		Creating a class for the game that exists in this mode
	//      corresponds to what has been called "Wizard Mode"

		this (DEFAULT_ROWS,DEFAULT_COLS,0);
		initializeFixed();
	}

	public GridGame (int numRows, int numCols, int numTreasures) {
	// constructs a game with the requested gameboard size & treasures
	//   (as long as the gameboard is no greater than 10x10)
	//   (and as long as the fraction of treasure spaces is no more than
	//   25% of the total number of spaces.
	// If any of these requirements is not met, a game will be created
	// that is as large as allowable in that criteria.

		if (numRows > MAX_ROWS)
			numRows = MAX_ROWS;
		if (numCols > MAX_COLS)
			numCols = MAX_COLS;
		board = new char [numRows][numCols];
		this.numRows = numRows;
		this.numCols = numCols;
		
		if ( (double)numTreasures/(numRows*numCols) > MAX_FRACTION_TOKENS)
			numTreasures = (int) MAX_FRACTION_TOKENS*numRows*numCols;
		numTokens = numTreasures;
		
		clear();		
		setupRandom();
	}
	
	public GridGame (int numTreasures) {
		this (DEFAULT_ROWS,DEFAULT_COLS,numTreasures);
	}

	public GridGame (int numRows, int numCols) {
		this (numRows,numCols,DEFAULT_TOKENS);
	}	

	/* *********************************
		Helper Methods
	    ********************************* */

	private void setupRandom() {
	// places the appropriate number of treasures randomly on the
	// game board.
		Random rng = new Random();
		int count = numTokens;
		while (count > 0) {
			int r = rng.nextInt(numRows);
			int c = rng.nextInt(numCols);
			if (board[r][c] != HIDDEN) {
				board[r][c] = HIDDEN;
				count--;
			}
		}
	}

	private void clear() {
	// clears the gameboard of all markers
		for (int r = 0; r < numRows; r++)
			for (int c = 0; c < numCols; c++)
				board[r][c] = EMPTY;
	}
	
	private void initializeFixed() {
	// used for debugging only:
	// puts game markers into the specific locations
		board[0][0] = HIDDEN;
		board[1][1] = HIDDEN;
		board[2][2] = HIDDEN;
		board[3][3] = HIDDEN;
		board[4][4] = HIDDEN;
		numTokens = 5;
	}

	private int rowMarkerCount( int row ) {
	// An example of row processing
		int count = 0;
		for (int c = 0; c < numCols; c++)
			if (board[row][c] == HIDDEN) // || board[row][c] == EXPOSED)
				count++;
		return count;
	}

	private int colMarkerCount( int col ) {
	// An example of column processing
		int count = 0;
		for (int r = 0; r < numRows; r++)
			if (board[r][col] == HIDDEN) // || board[r][col] == EXPOSED)
				count++;
		return count;
	}

	/* *********************************
		Accessor Methods
		(more to be written)
	   ********************************* */

	public char get (int row, int col) {
		if (row >= numRows || row < 0 ||
			col >= numCols || col < 0) 
			return EMPTY;
		return board[row][col];		
	}

	public int getScore() {
	// returns the current score
		return score;
	}
	
	public boolean isOver() {
	// returns true iff all the hidden treasure has been found
		return numFound == numTokens;
	}
	   
	/* *********************************
		Mutator Methods
	   ********************************* */	   

	public boolean makeGuess(int row, int col) {
	// returns true iff the guess corresponds to a hidden treasure;
	// in this case the score remains unchanged.
	// If the location (row,col) is invalid, if the location
	// is empty or if the location is an already uncovered treasure
	// the score is increased and false is returned.
	
		if (row >= numRows || row < 0 ||
			col >= numCols || col < 0) {
			// the guess was not part of the guess game board
			score++;
			return false;
		}
		if (board[row][col] == HIDDEN) {
			board[row][col] = EXPOSED;
			numFound++;
			return true;
		}
		// it was empty or already exposed
		score++;
		if (board[row][col] == EMPTY)
			board[row][col] = EXAMINED;
		return false;	
	}

	/* ********************************************* *
	 *  This last method provides a textual display of the game. 
	 *  This one method replaces the two methods in the original
	 *
	 *  However, it is less a part of the game itself than a 
	 *  part of display interface
	 * ********************************************* */

	public String display(boolean showHidden) {
	// returns a 2-D textual display of the
	// current state of the game.
	// if showHidden is true, the exact state of the game is displayed
	// if showHidden is false, hidden treasures will not be exposed
	
		String result = "";

		// Displayed above the rows:
		result += "    ";
		for (int c = 0; c < numCols; c++)
			result += "| <" + c + "> ";
		result += "|\n";
		
		for (int r = 0; r < numRows; r++) {
			// Display of row separator 
			result += "   -";
			for (int c = 0; c < numCols;c++)
				result += "+-----";
			result +="+-\n";

			// Displayed ahead of a row 
			result += "<" + r + "> ";
			
			// Displayed in the row
			for (int c = 0; c < numCols; c++)
				if (board[r][c] != HIDDEN || showHidden)
					result += "|  " + board[r][c] + "  ";
				else
					result += "|  " + EMPTY + "  ";

			// Displayed after a row
			result += "| = " + rowMarkerCount(r) +" \n";
		}
		// Display of final row separator 
		result += "   -";
		for (int c = 0; c < numCols;c++)
			result += "+-----";
		result +="+-\n";		
		
		// Displayed below the rows:
		result += "    ";
		for (int c = 0; c < numCols; c++)
			result += "| = "+ colMarkerCount(c) + " ";
		result += "|\n";
		result += "Treasures Found: " + numFound + 
				" Treasures to find: " + (numTokens-numFound) +
				" \nScore: " + score + "\n";
		return result;
	}

}

