/*
	For this final version of Fraction, we added an example of
	
	a) a static variable (& corresponding static method) to keep
	   track of the number of objects created over time
	   
	b) adding an instance variable (in conjunction with the static variable)
	   that creates a unique id for every objec
	   
    Although there is really no need to do this for the Fraction class,
    it shows how it could be done for a class that needs some sort of 
    unique identification 

*/

public class Fraction {

	// This approach is different from what we did in class.
	// The advantage is that the exceptions that we generate have
	// a unique name (instead of being instances of the class Exception
	//
	// However, this approach is associated with inheritance; I don't want to take
	// the time now to explain why this class is written as it is written
	//
	// This is also the first time you will see what is called a nested class
	 
	public class IllegalFractionException extends Exception { 
	
	     public IllegalFractionException () {} 
	     public IllegalFractionException (String reason) { 
	           super(reason); 
	    } 
	}
	 

	// instance variables	
	private int numerator;
	private int denominator = 1;
	private int id;

	// as a static variable, the counter is associated with the
	// class itself (rather than objects of the class). This variable
	// exists and has a value even if no objects are created. The
	// instance variables will never exist if no objects are created; 
	// on the other hand, if n objects are created, ther will be n
	// occurrences of each instance variable.  Yet still there will
	// be only one occurrence of counter.
	private static int counter = 0;

	/* *************************************************** *
			Constructors
	 * *************************************************** */
	public Fraction (int numerator, int denom) throws IllegalFractionException {
	// the most general constructor

	// When the parameter name matches the instance variable
	// name, the variable with the smaller scope is assumed.
	// The scope of the parameter is this method. Hence, when
	// we want to refer to the instance variable 'numerator'
	// we must qualify it with the reserved word 'this'
		this.numerator = numerator;
		if (denom != 0)
			denominator = denom;
		else
			throw new IllegalFractionException ("Can't create a fraction with a denominator of zero");
		counter++;
		id = counter;
	}
	
	public Fraction (int num) {
	// if one parameter is provided, the constructor
	// creates the fraction num / 1
		numerator = num;
		denominator = 1;
		counter++;
		id = counter;
	}
	
	public Fraction ()  {
	// default constructor that produces 1 / 1
		this (1);
	}
	
	
	/* *************************************************** *
			Accessors
	 * *************************************************** */
	public int getNumerator(){
		return numerator;
	}
	
	public int getDenominator(){
		return denominator;
	}
		
	public double getDecimalValue() {
		return (double) numerator / denominator;
	}
	
	public int getID() {
		return id;
	}
	
	public String toString () {
	// This method overrides the default 'toString' method
	// because it has the same signature
		return "#" + id + ": " + numerator + "/" + denominator;
	}

	public boolean equals (Object obj) {
	// This method overrides the default 'equals'
	// However, we must cast the object obj as a fraction
	// in order to access its numerator & denominator
		Fraction f = (Fraction) obj;

		return numerator*f.denominator == f.numerator*denominator;
	}
	
	/* *************************************************** *
			Mutators
	 * *************************************************** */
	 
	public void setNumerator ( int toValue) {
		numerator = toValue;
	}

	public void setDenominator ( int toValue) throws IllegalFractionException {
		if (toValue != 0)
			denominator = toValue;
		else
			throw new IllegalFractionException ("Can't set the denominator of a fraction to zero");
	}	

	public void changeToLowestTerms () {
	// This method changes the current fraction so that it
	// is in lowest terms
		int divisor = gcd(numerator,denominator);
		numerator /= divisor;
		denominator /= divisor;
		
	}

	
	/* *************************************************** *
			Static Methods
	 * *************************************************** */
	
	private static int bruteGCD (int n, int m) {
	// This method is a brute force determination of the
	// greatest common divisor of n and m
	// When n is large, this can be much slower compared
	// to the 
		int guess = m;
		if (n < m)
			guess = n;
		while (m%guess != 0 || n%guess != 0)
			guess--;
		return guess;
	}
	
	private static int gcd (int n, int m) {
	// This method is Euclid's algorithm to determine
	// the greatest common divisor of the ints n and m
	// it is faster than the brute force method
	
		if (m < n) { 
		// swap the roles of m and n (m must be the larger)
			int temp = m;
			m = n;
			n = temp;
		}
		
		// Find the remainder
		int r = m % n;
		
		// Euclid's Theorem states that gcd(m,n) = gcd (n,r)
		// (assuming m > n)
		
		while (r != 0) {
		
			// first, apply Euclid's theorem
			m = n;
			n = r;
			
			r = m % n;
		}
		// when r = 0, the previous remainder (namely n)
		// is the gcd.
		return n;
	}	
	
	public static Fraction toLowestTerms (Fraction f) {
		int divisor = gcd (f.numerator,f.denominator);
		try {
			return new Fraction (f.numerator/divisor, f.denominator/divisor);
		} catch (IllegalFractionException e) {
			System.out.println ("Mathematics is inconsistent");
			return null;
		}
	}

	public static int getObjectsCreated() {
		return counter;
	}

}