﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using System.Drawing;

namespace GOTCHA_Challenge
{
    static class GOTCHA_Challenge
    {
        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        [STAThread]
        static void Main()
        {
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);
            // We used the GUI form Create_Account to generate
            // the inputs to storePassword(int pwd, string [] labels)
            Application.Run(new Create_Account());
            // The user would use the following GUI form to Authenticate
            Application.Run(new Authenticate());

        }
        /// <summary>
        /// Generates n Inkblot puzzles of using the password pwd 
        /// as a source of randomness.
        /// <param name="pwd">The password provides the source of 
        /// randomness.</param>
        /// <param name="width">Width of the Inkblot Puzzle 
        /// (use width = 500)</param>
        /// <param name="height">Height of the Inkblot Puzzle 
        /// (use height = 500)</param>
        /// <param name="n">Number of Inkblot Puzzles to Generate 
        /// (use n = 10)</param>
        /// <returns>An array of Inkblot images</returns>
        /// </summary>
        public static Bitmap[] generatePuzzles(int pwd, int width=500, int height=500, int n=10)
        {
            Random r = new Random(pwd);
            Bitmap[] inkblots = new Bitmap[n];
            for (int i = 0; i < n; i++)
            {
                inkblots[i] = generateInkblot(width, height, r);
            }

            return inkblots;
        }
        /// <summary>
        /// Verifies whether or not the password pwd is correct.
        /// <param name="pwd">The password to verify</param>
        /// <param name="permutation">The solution to the Inkblot 
        /// matching challenge. </param>
        /// <param name="pwdHashBCrypt">The BCRYPT hash value of the original 
        /// password and permutation</param>
        /// <returns>True if pwd is correct, otherwise false.</returns>
        /// </summary>
        public static Boolean verifyPassword(int pwd, int[] permutation, string pwdHashBCrypt)
        {
            string password = ""+ pwd;
            for (int i = 0; i < permutation.Length; i++)
            {
                password += permutation[i];
            }
            return BCrypt.CheckPassword(password, pwdHashBCrypt);
        }
        /// <summary>
        /// Hashes the password pwd. Before the password is hashed a random permutation
        /// is generated and appended to the password. The hash value is stored in the 
        /// returned value along with the labels (in permuted order).
        /// <param name="pwd">The password</param>
        /// <param name="labels"> The human's labels for the Inkblot images returned 
        /// by generatePuzzles(500,500,10,pwd)</param>
        /// <returns>
        /// A string s, whose first line contains the BCRYPT hash of pwd with the 
        /// permutation appended. The labels are stored on the following lines 
        /// in permuted order.
        /// </returns>
        /// </summary>
        public static string storePassword(int pwd, string [] labels)
        {
            Random r = new Random();
            string password = "" + pwd;
            int[] permute = new int[labels.Length];
            for (int j = 0; j < labels.Length; j++) permute[j] = j;
            permute = permute.OrderBy(x => r.Next()).ToList().ToArray();

            string retValue = "";

            for (int i = 0; i < labels.Length; i++)
            {
                password += permute[i];
                retValue += "\n"+ labels[permute[i]];
            }

            return BCrypt.HashPassword(password, BCrypt.GenerateSalt(15))+retValue;
        }
        /// <summary>
        /// Returns a random Inkblot Image with the specified width and height
        /// </summary>
        /// <param name="width">the width of the Inkblot</param>
        /// <param name="height">the height of the Inkblot</param>
        /// <param name="r">source of randomness</param>
        /// <returns></returns>
        private static Bitmap generateInkblot(int width, int height, Random r)
        {
            Bitmap b = new Bitmap(width, height);
            System.Drawing.Drawing2D.Matrix oMatrix = new System.Drawing.Drawing2D.Matrix();

            Graphics g = Graphics.FromImage(b);
            g.Clear(Color.White);
            int numBigBlots = 40;
            System.Drawing.Brush[] brushes = new System.Drawing.Brush[] { Brushes.Black, Brushes.Purple, Brushes.Green, Brushes.Red, Brushes.Yellow, Brushes.Pink, Brushes.DeepPink, Brushes.DarkRed, Brushes.LightBlue, Brushes.Blue, Brushes.Orange };


            for (int i = 0; i < numBigBlots; i++)
            {
                int x = r.Next(0, width);
                int y = r.Next(0, height);
                int x1 = r.Next(0, width);
                int y1 = r.Next(0, height);
                System.Drawing.Brush brush  = brushes[r.Next(0, brushes.Length - 1)];

                //Rotate text Random

                g.FillEllipse(brush, x, y, 60, 60);

                g.FillEllipse(brush, width - x - 60, y, 60, 60);
                if (i < 20)
                {
                    int angle = r.Next(-45, 45);
                    oMatrix.Reset();
                    brush = brushes[r.Next(0, brushes.Length - 1)];
                    oMatrix.RotateAt(angle, new PointF(x + 30, y + 10 - 15));
                    g.Transform = oMatrix;
                    g.FillEllipse(brush, x, y + 10, 60, 30);

                    oMatrix.Reset();
                    oMatrix.RotateAt(-angle, new PointF(width - x - 30, y + 10 - 15));
                    g.Transform = oMatrix;
                    g.FillEllipse(brush, width - x - 60, y + 10, 60, 30);
                    g.ResetTransform();

                    brush = brushes[r.Next(0, brushes.Length - 1)];
                    g.FillEllipse(brush, x1, y1, 20, 20);
                    g.FillEllipse(brush, width - x1 - 20, y1, 20, 20);
                }

            }

            return b;
        }


    }
}
