-- Author: Mario Dehesa Azuara (mdehazu@gmail.com)

open import Context
open import Environment
open import NatUtil
open import Potential
open import Syntax

open import Data.Bool using (true;false)
open import Data.Nat using (ℕ;_≤_;_+_;less-than-or-equal;_≤?_)
open import Data.Nat.Properties using (n≤m+n;m≤m+n;≤⇒≤″;≰⇒>;≤″⇒≤)
open import Data.Nat.Properties.Simple using (+-assoc;+-comm;+-right-identity)
open import Data.List using (_∷_;_++_) renaming ([] to ∅)
open import Data.Product using (proj₁;proj₂;_,_;_×_;Σ)

open import Relation.Nullary using (Dec;yes;no)
import Relation.Binary.PropositionalEquality as PropEq
open PropEq using (_≡_;refl;subst₂;cong;sym;cong-app;subst;cong₂;trans)
open PropEq.≡-Reasoning


import Data.Nat as Nat
open Nat.≤-Reasoning
  renaming (begin_ to start_; _∎ to _□; _≡⟨_⟩_ to _≡⟨_⟩'_)



module Soundness(metric : Oper → ℕ)(Δ : Sig) where
    open import BigStep metric Δ
    open import Statics metric Δ
    open import Preservation metric Δ

    prod-eq₆ : ∀ {A B C D E F : Set}
                 {a₁ a₂ : A} {b₁ b₂ : B} {c₁ c₂ : C}
                 {d₁ d₂ : D} {e₁ e₂ : E} {f₁ f₂ : F}
               → ((a₁ , b₁ , c₁ , d₁) , e₁ , f₁) ≡ ((a₂ , b₂ , c₂ , d₂) , e₂ , f₂)
               → (a₁ ≡ a₂) × (b₁ ≡ b₂) × (c₁ ≡ c₂)
               × (d₁ ≡ d₂) × (e₁ ≡ e₂) × (f₁ ≡ f₂)
    prod-eq₆ refl = refl , refl , refl , refl , refl , refl

    soundness : ∀ (Γ : Ctx)(Ω : Env)(ok : EnvOK Ω Γ)
                  (E : Exp)(V : Val)(T : Tp){q q' p p' : ℕ}
                  (DS : Γ ⊢ E ∶ T ↓〈 q , q' 〉)
                  (DE : Ω ⊢ E ⇓ V ∣〈 p , p' 〉)
                  (sok : SigOK Δ)
                → ∀ (k z : ℕ)
                  → z + pot Γ ok + q ≤ k
                  → Σ ℕ (λ k' → Ω ⊢ E ⇓ V ∣〈 k , k' 〉 ×
                                z + pot-val (preservation DS DE ok sok) + q' ≤ k')

    soundness .((v , T) ∷ ∅) Ω ok .(var v) .(proj₁ V) T
              (of/var v .T q q' x) (step/var .Ω .v V p p' x₁) sok k z L =
       let
         Γ' = (v , T) ∷ ∅
         vok = preservation (of/var v T q q' x) (step/var Ω v V p p' x₁) ok sok

         less-than-or-equal {f} eq = ≤⇒≤″ L
         k' = z + pot Γ' ok + f + q'

         -- Prove that our k can pay for the cost of the var
         H₀ : k ≡ k' + metric var
         H₀ = rearrange-eq₁ k q z f q' (metric var) (pot Γ' ok) (sym eq) x

         H₁ : pot-val vok ≡ pot Γ' ok
         H₁ = pot-sing-eq ok vok (sym (proj₂ V))

         -- Prove that our k' has enough final potential
         step₀ = rearrange-le₁ z (pot-val vok) q' f
         step₁ = cong (λ x → (z + x + f) + q') H₁

         H₂ : (z + pot-val vok) + q' ≤ k'
         H₂ = start
           (z + pot-val vok) + q'              ≤⟨ step₀ ⟩
           (z + pot-val vok + f) + q'          ≡⟨ step₁ ⟩'
           z + pot Γ' ok + f + q'              ≡⟨ refl ⟩'
           k'                                  □

       in (k' , step/var Ω v V k k' H₀ , H₂)
    soundness .∅ Ω ok/nil .true .true .bool
              (of/true q q' x) (step/true .Ω p p' x₁) sok k z L =
      let
        vok = preservation (of/true q q' x) (step/true Ω p p' x₁) ok/nil sok
        less-than-or-equal {f} eq = ≤⇒≤″ L
        k' = z + pot {Ω} ∅ ok/nil + f + q'

        H₀ : k ≡ k' + metric true
        H₀ = rearrange-eq₁ k q z f q' (metric true) (pot {Ω} ∅ ok/nil) (sym eq) x

        H₁ : z + pot-val vok + q' ≤ k'
        H₁ = rearrange-le₁ z (pot-val vok) q' f
      in (k' , step/true Ω k k' H₀ , H₁)
    soundness .∅ Ω ok/nil .false .false .bool
             (of/false q q' x) (step/false .Ω p p' x₁) sok k z L =
      let
        vok = preservation (of/false q q' x) (step/false Ω p p' x₁) ok/nil sok
        less-than-or-equal {f} eq = ≤⇒≤″ L
        k' = z + pot {Ω} ∅ ok/nil + f + q'

        H₀ : k ≡ k' + metric false
        H₀ = rearrange-eq₁ k q z f q' (metric false) (pot {Ω} ∅ ok/nil) (sym eq) x

        H₁ : z + pot-val vok + q' ≤ k'
        H₁ = rearrange-le₁ z (pot-val vok) q' f
      in (k' , step/false Ω k k' H₀ , H₁)
    soundness .((x , bool) ∷ Γ) Ω (ok/cons eq vok ok) .(if x E₁ E₂) V T
              (of/if Γ x E₁ E₂ .T q₁ q₂ q q' DF x₁ DS₁ x₂ DS₂)
              (step/if₁ .Ω .x .E₁ .E₂ .V q₃ p p' Ω' DE x₄) sok k z L =
      let
        Γ' = ((x , bool) ∷ Γ)
        ok' = (ok/cons eq vok ok)
        vok' = preservation (of/if Γ x E₁ E₂ T q₁ q₂ q q' DF x₁ DS₁ x₂ DS₂)
                            (step/if₁ Ω x E₁ E₂ V q₃ p p' Ω' DE x₄)
                            ok'
                            sok
        IH : ∀ (n c : ℕ)
             → c + pot Γ ok + q₁ ≤ n
             → Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V ∣〈 n , n' 〉 × c + pot-val vok' + q' ≤ n')
        IH = soundness Γ Ω ok E₁ V T DS₁ DE sok

        less-than-or-equal {f} eq = ≤⇒≤″ L
        k₁ = z + (pot Γ ok) + f + q₁

        H₀ : Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V ∣〈 k₁ , n' 〉 × z + pot-val vok' + q' ≤ n')
        H₀ = IH k₁ z (rearrange-le₁ z (pot Γ ok) q₁ f)
        k' = proj₁ H₀

        H₁ : k ≡ z + pot Γ ok + q + f
        H₁ = trans (sym eq) (cong (λ x → z + x + q + f) (pot-bool-ctx {Γ} vok))

        H₂ : k ≡ k₁ + metric if₁
        H₂ = rearrange-eq₁ k q z f q₁ (metric if₁) (pot Γ ok) H₁ x₁
      in (k' ,
          step/if₁ Ω x E₁ E₂ V k₁ k k' Ω' (proj₁ (proj₂ H₀)) H₂ ,
          proj₂ (proj₂ H₀))
    soundness .((x , bool) ∷ Γ) Ω (ok/cons eq vok ok) .(if x E₁ E₂) V T
              (of/if Γ x E₁ E₂ .T q₁ q₂ q q' x₁ DF DS₁ x₂ DS₂)
              (step/if₂ .Ω .x .E₁ .E₂ .V q₃ p p' Ω' DE x₄)
              sok k z L =
      let
        Γ' = ((x , bool) ∷ Γ)
        ok' = (ok/cons eq vok ok)
        vok' = preservation (of/if Γ x E₁ E₂ T q₁ q₂ q q' x₁ DF DS₁ x₂ DS₂)
                            (step/if₂ Ω x E₁ E₂ V q₃ p p' Ω' DE x₄)
                            ok'
                            sok
        IH : ∀ (n c : ℕ)
             → c + pot Γ ok + q₂ ≤ n
             → Σ ℕ (λ n' → Ω ⊢ E₂ ⇓ V ∣〈 n , n' 〉 × c + pot-val vok' + q' ≤ n')
        IH = soundness Γ Ω ok E₂ V T DS₂ DE sok

        less-than-or-equal {f} eq = ≤⇒≤″ L
        k₁ = z + (pot Γ ok) + f + q₂

        H₀ : Σ ℕ (λ n' → Ω ⊢ E₂ ⇓ V ∣〈 k₁ , n' 〉 × z + pot-val vok' + q' ≤ n')
        H₀ = IH k₁ z (rearrange-le₁ z (pot Γ ok) q₂ f)
        k' = proj₁ H₀

        H₁ : k ≡ z + pot Γ ok + q + f
        H₁ = trans (sym eq) (cong (λ x → z + x + q + f) (pot-bool-ctx {Γ} vok))

        H₂ : k ≡ k₁ + metric if₂
        H₂ = rearrange-eq₁ k q z f q₂ (metric if₂) (pot Γ ok) H₁ x₂
      in (k' ,
          step/if₂ Ω x E₁ E₂ V k₁ k k' Ω' (proj₁ (proj₂ H₀)) H₂ ,
          proj₂ (proj₂ H₀))
    soundness .(Γ₁ ++ Γ₂) Ω ok .(elet E₁ (v , E₂)) V T
              (of/let Γ₁ Γ₂ E₁ E₂ v T₁ .T q₁ q₂ q q' DF DS₁ x DS)
              (step/let .Ω .E₁ .E₂ .v V₁ .V p₁ p₂ p p' DE₁ Ω' DE x₁)
              sok k z L =
      let
        Ω'' = proj₁ Ω'

        DF' : fresh v Γ₂
        DF' = fresh-++₂ v Γ₁ Γ₂ DF

        ok₁ : EnvOK Ω Γ₁
        ok₁ = ok-append₁ Γ₁ Γ₂ ok

        ok₂ : EnvOK Ω Γ₂
        ok₂ = ok-append₂ Γ₁ Γ₂ ok

        vok₁ : ValOK V₁ T₁
        vok₁ = preservation DS₁ DE₁ ok₁ sok

        ok-ex : EnvOK (v , V₁ ∣ Ω) Γ₂
        ok-ex = ok-extend Γ₂ v ok₂ DF'

        ok-ex' : EnvOK Ω'' Γ₂
        ok-ex' = subst₂ EnvOK (sym (proj₂ Ω')) refl ok-ex

        ok₂' : EnvOK Ω'' ((v , T₁) ∷ Γ₂)
        ok₂' = ok/cons (trans (cong-app (proj₂ Ω') v) (env-extend v))
                       vok₁
                       ok-ex'

        vok : ValOK V T
        vok = preservation (of/let Γ₁ Γ₂ E₁ E₂ v T₁ T q₁ q₂ q q' DF DS₁ x DS)
                           (step/let Ω E₁ E₂ v V₁ V p₁ p₂ p p' DE₁ Ω' DE x₁)
                           ok
                           sok

        IH₁ : ∀ (n z : ℕ)
                → z + pot Γ₁ ok₁ + q₁ ≤ n
                → Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V₁ ∣〈 n , n' 〉 × z + pot-val vok₁ + q₂ ≤ n')
        IH₁ = soundness Γ₁ Ω ok₁ E₁ V₁ T₁ DS₁ DE₁ sok

        IH₂ : ∀ (m z : ℕ)
              → z + (pot ((v , T₁) ∷ Γ₂) ok₂') + q₂ ≤ m
              → Σ ℕ (λ m' → Ω'' ⊢ E₂ ⇓ V ∣〈 m , m' 〉 × z + pot-val vok + q' ≤ m')
        IH₂ = soundness ((v , T₁) ∷ Γ₂) Ω'' ok₂' E₂ V T DS DE sok

        -- Get the difference between k and q
        less-than-or-equal {f} eq = ≤⇒≤″ L
        -- We need the resources left after applying IH₁ to be enough to apply IH₂
        -- which is why we want to add Φ Ω Γ₂ constant potential to the derivation
        k₁ = (z + pot Γ₂ ok₂) + (pot Γ₁ ok₁) + f + q₁

        -- Prove our choice of k₁ satisfies the conditions of IH₁
        H₀ : (z + pot Γ₂ ok₂) + (pot Γ₁ ok₁) + q₁ ≤ k₁
        H₀ = rearrange-le₁ (z + pot Γ₂ ok₂) (pot Γ₁ ok₁) q₁ f

        H₁ : Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V₁ ∣〈 k₁ , n' 〉 ×
                        (z + pot Γ₂ ok₂) + pot-val vok₁ + q₂ ≤ n')
        H₁ = IH₁ k₁ (z + pot Γ₂ ok₂) H₀
        k₂ = (proj₁ H₁)

        -- Prove that our k₂ satisfies the condition to apply IH₂
        step₀ = cong (λ x → z + x + q₂) (+-comm (pot-val vok₁) (pot Γ₂ ok-ex'))
        step₁ = cong (λ x → x + q₂) (sym (+-assoc z (pot Γ₂ ok-ex') (pot-val vok₁)))

        pot-eq₀ : pot Γ₂ ok-ex' ≡ pot Γ₂ ok-ex
        pot-eq₀ = pot-env-eq Γ₂ ok-ex' ok-ex (proj₂ Ω')

        pot-eq₁ : pot Γ₂ ok-ex ≡ pot Γ₂ ok₂
        pot-eq₁ = (sym (pot-extend-fresh {v} Γ₂ ok₂ ok-ex DF'))

        step₂ = cong (λ x → (z + x) + pot-val vok₁ + q₂) (trans pot-eq₀ pot-eq₁)

        H₂ : z + (pot ((v , T₁) ∷ Γ₂) ok₂') + q₂ ≤ k₂
        H₂ = start
           z + (pot ((v , T₁) ∷ Γ₂) ok₂') + q₂      ≡⟨ step₀ ⟩'
           z + (pot Γ₂ ok-ex' + pot-val vok₁) + q₂  ≡⟨ step₁ ⟩'
           (z + pot Γ₂ ok-ex') + pot-val vok₁ + q₂  ≡⟨ step₂ ⟩'
           (z + pot Γ₂ ok₂) + pot-val vok₁ + q₂     ≤⟨ proj₂ (proj₂ H₁) ⟩
           k₂                                     □

        H₃ : Σ ℕ (λ m' → proj₁ Ω' ⊢ E₂ ⇓ V ∣〈 k₂ , m' 〉 × z + pot-val vok + q' ≤ m')
        H₃ = IH₂ k₂ z H₂
        k' = proj₁ H₃

        -- Prove k is k₁ without metric elet potential
        step₀ = rearrange-eq₁ k q z f q₁ (metric elet) (pot (Γ₁ ++ Γ₂) ok) (sym eq) x
        step₁ = cong (λ x → z + x + f + q₁ + metric elet)
                     (trans (pot-concat Γ₁ Γ₂ ok ok₁ ok₂)
                            (+-comm (pot Γ₁ ok₁) (pot Γ₂ ok₂)))

        step₂ = cong (λ x → x + f + q₁ + metric elet)
                     (sym (+-assoc z (pot Γ₂ ok₂) (pot Γ₁ ok₁)))

        H₄ : k ≡ k₁ + metric elet
        H₄ =
          begin
            k                                                    ≡⟨ step₀ ⟩
            z + pot (Γ₁ ++ Γ₂) ok + f + q₁ + metric elet         ≡⟨ step₁ ⟩
            z + (pot Γ₂ ok₂ + pot Γ₁ ok₁) + f + q₁ + metric elet ≡⟨ step₂ ⟩
            z + pot Γ₂ ok₂ + pot Γ₁ ok₁ + f + q₁ + metric elet   ≡⟨ refl ⟩
            k₁ + metric elet                                     ∎
      in ( k' ,
          step/let Ω E₁ E₂ v V₁ V k₁ k₂ k k'
                   (proj₁ (proj₂ H₁)) Ω' (proj₁ (proj₂ H₃)) H₄ ,
          proj₂ (proj₂ H₃))
    soundness .∅ Ω ok/nil .nil .nil .(list m T)
              (of/nil T m q q' x) (step/nil .Ω p p' x₁) sok k z L =
      let
        less-than-or-equal {f} eq = ≤⇒≤″ L
        k' = z + pot {Ω} ∅ ok/nil + f + q'

        H₀ : k ≡ k' + metric nil
        H₀ = rearrange-eq₁ k q z f q' (metric nil) (pot {Ω} ∅ ok/nil) (sym eq) x

        H₁ : z + pot-val {nil} {list m T} vok/nil + q' ≤ k'
        H₁ = rearrange-le₁ z (pot-val {nil} {list m T} vok/nil) q' f
      in (k' , step/nil Ω k k' H₀ , H₁)
    soundness .((v₁ , T) ∷ (v₂ , list m T) ∷ ∅) Ω
              (ok/cons eqh vokh (ok/cons eqt vokt ok/nil))
              .(cons v₁ v₂)
              .(cons (proj₁ V) (proj₁ VS)) .(list m T)
              (of/cons v₁ v₂ T m q q' x)
              (step/cons .Ω .v₁ .v₂ p p' x₁ V VS) sok k z L =
      let
        Γ = (v₁ , T) ∷ (v₂ , list m T) ∷ ∅

        ok' : EnvOK Ω Γ
        ok' = (ok/cons eqh vokh (ok/cons eqt vokt ok/nil))

        vok : ValOK (cons (proj₁ V) (proj₁ VS)) (list m T)
        vok = preservation (of/cons v₁ v₂ T m q q' x)
                           (step/cons Ω v₁ v₂ p p' x₁ V VS)
                           ok'
                           sok

        less-than-or-equal {f} eq = ≤⇒≤″ L
        k' = z + pot-val vok + f + q'

        -- Prove that k ≡ k' + metric cons
        H₀ : k ≡ z + pot Γ ok' + f + q' + (m + metric cons)
        H₀ = rearrange-eq₁ k q z f q'
                          (m + metric cons)
                          (pot Γ ok')
                          (sym eq)
                          x

        H₁ : k ≡ z + (pot Γ ok' + m) + f + q' + metric cons
        H₁ = trans H₀ (rearrange-eq₂ z (pot Γ ok') f q' m (metric cons))

        step₀ = cong (λ x → (pot-val vokh + x) + m) (+-right-identity (pot-val vokt))
        step₁ = pot-cons
                 vokh
                 vokt
                 vok
                 (trans (sym eqh) (sym (proj₂ V)))
                 (trans (sym eqt) (sym (proj₂ VS)))

        H₂ : pot Γ ok' + m ≡ pot-val vok
        H₂ =
          begin
            pot Γ ok' + m                            ≡⟨ refl ⟩
            (pot-val vokh + (pot-val vokt + 0)) + m  ≡⟨ step₀ ⟩
            pot-val vokh + pot-val vokt + m          ≡⟨ step₁ ⟩
            pot-val vok                              ∎

        H₃ : k ≡ k' + metric cons
        H₃ = trans H₁ (cong (λ x → z + x + f + q' + metric cons) H₂)

        -- Prove:  z + (Φ-term (cons (proj₁ V) (proj₁ VS)) (list m T)) + q' ≤ k'
        H₄ : z + pot-val vok + q' ≤ k'
        H₄ = rearrange-le₁ z (pot-val vok) q' f

      in (k' , step/cons Ω v₁ v₂ k k' H₃ V VS , H₄)
    soundness .((v , list m T) ∷ Γ) Ω (ok/cons eqv vok ok)
              .(matl v E₁ (v₁ , v₂ , E₂)) V T₁
              (of/matl Γ v v₁ v₂ E₁ E₂ T .T₁ q q' q₁ q₂ m DF DF₁ DF₂ x DS₁ x₁ DS₂)
              (step/matl₁ .Ω .v .v₁ .v₂ .E₁ .E₂ .V p₁ p p' x₂ x₃ DE) sok k z L =
      let
        ok' = ok/cons eqv vok ok
        vok' = preservation
                (of/matl Γ v v₁ v₂ E₁ E₂ T T₁ q q' q₁ q₂ m DF DF₁ DF₂ x DS₁ x₁ DS₂)
                (step/matl₁ Ω v v₁ v₂ E₁ E₂ V p₁ p p' x₂ x₃ DE)
                ok'
                sok

        Γ' = (v , list m T) ∷ Γ
        less-than-or-equal {f} eq = ≤⇒≤″ L
        k₁ = z + pot Γ' ok' + f + q₁

        H₀ : k ≡ k₁ + metric matl₁
        H₀ = rearrange-eq₁ k q z f q₁
                           (metric matl₁)
                           (pot Γ' ok')
                           (sym eq)
                           x

        -- Proving that k₁ is sufficient to invoke the IH
        step₀ = m≤m+n (z + pot Γ ok + q₁) (pot-val vok + f)
        step₁ = sym (rearrange-eq₃ z (pot-val vok) (pot Γ ok) f q₁)

        H₂ : z + pot Γ ok + q₁ ≤ k₁
        H₂ =
          start
            z + pot Γ ok + q₁                     ≤⟨ step₀ ⟩
            z + pot Γ ok + q₁ + (pot-val vok + f) ≡⟨ step₁ ⟩'
            k₁                                    □

        IH : Σ ℕ (λ k' → (Ω ⊢ E₁ ⇓ V ∣〈 k₁ , k' 〉 × z + pot-val vok' + q' ≤ k' ))
        IH = soundness Γ Ω ok E₁ V T₁ DS₁ DE sok k₁ z H₂

        k' = proj₁ IH
      in (k' ,
          step/matl₁ Ω v v₁ v₂ E₁ E₂ V k₁ k k' x₂ H₀ (proj₁ (proj₂ IH)) ,
          proj₂ (proj₂ IH))
    soundness .((v , list m T) ∷ Γ) Ω (ok/cons eql vok ok)
              .(matl v E₁ (x , xs , E₂)) V T₁
              (of/matl Γ v x xs E₁ E₂ T .T₁ q q' q₁ q₂ m DF DF₁ DF₂ eq₀ DS eq₁ DS₁)
              (step/matl₂ .Ω .v .x .xs .E₁ .E₂ X XS .V q₃ p p' eq₂ H DE eq₃) sok
              k z L =
      let
        (vokh , vokt) = vok-cons vok (trans (sym eql) eq₂)
        Ω' = proj₁ H
        cost = metric matl₂

        less-than-or-equal {f} eq = ≤⇒≤″ L

        ok' : EnvOK Ω ((v , list m T) ∷ Γ)
        ok' = ok/cons eql vok ok

        vok' : ValOK V T₁
        vok' = preservation
                (of/matl Γ v x xs E₁ E₂ T T₁ q q' q₁ q₂ m DF DF₁ DF₂ eq₀ DS eq₁ DS₁)
                (step/matl₂ Ω v x xs E₁ E₂ X XS V q₃ p p' eq₂ H DE eq₃)
                ok'
                sok

        ok-ex₁ : EnvOK (xs , XS ∣ Ω) Γ
        ok-ex₁ = ok-extend Γ xs ok (fresh-wkn DF₂)

        ok-ex₂ : EnvOK (x , X ∣ (xs , XS ∣ Ω)) Γ
        ok-ex₂ = ok-extend Γ x ok-ex₁ (fresh-wkn DF₁)

        ok-ex : EnvOK Ω' Γ
        ok-ex = subst₂ EnvOK (sym (proj₂ H)) refl ok-ex₂

        Γ' : Ctx
        Γ' = (x , T) ∷ (xs , list m T) ∷ Γ

        P = pot-val vokh + pot-val vokt + pot Γ ok

        H₁ : (x , X ∣ (xs , XS ∣ Ω)) x ≡ X
        H₁ = env-extend x

        H₂ : (x , X ∣ (xs , XS ∣ Ω)) xs ≡ XS
        H₂ = env-extend₂ x xs DF₂

        H₃ : Ω' x ≡ X
        H₃ = trans (cong (λ y → y x) (proj₂ H)) H₁

        vok₁ : ValOK (Ω' x) T
        vok₁ = subst₂ ValOK (sym H₃) refl vokh

        H₄ : Ω' xs ≡ XS
        H₄ = trans (cong (λ y → y xs) (proj₂ H)) H₂

        vok₂ : ValOK (Ω' xs) (list m T)
        vok₂ = subst₂ ValOK (sym H₄) refl vokt

        H₅ : pot Γ ok-ex₂ ≡ pot Γ ok-ex
        H₅ = pot-env-eq Γ ok-ex₂ ok-ex (sym (proj₂ H))

        H₆ : pot Γ ok-ex₁ ≡ pot Γ ok-ex₂
        H₆ = pot-extend-fresh Γ ok-ex₁ ok-ex₂ (fresh-wkn DF₁)

        H₇ : pot Γ ok ≡ pot Γ ok-ex₁
        H₇ = pot-extend-fresh Γ ok ok-ex₁ (fresh-wkn DF₂)

        k₂ = z + (pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex) + f + q₂

        -- Show that k ≡ k₂ + metric matl₂
        step₀ = cong₂ (λ a b → z + (a + b) + f + q₂ + metric matl₂)
                      (cong₂ (λ a b → a + b)
                             (pot-val-eq vok₁ vokh H₃)
                             (pot-val-eq vok₂ vokt H₄))
                      (trans (sym H₅) (trans (sym H₆) (sym H₇)))
        step₁ = +-assoc (z + P + f) q₂ (metric matl₂)
        step₂ = cong (λ a → (z + P + f) + a) (sym eq₁)
        step₃ = rearrange-eq₅ z (pot-val vokh + pot-val vokt) (pot Γ ok) f q m
        step₄ = cong (λ a → z + (a + pot Γ ok) + f + q)
                     (pot-cons₂ vokh vokt vok (trans (sym eq₂) eql))
        step₅ = +-assoc-comm (z + pot ((v , list m T) ∷ Γ) ok') f q
        J₁ =
          begin
            k₂ + metric matl₂                                            ≡⟨ refl ⟩
            (z + (pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex) + f + q₂) + metric matl₂
                                                                         ≡⟨ step₀ ⟩
            (z + (pot-val vokh + pot-val vokt + pot Γ ok) + f + q₂) + metric matl₂
                                                                         ≡⟨ step₁ ⟩
            (z + (pot-val vokh + pot-val vokt + pot Γ ok) + f) + (q₂ + metric matl₂)
                                                                         ≡⟨ step₂ ⟩
            (z + (pot-val vokh + pot-val vokt + pot Γ ok) + f) + (q + m) ≡⟨ step₃ ⟩
            (z + (pot-val vokh + pot-val vokt + m + pot Γ ok) + f) + q   ≡⟨ step₄ ⟩
            (z + pot ((v , list m T) ∷ Γ) ok' + f) + q                   ≡⟨ step₅ ⟩
            (z + pot ((v , list m T) ∷ Γ) ok' + q) + f                   ≡⟨ eq ⟩
            k                                                            ∎

        -- Show that k₂ can be used for our IH
        step₀ = cong (λ a → z + a + q₂)
                     (sym (+-assoc (pot-val vok₁) (pot-val vok₂) (pot Γ ok-ex)))
        step₁ = rearrange-le₁ z (pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex) q₂ f

        M₁ =
         start
          z + (pot-val vok₁ + (pot-val vok₂ + pot Γ ok-ex)) + q₂ ≡⟨ step₀ ⟩'
          z + (pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex) + q₂   ≤⟨ step₁ ⟩
          k₂                                                     □

        -- Done.
        ok' : EnvOK Ω' Γ'
        ok' = ok/cons refl vok₁ (ok/cons refl vok₂ ok-ex)


        resvok : ValOK V T₁
        resvok = preservation DS₁ DE ok' sok

        IH : Σ ℕ (λ k' → (proj₁ H ⊢ E₂ ⇓ V ∣〈 k₂ , k' 〉) ×
                         z + pot-val resvok + q' ≤ k')
        IH = soundness Γ' Ω' ok' E₂ V T₁ DS₁ DE sok k₂ z M₁
        k' = proj₁ IH

        -- Since the proof that the resulting value is well typed, produced by
        -- preservation, depends on Ω' and not Ω we need to show that these proofs
        -- each produce the same potential
        step₀ = cong (λ a → z + a + q') (pot-val-eq vok' resvok refl)
        step₁ = proj₂ (proj₂ IH)
        M₂ : z + pot-val vok' + q' ≤ k'
        M₂ = start
            z + pot-val vok' + q'   ≡⟨ step₀ ⟩'
            z + pot-val resvok + q' ≤⟨ step₁ ⟩
            k'                      □
      in (k' ,
          step/matl₂ Ω v x xs E₁ E₂ X XS V k₂ k k' eq₂ H (proj₁ (proj₂ IH)) (sym J₁) ,
          M₂)
    soundness .((x , T) ∷ Γ) .Ω (ok/cons xeq xvok ok) .(share x (x₁ , x₂ , E)) .V₁ .M
              (of/share Γ x x₁ x₂ E T T₁ T₂ M q q' frsh frsh₁ frsh₂ shareok DS)
              (step/share Ω .x .x₁ .x₂ .E V V₁ p p' eq Ω' DE) sok k z L =
      let
         Γ' = (x₁ , T₁) ∷ (x₂ , T₂) ∷ Γ
         less-than-or-equal {f} eq' = ≤⇒≤″ L

         origok : EnvOK Ω ((x , T) ∷ Γ)
         origok = ok/cons xeq xvok ok

         vok' = preservation
                  (of/share Γ x x₁ x₂ E T T₁ T₂ M q q' frsh frsh₁ frsh₂ shareok DS)
                  (step/share Ω x x₁ x₂ E V V₁ p p' eq Ω' DE)
                  origok
                  sok

         -- Prove that the new environment is okay under the new context.
         ok-ex₁ : EnvOK (x₂ , V ∣ Ω) Γ
         ok-ex₁ = ok-extend Γ x₂ ok (fresh-wkn frsh₂)

         ok-ex₂ : EnvOK (x₁ , V ∣ (x₂ , V ∣ Ω)) Γ
         ok-ex₂ = ok-extend Γ x₁ ok-ex₁ (fresh-wkn frsh₁)

         ok-ex : EnvOK (proj₁ Ω') Γ
         ok-ex = subst₂ EnvOK (sym (proj₂ Ω')) refl ok-ex₂

         env-app : ∀ (v : Var){val : Val} →
                     (x₁ , V ∣ (x₂ , V ∣ Ω)) v ≡ val →
                     (proj₁ Ω') v ≡ val
         env-app v eq = trans (cong-app (proj₂ Ω') v) eq

         H₀ : (proj₁ Ω') x₁ ≡ V
         H₀ = env-app x₁ (env-extend x₁)

         H₁ : (proj₁ Ω') x₂ ≡ V
         H₁ = env-app x₂ (env-extend₂ x₁ x₂ frsh₂)

         vok₁ : ValOK V T₁
         vok₁ = subst₂ ValOK (trans (sym xeq) eq) refl (ok-share₁ shareok xvok)

         vok₂ : ValOK V T₂
         vok₂ = subst₂ ValOK (trans (sym xeq) eq) refl (ok-share₂ shareok xvok)

         ok' : EnvOK (proj₁ Ω') Γ'
         ok' = ok/cons H₀ vok₁ (ok/cons H₁ vok₂ ok-ex)

         H₂ : pot Γ ok ≡ pot Γ ok-ex₁
         H₂ = pot-extend-fresh Γ ok ok-ex₁ (fresh-wkn frsh₂)

         H₃ : pot Γ ok-ex₁ ≡ pot Γ ok-ex₂
         H₃ = pot-extend-fresh Γ ok-ex₁ ok-ex₂ (fresh-wkn frsh₁)

         H₄ : pot Γ ok-ex ≡ pot Γ ok-ex₂
         H₄ = pot-env-eq Γ ok-ex ok-ex₂ (proj₂ Ω')

         -- Prove that the new context has the same amount of potential
         H₅ : pot Γ ok-ex ≡ pot Γ ok
         H₅ = trans H₄ (trans (sym H₃) (sym H₂))

         H₆ : pot-val vok₁ + pot-val vok₂ ≡ pot-val xvok
         H₆ = pot-share shareok xvok vok₁ vok₂ (trans (sym xeq) eq)

         step₀ = sym (+-assoc (pot-val vok₁) (pot-val vok₂) (pot Γ ok-ex))
         step₁ = cong₂ (λ a b → a + b) H₆ H₅

         H₇ : pot Γ' ok' ≡ pot ((x , T) ∷ Γ) origok
         H₇ = begin
            pot Γ' ok'                                  ≡⟨ refl ⟩
            pot-val vok₁ + (pot-val vok₂ + pot Γ ok-ex) ≡⟨ step₀ ⟩
            pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex   ≡⟨ step₁ ⟩
            pot-val xvok + pot Γ ok                     ≡⟨ refl ⟩
            pot ((x , T) ∷ Γ) origok                    ∎

         H₈ : z + pot Γ' ok' + q ≤ k
         H₈ = start
          z + pot Γ' ok' + q                ≡⟨ cong (λ a → z + a + q) H₇ ⟩'
          z + pot ((x , T) ∷ Γ) origok + q  ≤⟨ L ⟩
          k                                 □

         resok : ValOK V₁ M
         resok = preservation DS DE ok' sok

         IH : Σ ℕ (λ k' → (proj₁ Ω') ⊢ E ⇓ V₁ ∣〈 k , k' 〉 × z + pot-val resok + q' ≤ k')
         IH = soundness Γ' (proj₁ Ω') ok' E V₁ M DS DE sok k z H₈

         k' = proj₁ IH

         step₀ = cong (λ a → z + a + q') (pot-val-eq vok' resok refl)
         step₁ = proj₂ (proj₂ IH)
         H₉ : z + pot-val vok' + q' ≤ k'
         H₉ = start
             z + pot-val vok' + q'   ≡⟨ step₀ ⟩'
             z + pot-val resok + q'  ≤⟨ step₁ ⟩
             k'                      □

      in  (k' , step/share Ω x x₁ x₂ E V V₁ k k' eq Ω' (proj₁ (proj₂ IH)) , H₉)
    soundness .((x , T₁) ∷ ∅) Ω (ok/cons refl vok ok/nil) .(app fd x) V₂ T₂
              (of/app x x₁ T₁ .T₂ fd q q₁ q' E eq eq₂)
              (step/app .Ω .fd .x x₂ V₁ .V₂ E₁ p p₁ p' T₃ T₄ j j' eq₃ eq₄ eq₅ DE)
              sok k z L =
      let
        resok = preservation
                  (of/app x x₁ T₁ T₂ fd q q₁ q' E eq eq₂)
                  (step/app Ω fd x x₂ V₁ V₂ E₁ p p₁ p' T₃ T₄ j j' eq₃ eq₄ eq₅ DE)
                  (ok/cons refl vok ok/nil)
                  sok

        less-than-or-equal {f} eq₆ = ≤⇒≤″ L
        k₁ = z + (pot-val vok + 0) + f + q₁

        H₀ : ((T₁ , q₁ , q' , T₂) , x₁ , E) ≡ ((T₃ , j , j' , T₄) , x₂ , E₁)
        H₀ = trans (sym eq) eq₄

        (eqT₁T₃ , eqq₁j , eqq'j' , eqT₂T₄ , eqx₁x₂ , eqEE₁ ) = prod-eq₆ H₀

        H₁ : ValOK V₁ T₃
        H₁ = subst₂ ValOK eq₃ eqT₁T₃ vok

        step₀ = cong₂ (λ a b → z + a + b)
                      (cong (λ a → a + 0) (sym (pot-val-eq₂ vok H₁ eq₃ eqT₁T₃)))
                      (sym eqq₁j)
        step₁ = m≤m+n (z + (pot-val vok + 0) + q₁) f
        step₂ = +-assoc-comm (z + (pot-val vok + 0)) q₁ f

        H₂ = start
          z + (pot-val H₁ + 0) + j        ≡⟨ step₀ ⟩'
          z + (pot-val vok + 0) + q₁      ≤⟨ step₁ ⟩
          z + (pot-val vok + 0) + q₁ + f  ≡⟨ step₂ ⟩'
          z + (pot-val vok + 0) + f + q₁  ≡⟨ refl ⟩'
          k₁                              □


        H₃ : EnvOK ((x₂ , V₁ ∣ ◎)) ((x₂ , T₃) ∷ ∅)
        H₃ = ok/cons (env-extend x₂) H₁ ok/nil

        resok' = preservation (sok eq₄) DE H₃ sok

        IH : Σ ℕ (λ k' → ((x₂ , V₁ ∣ ◎) ⊢ E₁ ⇓ V₂ ∣〈 k₁ , k' 〉) ×
                          z + pot-val resok' + j' ≤ k')
        IH = soundness ((x₂ , T₃) ∷ ∅) ((x₂ , V₁ ∣ ◎)) H₃
                       E₁ V₂ T₄ (sok eq₄) DE sok k₁ z H₂

        step₀ = +-assoc-comm (z + (pot-val vok + 0)) q f
        step₁ = cong (λ a → z + (pot-val vok + 0) + f + a) eq₂
        step₂ = sym (+-assoc (z + (pot-val vok + 0) + f) q₁ (metric app))
        H₄ : k ≡ k₁ + metric app
        H₄ = begin
          k                                             ≡⟨ sym eq₆ ⟩
          z + (pot-val vok + 0) + q + f                 ≡⟨ step₀ ⟩
          z + (pot-val vok + 0) + f + q                 ≡⟨ step₁ ⟩
          z + (pot-val vok + 0) + f + (q₁ + metric app) ≡⟨ step₂ ⟩
          z + (pot-val vok + 0) + f + q₁ + metric app   ≡⟨ refl ⟩
          k₁ + metric app                               ∎

        k' = proj₁ IH

        H₅ : z + pot-val resok + q' ≤ k'
        H₅ = subst₂ (λ a b → z + a + b ≤ k')
                    (pot-val-eq₂ resok' resok refl (sym eqT₂T₄))
                    (sym eqq'j')
                    (proj₂ (proj₂ IH))

      in k' ,
         step/app Ω fd x x₂ V₁ V₂ E₁ k k₁ k' T₃
                  T₄ j j' eq₃ eq₄ H₄ (proj₁ (proj₂ IH))  ,
         H₅
    soundness .((v , T') ∷ Γ) Ω (ok/cons eq vok ok) E V T
              (of/wkn Γ .E .T q q' v T' DF DS) DE sok k z L =
     let
         resok = preservation DS DE ok sok
         IH : ∀ (n c : ℕ)
              → c + (pot Γ ok) + q ≤ n
              → Σ ℕ (λ n' → Ω ⊢ E ⇓ V ∣〈 n , n' 〉 × c + pot-val resok + q' ≤ n')
         IH = soundness Γ Ω ok E V T DS DE sok

         step₀ = m≤m+n (z + (pot Γ ok) + q) (pot-val vok)
         step₁ = +-assoc-comm (z + (pot Γ ok)) q (pot-val vok)
         step₂ = cong (λ y → y + q) (+-assoc z (pot Γ ok) (pot-val vok))
         step₃ = cong (λ y → z + y + q) (+-comm (pot Γ ok) (pot-val vok))

         H₀ : z + pot Γ ok + q ≤ k
         H₀ = start
            z + (pot Γ ok) + q                 ≤⟨ step₀ ⟩
            z + (pot Γ ok) + q + (pot-val vok) ≡⟨ step₁ ⟩'
            z + (pot Γ ok) + (pot-val vok) + q ≡⟨ step₂ ⟩'
            z + ((pot Γ ok) + pot-val vok) + q ≡⟨ step₃ ⟩'
            z + (pot-val vok + pot Γ ok) + q   ≤⟨ L ⟩
            k                                  □

       in IH k z H₀
    soundness Γ Ω ok E V T (of/rlx .Γ .E .T p p' q q' DS x x₁) DE sok k z L 
      with q' ≤? p'
    ... | (yes le) =
      let
        H₀ : z + pot Γ ok + p ≤ k
        H₀ = start
          z + pot Γ ok + p ≤⟨ p≤q⇒n+p≤n+q (z + pot Γ ok) x ⟩
          z + pot Γ ok + q ≤⟨ L ⟩
          k                □

        vok = preservation (of/rlx Γ E T p p' q q' DS x x₁) DE ok sok

        IH : Σ ℕ (λ n' → Ω ⊢ E ⇓ V ∣〈 k , n' 〉 × z + pot-val vok + p' ≤ n')
        IH = soundness Γ Ω ok E V T DS DE sok k z H₀
        k' = proj₁ IH

        H₁ = start
          z + pot-val vok + q' ≤⟨ p≤q⇒n+p≤n+q (z + pot-val vok) le ⟩
          z + pot-val vok + p' ≤⟨ proj₂ (proj₂ IH) ⟩
          k'                  □

      in (k' , proj₁ (proj₂ IH) , H₁)
    ... | (no nle) =
      let
        less-than-or-equal {f} p+f≡q = ≤⇒≤″ x
        less-than-or-equal {f'} 1+p'+f'≡q' = ≤⇒≤″ (≰⇒> nle)
        less-than-or-equal {j} p+q'+j≡q+p' = ≤⇒≤″ x₁
        vok = preservation (of/rlx Γ E T p p' q q' DS x x₁) DE ok sok

        -- Here we prove that f' ≤ f
        step₁ = cong (λ y → p + y) (sym (+-assoc p' 1 (f' + j)))
        step₂ = cong (λ y → p + (y + (f' + j))) (+-comm p' 1)
        step₃ = cong (λ y → p + y) (sym (+-assoc (1 + p') f' j))

        H₀ : (p + p') + (1 + (f' + j)) ≡ p + p' + f
        H₀ = begin
          (p + p') + (1 + (f' + j)) ≡⟨ +-assoc p p' (1 + (f' + j)) ⟩
          p + (p' + (1 + (f' + j))) ≡⟨ step₁ ⟩
          p + ((p' + 1) + (f' + j)) ≡⟨ step₂ ⟩
          p + ((1 + p') + (f' + j)) ≡⟨ step₃ ⟩
          p + (((1 + p') + f') + j) ≡⟨ sym (+-assoc p ((1 + p') + f') j) ⟩
          (p + (1 + p' + f')) + j   ≡⟨ cong (λ y → p + y + j) 1+p'+f'≡q' ⟩
          p + q' + j                ≡⟨ p+q'+j≡q+p' ⟩
          q + p'                    ≡⟨ cong (λ y → y + p') (sym p+f≡q) ⟩
          p + f + p'                ≡⟨ +-assoc-comm p f p' ⟩
          p + p' + f                ∎

        H₁ : (1 + (f' + j)) ≡ f
        H₁ = n+m≡n+k⇒m≡k (p + p') (1 + (f' + j)) f H₀

        f'' = 1 + f'

        H₂ : (1 + f') + j ≡ f
        H₂ = begin
           (1 + f') + j ≡⟨ +-assoc 1 f' j ⟩
           1 + (f' + j) ≡⟨ H₁ ⟩
           f            ∎

        H₃ : f'' ≤ f
        H₃ = ≤″⇒≤ (less-than-or-equal H₂)

        step₂ = cong (λ y → f'' + y) (sym (+-assoc z (pot Γ ok) p))
        H₄ : (f'' +  z) + pot Γ ok + p ≤ k
        H₄ = start
           (f'' +  z) + pot Γ ok + p  ≡⟨ +-assoc (f'' + z) (pot Γ ok) p ⟩'
           (f'' + z) + (pot Γ ok + p) ≡⟨ +-assoc f'' z (pot Γ ok + p) ⟩'
           f'' + (z + (pot Γ ok + p)) ≡⟨ step₂ ⟩'
           f'' + (z + pot Γ ok + p)   ≤⟨ n≤m⇒n+k≤m+k f'' f (z + pot Γ ok + p) H₃ ⟩
           f + (z + pot Γ ok + p)     ≡⟨ +-comm f (z + pot Γ ok + p) ⟩'
           z + pot Γ ok + p + f       ≡⟨ +-assoc (z + pot Γ ok) p f ⟩'
           z + pot Γ ok + (p + f)     ≡⟨ cong (λ y → z + pot Γ ok + y) p+f≡q ⟩'
           z + pot Γ ok + q           ≤⟨ L ⟩
           k                          □

        IH : Σ ℕ (λ n' → Ω ⊢ E ⇓ V ∣〈 k , n' 〉 × (f'' + z) + pot-val vok + p' ≤ n')
        IH = soundness Γ Ω ok E V T DS DE sok k (f'' + z) H₄
        k' = proj₁ IH

        step₀ = cong (λ y → z + pot-val vok + y) (sym 1+p'+f'≡q')
        step₁ = cong (λ y → z + pot-val vok + (y + f')) (+-comm 1 p')
        step₂ = cong (λ y → z + pot-val vok + y) (+-assoc p' 1 f')
        step₅ = cong (λ y → y + f'') (+-assoc z (pot-val vok) p')
        step₈ = sym (+-assoc (f'' + z) (pot-val vok) p')

        H₅ : z + pot-val vok + q' ≤ k'
        H₅ = start
          z + pot-val vok + q'               ≡⟨ step₀ ⟩'
          z + pot-val vok + (1 + p' + f')    ≡⟨ step₁ ⟩'
          z + pot-val vok + ((p' +  1) + f') ≡⟨ step₂ ⟩'
          z + pot-val vok + (p' + (1 + f'))  ≡⟨ refl ⟩'
          z + pot-val vok + (p' + f'')       ≡⟨ sym (+-assoc (z + pot-val vok) p' f'') ⟩'
          z + pot-val vok + p' + f''         ≡⟨ step₅ ⟩'
          (z + (pot-val vok + p')) + f''     ≡⟨ +-comm (z + (pot-val vok + p')) f'' ⟩'
          f'' + (z + (pot-val vok + p'))     ≡⟨ sym (+-assoc f'' z (pot-val vok + p')) ⟩'
          (f'' + z) + (pot-val vok + p')     ≡⟨ step₈ ⟩'
          (f'' + z) + pot-val vok + p'       ≤⟨ proj₂ (proj₂ IH) ⟩
          k'                                 □

      in (k' , proj₁ (proj₂ IH) , H₅ )
    soundness .(Γ₂ ++ Γ₁) Ω ok E V T (of/exg Γ₁ Γ₂ .E .T q q' DS) DE sok k z L =
      let
        vok = preservation (of/exg Γ₁ Γ₂ E T q q' DS) DE ok sok
        okexg = ok-exg Γ₂ Γ₁ ok
        ok₁ : EnvOK Ω Γ₁
        ok₁ = ok-append₂ Γ₂ Γ₁ ok

        ok₂ : EnvOK Ω Γ₂
        ok₂ = ok-append₁ Γ₂ Γ₁ ok

        step₀ = cong (λ y → z + y + q) (pot-concat Γ₁ Γ₂ okexg ok₁ ok₂)
        step₁ = cong (λ y → z + y + q) (sym (+-comm (pot Γ₂ ok₂) (pot Γ₁ ok₁)))
        step₂ = cong (λ y → z + y + q) (sym (pot-concat Γ₂ Γ₁ ok ok₂ ok₁))

        H₀ : z + pot (Γ₁ ++ Γ₂) okexg + q ≤ k
        H₀ = start
          z + pot (Γ₁ ++ Γ₂) okexg + q      ≡⟨ step₀ ⟩'
          z + (pot Γ₁ ok₁ + pot Γ₂ ok₂) + q ≡⟨ step₁ ⟩'
          z + (pot Γ₂ ok₂ + pot Γ₁ ok₁) + q ≡⟨ step₂ ⟩'
          z + pot (Γ₂ ++ Γ₁) ok + q         ≤⟨ L ⟩
          k                                 □

        IH : Σ ℕ (λ k' → (Ω ⊢ E ⇓ V ∣〈 k , k' 〉) × z + pot-val vok + q' ≤ k')
        IH = soundness (Γ₁ ++ Γ₂) Ω okexg E V T DS DE sok k z H₀
      in (proj₁ IH , proj₁ (proj₂ IH) , proj₂ (proj₂ IH))
    soundness Γ Ω ok E V T₂ (of/sub .Γ .E T₁ .T₂ q q' sub DS) DE sok k z L =
      let
        vok = preservation (of/sub Γ E T₁ T₂ q q' sub DS) DE ok sok
        resok = preservation DS DE ok sok

        IH : Σ ℕ (λ k' → (Ω ⊢ E ⇓ V ∣〈 k , k' 〉) ×
                         z + pot-val (preservation DS DE ok sok) + q' ≤ k')
        IH = soundness Γ Ω ok E V T₁ DS DE sok k z L
        k' = proj₁ IH

        step₀ = +-assoc-comm z (pot-val vok) q'
        step₁ = p≤q⇒n+p≤n+q (z + q') (pot-sub resok vok sub)
        step₂ = +-assoc-comm z q' (pot-val resok)
        step₃ = proj₂ (proj₂ IH)

        H₀ = start
          z + pot-val vok + q'   ≡⟨ step₀ ⟩'
          z + q' + pot-val vok   ≤⟨ step₁ ⟩
          z + q' + pot-val resok ≡⟨ step₂ ⟩'
          z + pot-val resok + q' ≤⟨ step₃ ⟩
          k'                     □

      in (k' , proj₁ (proj₂ IH) , H₀)
    soundness .((v , T₁) ∷ Γ) Ω (ok/cons {.(Ω v)} refl vokh₁ okt) E V T₃
              (of/sup v Γ .E T₁ T₂ .T₃ q q' sup DS) DE sok k z L =
      let
        vokh₂ : ValOK (Ω v) T₂
        vokh₂ = ok-sub sup vokh₁

        H₀ : pot-val vokh₂ ≤ pot-val vokh₁
        H₀ = pot-sub vokh₁ vokh₂ sup

        H₁ : pot-val vokh₂ + pot Γ okt ≤ pot-val vokh₁ + pot Γ okt
        H₁ = n≤m⇒n+k≤m+k (pot-val vokh₂) (pot-val vokh₁) (pot Γ okt) H₀

        step₀ = +-assoc-comm z (pot-val vokh₂ + pot Γ okt) q
        step₁ = p≤q⇒n+p≤n+q (z + q) H₁
        step₂ = +-assoc-comm z q (pot-val vokh₁ + pot Γ okt)

        H₂ = start
          z + (pot-val vokh₂ + pot Γ okt) + q ≡⟨ step₀ ⟩'
          z + q + (pot-val vokh₂ + pot Γ okt) ≤⟨ step₁ ⟩
          z + q + (pot-val vokh₁ + pot Γ okt) ≡⟨ step₂ ⟩'
          z + (pot-val vokh₁ + pot Γ okt) + q ≤⟨ L ⟩
          k                                   □

      in soundness ((v , T₂) ∷ Γ) Ω (ok/cons refl vokh₂ okt) E V T₃ DS DE sok k z H₂
