-- Author: Mario Dehesa Azuara (mdehazu@gmail.com)

open import Context
open import Environment
open import NatUtil
open import Potential
open import Syntax

open import Data.Bool using (true;false)
open import Data.Nat using (ℕ;_≤_;_+_;less-than-or-equal;_≤?_)
open import Data.Nat.Properties using (n≤m+n;m≤m+n;≤⇒≤″;≰⇒>;≤″⇒≤)
open import Data.Nat.Properties.Simple using (+-assoc;+-comm;+-right-identity)
open import Data.List using (_∷_;_++_) renaming ([] to ∅)
open import Data.Product using (proj₁;proj₂;_,_;_×_;Σ)

open import Relation.Nullary using (Dec;yes;no)
import Relation.Binary.PropositionalEquality as PropEq
open PropEq using (_≡_;refl;subst₂;cong;sym;cong-app;subst;cong₂;trans)
open PropEq.≡-Reasoning


import Data.Nat as Nat
open Nat.≤-Reasoning
  renaming (begin_ to start_; _∎ to _□; _≡⟨_⟩_ to _≡⟨_⟩'_)


module Soundness(metric : Oper → ℕ)(Δ : Sig) where
    open import BigStep metric Δ
    open import Statics metric Δ
    open import Preservation metric Δ

    prod-eq₆ : ∀ {A B C D E F : Set}
                 {a₁ a₂ : A} {b₁ b₂ : B} {c₁ c₂ : C}
                 {d₁ d₂ : D} {e₁ e₂ : E} {f₁ f₂ : F}
               → ((a₁ , b₁ , c₁ , d₁) , e₁ , f₁) ≡ ((a₂ , b₂ , c₂ , d₂) , e₂ , f₂)
               → (a₁ ≡ a₂) × (b₁ ≡ b₂) × (c₁ ≡ c₂)
               × (d₁ ≡ d₂) × (e₁ ≡ e₂) × (f₁ ≡ f₂)
    prod-eq₆ refl = refl , refl , refl , refl , refl , refl


    soundness : ∀ (Γ : Ctx)(Ω : Env)(ok : EnvOK Ω Γ)
                  (E : Exp)(V : Val)(T : Tp){q q' p p' : ℕ}
                  (DS : Γ ⊢ E ∶ T ↓〈 q , q' 〉)
                  (DE : Ω ⊢ E ⇓ V ∣〈 p , p' 〉)
                  (sok : SigOK Δ)
                → ∀ (k z : ℕ)
                  → z + pot Γ ok + q ≡ k
                  → Σ ℕ (λ k' → Ω ⊢ E ⇓ V ∣〈 k , k' 〉 ×
                                z + pot-val (preservation DS DE ok sok) + q' ≡ k')

    soundness .((v , T) ∷ ∅) Ω ok .(var v) .(proj₁ V) T
              (of/var v .T q q' x) (step/var .Ω .v V p p' x₁) sok k z L =
       let
         Γ' = (v , T) ∷ ∅
         vok = preservation (of/var v T q q' x) (step/var Ω v V p p' x₁) ok sok
         k' = z + pot-val vok + q'

         H₀ : pot-val vok ≡ pot Γ' ok
         H₀ = pot-sing-eq ok vok (sym (proj₂ V))

         -- Prove that our k can pay for the cost of the var
         step₀ = cong₂ (λ a b → z + a + b) (sym H₀) x
         step₁ = sym (+-assoc (z + pot-val vok) q' (metric var))
         H₁ : k ≡ k' + metric var
         H₁ = begin
                k                                   ≡⟨ sym L ⟩
                z + pot Γ' ok + q                   ≡⟨ step₀ ⟩
                z + pot-val vok + (q' + metric var) ≡⟨ step₁ ⟩
                k' + metric var                     ∎
      in (k' , step/var Ω v V k k' H₁ , refl)
    soundness .∅ Ω ok/nil .true .true .bool
              (of/true q q' x) (step/true .Ω p p' x₁) sok k z L =
      let
        vok = preservation (of/true q q' x) (step/true Ω p p' x₁) ok/nil sok
        k' = z + q'

        step₀ = cong (λ a → a + q) (+-right-identity z)
        step₁ = cong (λ a → z + a) x
        step₂ = sym (+-assoc z q' (metric true))
        H₀ : k ≡ k' + metric true
        H₀ = begin
          k                      ≡⟨ sym L ⟩
          z + 0 + q              ≡⟨ step₀ ⟩
          z + q                  ≡⟨ step₁ ⟩
          z + (q' + metric true) ≡⟨ step₂ ⟩
          k' + metric true       ∎

        H₁ : z + pot-val vok + q' ≡ k'
        H₁ = begin
          z + pot-val vok + q'    ≡⟨ refl ⟩
          z + pot-val vok/tt + q' ≡⟨ refl ⟩
          z + 0 + q'              ≡⟨ cong (λ a → a + q') (+-right-identity z) ⟩
          z + q'                  ∎
      in (k' , step/true Ω k k' H₀ , H₁)
    soundness .∅ Ω ok/nil .false .false .bool
             (of/false q q' x) (step/false .Ω p p' x₁) sok k z L =
      let
        vok = preservation (of/false q q' x) (step/false Ω p p' x₁) ok/nil sok
        k' = z + q'

        step₀ = cong (λ a → a + q) (+-right-identity z)
        step₁ = cong (λ a → z + a) x
        step₂ = sym (+-assoc z q' (metric false))
        H₀ : k ≡ k' + metric false
        H₀ = begin
          k                       ≡⟨ sym L ⟩
          z + 0 + q               ≡⟨ step₀ ⟩
          z + q                   ≡⟨ step₁ ⟩
          z + (q' + metric false) ≡⟨ step₂ ⟩
          k' + metric false       ∎

        H₁ : z + pot-val vok + q' ≡ k'
        H₁ = begin
          z + pot-val vok + q'    ≡⟨ refl ⟩
          z + pot-val vok/tt + q' ≡⟨ refl ⟩
          z + 0 + q'              ≡⟨ cong (λ a → a + q') (+-right-identity z) ⟩
          z + q'                  ∎
      in (k' , step/false Ω k k' H₀ , H₁)
    soundness .((x , bool) ∷ Γ) Ω (ok/cons refl vok ok) .(if x E₁ E₂) V T
              (of/if Γ x E₁ E₂ .T q₁ q₂ q q' DF x₁ DS₁ x₂ DS₂)
              (step/if₁ .Ω .x .E₁ .E₂ .V q₃ p p' Ω' DE x₄) sok k z L =
      let
        Γ' = ((x , bool) ∷ Γ)
        ok' = (ok/cons refl vok ok)
        vok' = preservation (of/if Γ x E₁ E₂ T q₁ q₂ q q' DF x₁ DS₁ x₂ DS₂)
                            (step/if₁ Ω x E₁ E₂ V q₃ p p' Ω' DE x₄)
                            ok'
                            sok
        IH : ∀ (n c : ℕ)
             → c + pot Γ ok + q₁ ≡ n
             → Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V ∣〈 n , n' 〉 × c + pot-val vok' + q' ≡ n')
        IH = soundness Γ Ω ok E₁ V T DS₁ DE sok

        k₁ = z + (pot Γ ok) + q₁

        H₀ : Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V ∣〈 k₁ , n' 〉 × z + pot-val vok' + q' ≡ n')
        H₀ = IH k₁ z refl
        k' = proj₁ H₀

        H₁ : pot-val vok + pot Γ ok ≡ pot Γ ok
        H₁ = cong (λ a → a + pot Γ ok) (pot-val-eq vok vok/tt Ω')

        step₂ = sym (+-assoc (z + pot Γ ok) q₁ (metric if₁))
        H₂ : k ≡ k₁ + metric if₁
        H₂ = begin
          k                                ≡⟨ sym L ⟩
          z + (pot-val vok + pot Γ ok) + q ≡⟨ cong (λ a → z + a + q) H₁ ⟩
          z + pot Γ ok + q                 ≡⟨ cong (λ a → z + pot Γ ok + a) x₁ ⟩
          z + pot Γ ok + (q₁ + metric if₁) ≡⟨ step₂ ⟩
          k₁ + metric if₁                  ∎
      in (k' ,
          step/if₁ Ω x E₁ E₂ V k₁ k k' Ω' (proj₁ (proj₂ H₀)) H₂ ,
          proj₂ (proj₂ H₀))
    soundness .((x , bool) ∷ Γ) Ω (ok/cons refl vok ok) .(if x E₁ E₂) V T
              (of/if Γ x E₁ E₂ .T q₁ q₂ q q' x₁ DF DS₁ x₂ DS₂)
              (step/if₂ .Ω .x .E₁ .E₂ .V q₃ p p' Ω' DE x₄) sok k z L =
      let
        Γ' = ((x , bool) ∷ Γ)
        ok' = (ok/cons refl vok ok)
        vok' = preservation (of/if Γ x E₁ E₂ T q₁ q₂ q q' x₁ DF DS₁ x₂ DS₂)
                            (step/if₂ Ω x E₁ E₂ V q₃ p p' Ω' DE x₄)
                            ok'
                            sok
        IH : ∀ (n c : ℕ)
             → c + pot Γ ok + q₂ ≡ n
             → Σ ℕ (λ n' → Ω ⊢ E₂ ⇓ V ∣〈 n , n' 〉 × c + pot-val vok' + q' ≡ n')
        IH = soundness Γ Ω ok E₂ V T DS₂ DE sok

        k₂ = z + (pot Γ ok) + q₂

        H₀ : Σ ℕ (λ n' → Ω ⊢ E₂ ⇓ V ∣〈 k₂ , n' 〉 × z + pot-val vok' + q' ≡ n')
        H₀ = IH k₂ z refl
        k' = proj₁ H₀

        H₁ : pot-val vok + pot Γ ok ≡ pot Γ ok
        H₁ = cong (λ a → a + pot Γ ok) (pot-val-eq vok vok/ff Ω')

        step₂ = sym (+-assoc (z + pot Γ ok) q₂ (metric if₂))
        H₂ : k ≡ k₂ + metric if₂
        H₂ = begin
          k                                ≡⟨ sym L ⟩
          z + (pot-val vok + pot Γ ok) + q ≡⟨ cong (λ a → z + a + q) H₁ ⟩
          z + pot Γ ok + q                 ≡⟨ cong (λ a → z + pot Γ ok + a) x₂ ⟩
          z + pot Γ ok + (q₂ + metric if₂) ≡⟨ step₂ ⟩
          k₂ + metric if₂                  ∎
      in (k' ,
          step/if₂ Ω x E₁ E₂ V k₂ k k' Ω' (proj₁ (proj₂ H₀)) H₂ ,
          proj₂ (proj₂ H₀))
    soundness .(Γ₁ ++ Γ₂) Ω ok .(elet E₁ (v , E₂)) V T
              (of/let Γ₁ Γ₂ E₁ E₂ v T₁ .T q₁ q₂ q q' DF DS₁ x DS)
              (step/let .Ω .E₁ .E₂ .v V₁ .V p₁ p₂ p p' DE₁ Ω' DE x₁) sok k z L =
      let
        Ω'' = proj₁ Ω'

        DF' : fresh v Γ₂
        DF' = fresh-++₂ v Γ₁ Γ₂ DF

        ok₁ : EnvOK Ω Γ₁
        ok₁ = ok-append₁ Γ₁ Γ₂ ok

        ok₂ : EnvOK Ω Γ₂
        ok₂ = ok-append₂ Γ₁ Γ₂ ok

        vok₁ : ValOK V₁ T₁
        vok₁ = preservation DS₁ DE₁ ok₁ sok

        ok-ex : EnvOK (v , V₁ ∣ Ω) Γ₂
        ok-ex = ok-extend Γ₂ v ok₂ DF'

        ok-ex' : EnvOK Ω'' Γ₂
        ok-ex' = subst₂ EnvOK (sym (proj₂ Ω')) refl ok-ex

        ok₂' : EnvOK Ω'' ((v , T₁) ∷ Γ₂)
        ok₂' = ok/cons (trans (cong-app (proj₂ Ω') v) (env-extend v))
                       vok₁
                       ok-ex'

        vok : ValOK V T
        vok = preservation (of/let Γ₁ Γ₂ E₁ E₂ v T₁ T q₁ q₂ q q' DF DS₁ x DS)
                           (step/let Ω E₁ E₂ v V₁ V p₁ p₂ p p' DE₁ Ω' DE x₁)
                           ok
                           sok

        IH₁ : ∀ (n z : ℕ)
                → z + pot Γ₁ ok₁ + q₁ ≡ n
                → Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V₁ ∣〈 n , n' 〉 × z + pot-val vok₁ + q₂ ≡ n')
        IH₁ = soundness Γ₁ Ω ok₁ E₁ V₁ T₁ DS₁ DE₁ sok

        IH₂ : ∀ (m z : ℕ)
              → z + (pot ((v , T₁) ∷ Γ₂) ok₂') + q₂ ≡ m
              → Σ ℕ (λ m' → Ω'' ⊢ E₂ ⇓ V ∣〈 m , m' 〉 × z + pot-val vok + q' ≡ m')
        IH₂ = soundness ((v , T₁) ∷ Γ₂) Ω'' ok₂' E₂ V T DS DE sok

        -- Get the difference between k and q
        -- We need the resources left after applying IH₁ to be enough to apply IH₂
        -- which is why we want to add Φ Ω Γ₂ constant potential to the derivation
        k₁ = (z + pot Γ₂ ok₂) + (pot Γ₁ ok₁) + q₁

        -- Prove our choice of k₁ satisfies the conditions of IH₁
        H₁ : Σ ℕ (λ n' → Ω ⊢ E₁ ⇓ V₁ ∣〈 k₁ , n' 〉 ×
                        (z + pot Γ₂ ok₂) + pot-val vok₁ + q₂ ≡ n')
        H₁ = IH₁ k₁ (z + pot Γ₂ ok₂) refl
        k₂ = (proj₁ H₁)

        -- Prove that our k₂ satisfies the condition to apply IH₂
        step₀ = cong (λ x → z + x + q₂) (+-comm (pot-val vok₁) (pot Γ₂ ok-ex'))
        step₁ = cong (λ x → x + q₂) (sym (+-assoc z (pot Γ₂ ok-ex') (pot-val vok₁)))

        pot-eq₀ : pot Γ₂ ok-ex' ≡ pot Γ₂ ok-ex
        pot-eq₀ = pot-env-eq Γ₂ ok-ex' ok-ex (proj₂ Ω')
        pot-eq₁ : pot Γ₂ ok-ex ≡ pot Γ₂ ok₂
        pot-eq₁ = (sym (pot-extend-fresh {v} Γ₂ ok₂ ok-ex DF'))

        step₂ = cong (λ x → (z + x) + pot-val vok₁ + q₂) (trans pot-eq₀ pot-eq₁)

        H₂ : z + (pot ((v , T₁) ∷ Γ₂) ok₂') + q₂ ≡ k₂
        H₂ = begin
           z + (pot ((v , T₁) ∷ Γ₂) ok₂') + q₂      ≡⟨ step₀ ⟩
           z + (pot Γ₂ ok-ex' + pot-val vok₁) + q₂  ≡⟨ step₁ ⟩
           (z + pot Γ₂ ok-ex') + pot-val vok₁ + q₂  ≡⟨ step₂ ⟩
           (z + pot Γ₂ ok₂) + pot-val vok₁ + q₂     ≡⟨ proj₂ (proj₂ H₁) ⟩
           k₂                                       ∎

        H₃ : Σ ℕ (λ m' → proj₁ Ω' ⊢ E₂ ⇓ V ∣〈 k₂ , m' 〉 × z + pot-val vok + q' ≡ m')
        H₃ = IH₂ k₂ z H₂
        k' = proj₁ H₃

        -- Prove k is k₁ without metric elet potential
        step₀ = cong (λ a → z + pot (Γ₁ ++ Γ₂) ok + a) x
        step₁ = sym (+-assoc (z + pot (Γ₁ ++ Γ₂) ok) q₁ (metric elet))
        step₂ = cong (λ x → z + x + q₁ + metric elet)
                     (trans (pot-concat Γ₁ Γ₂ ok ok₁ ok₂)
                            (+-comm (pot Γ₁ ok₁) (pot Γ₂ ok₂)))

        step₃ = cong (λ x → x + q₁ + metric elet)
                     (sym (+-assoc z (pot Γ₂ ok₂) (pot Γ₁ ok₁)))

        H₄ : k ≡ k₁ + metric elet
        H₄ =
          begin
            k                                                ≡⟨ sym L ⟩
            z + pot (Γ₁ ++ Γ₂) ok + q                        ≡⟨ step₀ ⟩
            z + pot (Γ₁ ++ Γ₂) ok + (q₁ + metric elet)       ≡⟨ step₁ ⟩
            z + pot (Γ₁ ++ Γ₂) ok + q₁ + metric elet         ≡⟨ step₂ ⟩
            z + (pot Γ₂ ok₂ + pot Γ₁ ok₁) + q₁ + metric elet ≡⟨ step₃ ⟩
            z + pot Γ₂ ok₂ + pot Γ₁ ok₁ + q₁ + metric elet   ≡⟨ refl ⟩
            k₁ + metric elet                                 ∎
      in ( k' ,
          step/let Ω E₁ E₂ v V₁ V k₁ k₂ k k'
                   (proj₁ (proj₂ H₁)) Ω' (proj₁ (proj₂ H₃)) H₄ ,
          proj₂ (proj₂ H₃))
    soundness .∅ Ω ok/nil .nil .nil .(list m T)
              (of/nil T m q q' x) (step/nil .Ω p p' x₁) sok k z L =
      let
        k' = z + pot {Ω} ∅ ok/nil + q'

        H₀ : k ≡ k' + metric nil
        H₀ = begin
          k                         ≡⟨ sym L ⟩
          z + 0 + q                 ≡⟨ cong (λ a → z + 0 + a) x ⟩
          z + 0 + (q' + metric nil) ≡⟨ sym (+-assoc (z + 0) q' (metric nil)) ⟩
          k' + metric nil           ∎

      in (k' , step/nil Ω k k' H₀ , refl)
    soundness .((v₁ , T) ∷ (v₂ , list m T) ∷ ∅) Ω
              (ok/cons refl vokh (ok/cons refl vokt ok/nil))
              .(cons v₁ v₂)
              .(cons (proj₁ V) (proj₁ VS)) .(list m T)
              (of/cons v₁ v₂ T m q q' x)
              (step/cons .Ω .v₁ .v₂ p p' x₁ V VS)
              sok k z L =
      let
        Γ = (v₁ , T) ∷ (v₂ , list m T) ∷ ∅

        ok' : EnvOK Ω Γ
        ok' = (ok/cons refl vokh (ok/cons refl vokt ok/nil))

        vok : ValOK (cons (proj₁ V) (proj₁ VS)) (list m T)
        vok = preservation (of/cons v₁ v₂ T m q q' x)
                           (step/cons Ω v₁ v₂ p p' x₁ V VS)
                           ok'
                           sok

        k' = z + pot-val vok + q'

        -- Prove that k ≡ k' + metric cons
        step₀ = cong (λ x → (pot-val vokh + x) + m) (+-right-identity (pot-val vokt))
        step₁ = pot-cons vokh vokt vok (sym (proj₂ V)) (sym (proj₂ VS))

        H₂ : pot Γ ok' + m ≡ pot-val vok
        H₂ =
          begin
            pot Γ ok' + m                            ≡⟨ refl ⟩
            (pot-val vokh + (pot-val vokt + 0)) + m  ≡⟨ step₀ ⟩
            pot-val vokh + pot-val vokt + m          ≡⟨ step₁ ⟩
            pot-val vok                              ∎

        step₀ = cong (λ a → z + pot Γ ok' + a) x
        step₁ = sym (+-assoc (z + pot Γ ok') m (q' + metric cons))
        step₂ = cong (λ a → a + (q' + metric cons)) (+-assoc z (pot Γ ok') m)
        step₃ = sym (+-assoc (z + (pot Γ ok' + m)) q' (metric cons))
        step₄ = cong (λ a → z + a + q' + metric cons) H₂

        H₃ : k ≡ k' + metric cons
        H₃ = begin
          k                                        ≡⟨ sym L ⟩
          z + pot Γ ok' + q                        ≡⟨ step₀ ⟩
          z + pot Γ ok' + (m + (q' + metric cons)) ≡⟨ step₁ ⟩
          z + pot Γ ok' + m + (q' + metric cons)   ≡⟨ step₂ ⟩
          z + (pot Γ ok' + m) + (q' + metric cons) ≡⟨ step₃ ⟩
          z + (pot Γ ok' + m) + q' + metric cons   ≡⟨ step₄ ⟩
          z + pot-val vok + q' + metric cons       ≡⟨ refl ⟩
          k' + metric cons                         ∎
      in (k' , step/cons Ω v₁ v₂ k k' H₃ V VS , refl)
    soundness .((v , list m T) ∷ Γ) Ω (ok/cons refl vok ok)
              .(matl v E₁ (v₁ , v₂ , E₂)) V T₁
              (of/matl Γ v v₁ v₂ E₁ E₂ T .T₁ q q' q₁ q₂ m DF DF₁ DF₂ x DS₁ x₁ DS₂)
              (step/matl₁ .Ω .v .v₁ .v₂ .E₁ .E₂ .V p₁ p p' eqnil x₃ DE) sok k z L =
      let
        ok' = ok/cons refl vok ok
        vok' = preservation
                (of/matl Γ v v₁ v₂ E₁ E₂ T T₁ q q' q₁ q₂ m DF DF₁ DF₂ x DS₁ x₁ DS₂)
                (step/matl₁ Ω v v₁ v₂ E₁ E₂ V p₁ p p' eqnil x₃ DE)
                ok'
                sok

        Γ' = (v , list m T) ∷ Γ
        k₁ = z + pot Γ' ok' + q₁

        H₀ : pot-val vok ≡ 0
        H₀ = pot-val-eq vok vok/nil eqnil

        H₁ : pot Γ' ok' ≡ pot Γ ok
        H₁ = cong (λ a → a + pot Γ ok) H₀

        H₂ : k₁ ≡ z + pot Γ ok + q₁
        H₂ = cong (λ a → z + a + q₁) H₁

        IH : Σ ℕ (λ k' → (Ω ⊢ E₁ ⇓ V ∣〈 k₁ , k' 〉 × z + pot-val vok' + q' ≡ k' ))
        IH = soundness Γ Ω ok E₁ V T₁ DS₁ DE sok k₁ z (sym H₂)

        k' = proj₁ IH

        step₁ = sym (+-assoc (z + pot Γ' ok') q₁ (metric matl₁))

        H₃ : k ≡ k₁ + metric matl₁
        H₃ = begin
          k                                    ≡⟨ sym L ⟩
          z + pot Γ' ok' + q                   ≡⟨ cong (λ a → z + pot Γ' ok' + a) x ⟩
          z + pot Γ' ok' + (q₁ + metric matl₁) ≡⟨ step₁ ⟩
          z + pot Γ' ok' + q₁ + metric matl₁   ≡⟨ refl ⟩
          k₁ + metric matl₁                    ∎

      in (k' ,
          step/matl₁ Ω v v₁ v₂ E₁ E₂ V k₁ k k' eqnil H₃ (proj₁ (proj₂ IH)) ,
          proj₂ (proj₂ IH))
    soundness .((v , list m T) ∷ Γ) Ω (ok/cons refl vok ok)
              .(matl v E₁ (x , xs , E₂)) V T₁
              (of/matl Γ v x xs E₁ E₂ T .T₁ q q' q₁ q₂ m DF DF₁ DF₂ eq₀ DS eq₁ DS₁)
              (step/matl₂ .Ω .v .x .xs .E₁ .E₂ X XS .V q₃ p p' eq₂ H DE eq₃) sok
              k z L =
      let
        (vokh , vokt) = vok-cons vok eq₂
        Ω' = proj₁ H
        cost = metric matl₂

        ok' : EnvOK Ω ((v , list m T) ∷ Γ)
        ok' = ok/cons refl vok ok

        vok' : ValOK V T₁
        vok' = preservation
                (of/matl Γ v x xs E₁ E₂ T T₁ q q' q₁ q₂ m DF DF₁ DF₂ eq₀ DS eq₁ DS₁)
                (step/matl₂ Ω v x xs E₁ E₂ X XS V q₃ p p' eq₂ H DE eq₃)
                ok'
                sok

        ok-ex₁ : EnvOK (xs , XS ∣ Ω) Γ
        ok-ex₁ = ok-extend Γ xs ok (fresh-wkn DF₂)

        ok-ex₂ : EnvOK (x , X ∣ (xs , XS ∣ Ω)) Γ
        ok-ex₂ = ok-extend Γ x ok-ex₁ (fresh-wkn DF₁)

        ok-ex : EnvOK Ω' Γ
        ok-ex = subst₂ EnvOK (sym (proj₂ H)) refl ok-ex₂

        Γ' : Ctx
        Γ' = (x , T) ∷ (xs , list m T) ∷ Γ

        P = pot-val vokh + pot-val vokt + pot Γ ok

        H₁ : (x , X ∣ (xs , XS ∣ Ω)) x ≡ X
        H₁ = env-extend x

        H₂ : (x , X ∣ (xs , XS ∣ Ω)) xs ≡ XS
        H₂ = env-extend₂ x xs DF₂

        H₃ : Ω' x ≡ X
        H₃ = trans (cong (λ y → y x) (proj₂ H)) H₁

        vok₁ : ValOK (Ω' x) T
        vok₁ = subst₂ ValOK (sym H₃) refl vokh

        H₄ : Ω' xs ≡ XS
        H₄ = trans (cong (λ y → y xs) (proj₂ H)) H₂

        vok₂ : ValOK (Ω' xs) (list m T)
        vok₂ = subst₂ ValOK (sym H₄) refl vokt

        H₅ : pot Γ ok-ex₂ ≡ pot Γ ok-ex
        H₅ = pot-env-eq Γ ok-ex₂ ok-ex (sym (proj₂ H))

        H₆ : pot Γ ok-ex₁ ≡ pot Γ ok-ex₂
        H₆ = pot-extend-fresh Γ ok-ex₁ ok-ex₂ (fresh-wkn DF₁)

        H₇ : pot Γ ok ≡ pot Γ ok-ex₁
        H₇ = pot-extend-fresh Γ ok ok-ex₁ (fresh-wkn DF₂)

        k₂ = z + (pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex) + q₂

        -- Show that k ≡ k₂ + metric matl₂
        step₀ = cong₂ (λ a b → z + (a + b) + q₂ + metric matl₂)
                      (cong₂ (λ a b → a + b)
                             (pot-val-eq vok₁ vokh H₃)
                             (pot-val-eq vok₂ vokt H₄))
                      (trans (sym H₅) (trans (sym H₆) (sym H₇)))
        step₁ = +-assoc (z + P) q₂ (metric matl₂)
        step₂ = cong (λ a → (z + P) + a) (sym eq₁)
        step₃ = rearrange-eq₆ z (pot-val vokh + pot-val vokt) (pot Γ ok) q m
        step₄ = cong (λ a → z + (a + pot Γ ok) + q)
                     (pot-cons₂ vokh vokt vok (sym eq₂))
        J₁ =
          begin
            k₂ + metric matl₂                                        ≡⟨ refl ⟩
            (z + (pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex) + q₂) + metric matl₂
                                                                     ≡⟨ step₀ ⟩
            (z + (pot-val vokh + pot-val vokt + pot Γ ok) + q₂) + metric matl₂
                                                                     ≡⟨ step₁ ⟩
            (z + (pot-val vokh + pot-val vokt + pot Γ ok)) + (q₂ + metric matl₂)
                                                                     ≡⟨ step₂ ⟩
            (z + (pot-val vokh + pot-val vokt + pot Γ ok)) + (q + m) ≡⟨ step₃ ⟩
            (z + (pot-val vokh + pot-val vokt + m + pot Γ ok)) + q   ≡⟨ step₄ ⟩
            (z + pot ((v , list m T) ∷ Γ) ok' + q)                   ≡⟨ L ⟩
            k                                                        ∎

        -- Show that k₂ can be used for our IH
        M₁ = cong (λ a → z + a + q₂)
                  (sym (+-assoc (pot-val vok₁) (pot-val vok₂) (pot Γ ok-ex)))
        -- Done.
        ok' : EnvOK Ω' Γ'
        ok' = ok/cons refl vok₁ (ok/cons refl vok₂ ok-ex)


        resvok : ValOK V T₁
        resvok = preservation DS₁ DE ok' sok

        IH : Σ ℕ (λ k' → (proj₁ H ⊢ E₂ ⇓ V ∣〈 k₂ , k' 〉) ×
                         z + pot-val resvok + q' ≡ k')
        IH = soundness Γ' Ω' ok' E₂ V T₁ DS₁ DE sok k₂ z M₁
        k' = proj₁ IH

        -- Since the proof that the resulting value is well typed, produced by
        -- preservation, depends on Ω' and not Ω we need to show that these proofs
        -- each produce the same potential
        step₀ = cong (λ a → z + a + q') (pot-val-eq vok' resvok refl)
        step₁ = proj₂ (proj₂ IH)
        M₂ : z + pot-val vok' + q' ≡ k'
        M₂ = begin
            z + pot-val vok' + q'   ≡⟨ step₀ ⟩
            z + pot-val resvok + q' ≡⟨ step₁ ⟩
            k'                      ∎
      in (k' ,
          step/matl₂ Ω v x xs E₁ E₂ X XS V k₂ k k' eq₂ H (proj₁ (proj₂ IH)) (sym J₁) ,
          M₂)
    soundness .((x , T) ∷ Γ) .Ω (ok/cons xeq xvok ok) .(share x (x₁ , x₂ , E)) .V₁ .M
              (of/share Γ x x₁ x₂ E T T₁ T₂ M q q' frsh frsh₁ frsh₂ shareok DS)
              (step/share Ω .x .x₁ .x₂ .E V V₁ p p' eq Ω' DE) sok k z L =
      let
         Γ' = (x₁ , T₁) ∷ (x₂ , T₂) ∷ Γ

         origok : EnvOK Ω ((x , T) ∷ Γ)
         origok = ok/cons xeq xvok ok

         vok' = preservation
                  (of/share Γ x x₁ x₂ E T T₁ T₂ M q q' frsh frsh₁ frsh₂ shareok DS)
                  (step/share Ω x x₁ x₂ E V V₁ p p' eq Ω' DE)
                  origok
                  sok

         -- Prove that the new environment is okay under the new context.
         ok-ex₁ : EnvOK (x₂ , V ∣ Ω) Γ
         ok-ex₁ = ok-extend Γ x₂ ok (fresh-wkn frsh₂)

         ok-ex₂ : EnvOK (x₁ , V ∣ (x₂ , V ∣ Ω)) Γ
         ok-ex₂ = ok-extend Γ x₁ ok-ex₁ (fresh-wkn frsh₁)

         ok-ex : EnvOK (proj₁ Ω') Γ
         ok-ex = subst₂ EnvOK (sym (proj₂ Ω')) refl ok-ex₂

         env-app : ∀ (v : Var){val : Val} →
                     (x₁ , V ∣ (x₂ , V ∣ Ω)) v ≡ val →
                     (proj₁ Ω') v ≡ val
         env-app v eq = trans (cong-app (proj₂ Ω') v) eq

         H₀ : (proj₁ Ω') x₁ ≡ V
         H₀ = env-app x₁ (env-extend x₁)

         H₁ : (proj₁ Ω') x₂ ≡ V
         H₁ = env-app x₂ (env-extend₂ x₁ x₂ frsh₂)

         vok₁ : ValOK V T₁
         vok₁ = subst₂ ValOK (trans (sym xeq) eq) refl (ok-share₁ shareok xvok)

         vok₂ : ValOK V T₂
         vok₂ = subst₂ ValOK (trans (sym xeq) eq) refl (ok-share₂ shareok xvok)

         ok' : EnvOK (proj₁ Ω') Γ'
         ok' = ok/cons H₀ vok₁ (ok/cons H₁ vok₂ ok-ex)

         H₂ : pot Γ ok ≡ pot Γ ok-ex₁
         H₂ = pot-extend-fresh Γ ok ok-ex₁ (fresh-wkn frsh₂)

         H₃ : pot Γ ok-ex₁ ≡ pot Γ ok-ex₂
         H₃ = pot-extend-fresh Γ ok-ex₁ ok-ex₂ (fresh-wkn frsh₁)

         H₄ : pot Γ ok-ex ≡ pot Γ ok-ex₂
         H₄ = pot-env-eq Γ ok-ex ok-ex₂ (proj₂ Ω')

         -- Prove that the new context has the same amount of potential
         H₅ : pot Γ ok-ex ≡ pot Γ ok
         H₅ = trans H₄ (trans (sym H₃) (sym H₂))

         H₆ : pot-val vok₁ + pot-val vok₂ ≡ pot-val xvok
         H₆ = pot-share shareok xvok vok₁ vok₂ (trans (sym xeq) eq)

         step₀ = sym (+-assoc (pot-val vok₁) (pot-val vok₂) (pot Γ ok-ex))
         step₁ = cong₂ (λ a b → a + b) H₆ H₅

         H₇ : pot Γ' ok' ≡ pot ((x , T) ∷ Γ) origok
         H₇ = begin
            pot Γ' ok'                                  ≡⟨ refl ⟩
            pot-val vok₁ + (pot-val vok₂ + pot Γ ok-ex) ≡⟨ step₀ ⟩
            pot-val vok₁ + pot-val vok₂ + pot Γ ok-ex   ≡⟨ step₁ ⟩
            pot-val xvok + pot Γ ok                     ≡⟨ refl ⟩
            pot ((x , T) ∷ Γ) origok                    ∎

         H₈ : z + pot Γ' ok' + q ≡ k
         H₈ = begin
          z + pot Γ' ok' + q                ≡⟨ cong (λ a → z + a + q) H₇ ⟩
          z + pot ((x , T) ∷ Γ) origok + q  ≡⟨ L ⟩
          k                                 ∎

         resok : ValOK V₁ M
         resok = preservation DS DE ok' sok

         IH : Σ ℕ (λ k' → (proj₁ Ω') ⊢ E ⇓ V₁ ∣〈 k , k' 〉 × z + pot-val resok + q' ≡ k')
         IH = soundness Γ' (proj₁ Ω') ok' E V₁ M DS DE sok k z H₈

         k' = proj₁ IH

         step₀ = cong (λ a → z + a + q') (pot-val-eq vok' resok refl)
         step₁ = proj₂ (proj₂ IH)

         H₉ : z + pot-val vok' + q' ≡ k'
         H₉ = begin
             z + pot-val vok' + q'   ≡⟨ step₀ ⟩
             z + pot-val resok + q'  ≡⟨ step₁ ⟩
             k'                      ∎

      in  (k' , step/share Ω x x₁ x₂ E V V₁ k k' eq Ω' (proj₁ (proj₂ IH)) , H₉)
    soundness .((x , T₁) ∷ ∅) Ω (ok/cons refl vok ok/nil) .(app fd x) V₂ T₂
              (of/app x x₁ T₁ .T₂ fd q q₁ q' E eq eq₂)
              (step/app .Ω .fd .x x₂ V₁ .V₂ E₁ p p₁ p' T₃ T₄ j j' eq₃ eq₄ eq₅ DE)
              sok k z L =
      let
        resok = preservation
                  (of/app x x₁ T₁ T₂ fd q q₁ q' E eq eq₂)
                  (step/app Ω fd x x₂ V₁ V₂ E₁ p p₁ p' T₃ T₄ j j' eq₃ eq₄ eq₅ DE)
                  (ok/cons refl vok ok/nil)
                  sok

        k₁ = z + (pot-val vok + 0) + q₁

        H₀ : ((T₁ , q₁ , q' , T₂) , x₁ , E) ≡ ((T₃ , j , j' , T₄) , x₂ , E₁)
        H₀ = trans (sym eq) eq₄

        (eqT₁T₃ , eqq₁j , eqq'j' , eqT₂T₄ , eqx₁x₂ , eqEE₁ ) = prod-eq₆ H₀

        H₁ : ValOK V₁ T₃
        H₁ = subst₂ ValOK eq₃ eqT₁T₃ vok

        H₂ : z + (pot-val H₁ + 0) + j ≡ k₁
        H₂ = cong₂ (λ a b → z + a + b)
                      (cong (λ a → a + 0) (sym (pot-val-eq₂ vok H₁ eq₃ eqT₁T₃)))
                      (sym eqq₁j)

        H₃ : EnvOK ((x₂ , V₁ ∣ ◎)) ((x₂ , T₃) ∷ ∅)
        H₃ = ok/cons (env-extend x₂) H₁ ok/nil

        resok' = preservation (sok eq₄) DE H₃ sok

        IH : Σ ℕ (λ k' → ((x₂ , V₁ ∣ ◎) ⊢ E₁ ⇓ V₂ ∣〈 k₁ , k' 〉) ×
                          z + pot-val resok' + j' ≡ k')
        IH = soundness ((x₂ , T₃) ∷ ∅) ((x₂ , V₁ ∣ ◎)) H₃
                       E₁ V₂ T₄ (sok eq₄) DE sok k₁ z H₂

        step₀ = cong (λ a → z + (pot-val vok + 0) + a) eq₂
        step₁ = sym (+-assoc (z + (pot-val vok + 0)) q₁ (metric app))
        H₄ : k ≡ k₁ + metric app
        H₄ = begin
          k                                         ≡⟨ sym L ⟩
          z + (pot-val vok + 0) + q                 ≡⟨ step₀ ⟩
          z + (pot-val vok + 0) + (q₁ + metric app) ≡⟨ step₁ ⟩
          z + (pot-val vok + 0) + q₁ + metric app   ≡⟨ refl ⟩
          k₁ + metric app                           ∎

        k' = proj₁ IH

        H₅ : z + pot-val resok + q' ≡ k'
        H₅ = subst₂ (λ a b → z + a + b ≡ k')
                    (pot-val-eq₂ resok' resok refl (sym eqT₂T₄))
                    (sym eqq'j')
                    (proj₂ (proj₂ IH))

      in k' ,
         step/app Ω fd x x₂ V₁ V₂ E₁ k k₁ k' T₃
                  T₄ j j' eq₃ eq₄ H₄ (proj₁ (proj₂ IH))  ,
         H₅
    soundness .((v , T') ∷ Γ) Ω (ok/cons eq vok ok) E V T
              (of/wkn Γ .E .T q q' v T' DF shareok DS) DE sok k z L =
      let
        resok = preservation DS DE ok sok
        IH : ∀ (n c : ℕ)
             → c + (pot Γ ok) + q ≡ n
             → Σ ℕ (λ n' → Ω ⊢ E ⇓ V ∣〈 n , n' 〉 × c + pot-val resok + q' ≡ n')
        IH = soundness Γ Ω ok E V T DS DE sok

        step₀ = cong (λ a → z + (a + pot Γ ok) + q) (sym (pot-share-same shareok vok))

        H₀ : z + pot Γ ok + q ≡ k
        H₀ = begin
          z + (pot Γ ok) + q               ≡⟨ refl ⟩
          z + (0 + pot Γ ok) + q           ≡⟨ step₀ ⟩
          z + (pot-val vok + pot Γ ok) + q ≡⟨ L ⟩
          k                                ∎
      in IH k z H₀
    soundness Γ Ω ok E V T (of/rlx .Γ .E .T p p' q q' DS x x₁) DE sok k z L =
      let
        resok = preservation DS DE ok sok
        less-than-or-equal {f} p+f≡q = ≤⇒≤″ x

        H₀ : z + f + pot Γ ok + p ≡ k
        H₀ = begin
          z + f + pot Γ ok + p   ≡⟨ cong (λ a → a + p) (+-assoc-comm z f (pot Γ ok)) ⟩
          z + pot Γ ok + f + p   ≡⟨ +-assoc-comm (z + pot Γ ok) f p ⟩
          z + pot Γ ok + p + f   ≡⟨ +-assoc (z + pot Γ ok) p f ⟩
          z + pot Γ ok + (p + f) ≡⟨ cong (λ a → z + pot Γ ok + a) p+f≡q ⟩
          z + pot Γ ok + q       ≡⟨ L ⟩
          k                      ∎

        IH : Σ ℕ (λ k' → (Ω ⊢ E ⇓ V ∣〈 k , k' 〉) × z + f + pot-val resok + p' ≡ k')
        IH = soundness Γ Ω ok E V T DS DE sok k (z + f) H₀
        k' = proj₁ IH

        H₁ : p + q' ≡ p + (p' + f)
        H₁ = begin
          p + q'       ≡⟨ x₁ ⟩
          q + p'       ≡⟨ cong (λ a → a + p') (sym p+f≡q) ⟩
          (p + f) + p' ≡⟨ +-assoc p f p' ⟩
          p + (f + p') ≡⟨ cong (λ a → p + a) (+-comm f p') ⟩
          p + (p' + f) ∎

        H₂ : p' + f ≡ q'
        H₂ = n+m≡n+k⇒m≡k p (p' + f) q' (sym H₁)

        step₁ = cong (λ a → a + p') (+-assoc-comm z f (pot-val resok))
        H₃ : k' ≡ z + pot-val resok + q'
        H₃ = begin
          k'                           ≡⟨ sym (proj₂ (proj₂ IH)) ⟩
          z + f + pot-val resok + p'   ≡⟨ step₁ ⟩
          z + pot-val resok + f + p'   ≡⟨ +-assoc-comm (z + pot-val resok) f p' ⟩
          z + pot-val resok + p' + f   ≡⟨ +-assoc (z + pot-val resok) p' f ⟩
          z + pot-val resok + (p' + f) ≡⟨ cong (λ a → z + pot-val resok + a) H₂ ⟩
          z + pot-val resok + q'       ∎
      in  (k' , proj₁ (proj₂ IH) , sym H₃)
    soundness .(Γ₂ ++ Γ₁) Ω ok E V T (of/exg Γ₁ Γ₂ .E .T q q' DS) DE sok k z L =
      let
        vok = preservation (of/exg Γ₁ Γ₂ E T q q' DS) DE ok sok
        okexg = ok-exg Γ₂ Γ₁ ok
        ok₁ : EnvOK Ω Γ₁
        ok₁ = ok-append₂ Γ₂ Γ₁ ok

        ok₂ : EnvOK Ω Γ₂
        ok₂ = ok-append₁ Γ₂ Γ₁ ok

        step₀ = cong (λ y → z + y + q) (pot-concat Γ₁ Γ₂ okexg ok₁ ok₂)
        step₁ = cong (λ y → z + y + q) (sym (+-comm (pot Γ₂ ok₂) (pot Γ₁ ok₁)))
        step₂ = cong (λ y → z + y + q) (sym (pot-concat Γ₂ Γ₁ ok ok₂ ok₁))

        H₀ : z + pot (Γ₁ ++ Γ₂) okexg + q ≡ k
        H₀ = begin
          z + pot (Γ₁ ++ Γ₂) okexg + q      ≡⟨ step₀ ⟩
          z + (pot Γ₁ ok₁ + pot Γ₂ ok₂) + q ≡⟨ step₁ ⟩
          z + (pot Γ₂ ok₂ + pot Γ₁ ok₁) + q ≡⟨ step₂ ⟩
          z + pot (Γ₂ ++ Γ₁) ok + q         ≡⟨ L ⟩
          k                                 ∎

        IH : Σ ℕ (λ k' → (Ω ⊢ E ⇓ V ∣〈 k , k' 〉) × z + pot-val vok + q' ≡ k')
        IH = soundness (Γ₁ ++ Γ₂) Ω okexg E V T DS DE sok k z H₀
      in (proj₁ IH , proj₁ (proj₂ IH) , proj₂ (proj₂ IH))
