-- Author: Mario Dehesa Azuara (mdehazu@gmail.com)

open import Context
open import Environment
open import Syntax
open import NatUtil

import Relation.Binary.PropositionalEquality as PropEq
open PropEq using (_≡_;refl;cong;sym;subst₂;cong-app;cong₂;trans)
open PropEq.≡-Reasoning

open import Relation.Nullary using (yes;no)
open import Relation.Nullary.Decidable using (⌊_⌋)

open import Data.Bool using (true;false)
open import Data.Empty using (⊥-elim)
open import Data.List using (_∷_;_++_) renaming ([] to ∅)
open import Data.Nat using (ℕ;_+_;_≤_;z≤n;s≤s)
open import Data.Nat.Properties.Simple using (+-comm;+-assoc;+-right-identity)
open import Data.Product using (_,_;Σ;proj₁;proj₂)


module Potential where
    pot-val : ∀ {v : Val}{T : Tp} → ValOK v T → ℕ
    pot-val vok/tt = 0
    pot-val vok/ff = 0
    pot-val vok/nil = 0
    pot-val (vok/cons {n} _ _ hvok tvok) = pot-val hvok + pot-val tvok + n

    pot : ∀{Ω : Env}(Γ : Ctx) → EnvOK Ω Γ → ℕ
    pot ∅ ok/nil = 0
    pot (_ ∷ Γ) (ok/cons _ hok tok) = (pot-val hok) + pot Γ tok

    -- Useful lemmas for dealing with potential

    pot-val-eq : ∀ {v₁ v₂ : Val}{T : Tp}
                   (ok₁ : ValOK v₁ T)
                   (ok₂ : ValOK v₂ T)
                 → v₁ ≡ v₂
                 → pot-val ok₁ ≡ pot-val ok₂
    pot-val-eq vok/tt vok/tt refl = refl
    pot-val-eq vok/ff vok/ff refl = refl
    pot-val-eq vok/nil vok/nil refl = refl
    pot-val-eq (vok/cons {n} v vs ok₁ ok₂) (vok/cons {.n} .v .vs ok₃ ok₄) refl =
      cong₂ (λ a b → a + b + n) (pot-val-eq ok₁ ok₃ refl) (pot-val-eq ok₂ ok₄ refl)

    pot-val-eq₂ : ∀ {v₁ v₂ : Val}{T₁ T₂ : Tp}
                   (ok₁ : ValOK v₁ T₁)
                   (ok₂ : ValOK v₂ T₂)
                 → v₁ ≡ v₂
                 → T₁ ≡ T₂
                 → pot-val ok₁ ≡ pot-val ok₂
    pot-val-eq₂ vok/tt vok/tt refl refl = refl
    pot-val-eq₂ vok/ff vok/ff refl refl = refl
    pot-val-eq₂ vok/nil vok/nil refl refl = refl
    pot-val-eq₂ (vok/cons {n} v vs ok₁ ok₂) (vok/cons {.n} .v .vs ok₃ ok₄) refl refl =
      cong₂ (λ a b → a + b + n) (pot-val-eq ok₁ ok₃ refl) (pot-val-eq ok₂ ok₄ refl)

    pot-sing-eq : ∀ {Ω : Env}{V : Val}{T : Tp}{v : Var}
                    (ok : EnvOK Ω ((v , T) ∷ ∅))
                    (vok : ValOK V T)
                  → Ω v ≡ V
                  → pot-val vok ≡ pot ((v , T) ∷ ∅) ok
    pot-sing-eq (ok/cons eq vok ok/nil) vok₁ refl =
      trans (sym (+-right-identity (pot-val vok₁)))
            (cong (λ a → a + 0) (pot-val-eq vok₁ vok eq))

    pot-bool-ctx : ∀ {Γ : Ctx}{Ω : Env}{v : Val}{ok : EnvOK Ω Γ}
                     (vok : ValOK v bool)
                   → pot-val vok + pot Γ ok ≡ pot Γ ok
    pot-bool-ctx vok/tt = refl
    pot-bool-ctx vok/ff = refl

    pot-env-eq : ∀ {Ω₁ Ω₂ : Env}
                   (Γ : Ctx)
                   (ok₁ : EnvOK Ω₁ Γ)
                   (ok₂ : EnvOK Ω₂ Γ)
                 → Ω₁ ≡ Ω₂
                 → pot Γ ok₁ ≡ pot Γ ok₂
    pot-env-eq ∅ ok/nil ok/nil _ = refl
    pot-env-eq ((v , _) ∷ Γ) (ok/cons eq₁ vok₁ ok₁) (ok/cons eq₂ vok₂ ok₂) eq =
      cong₂ (λ a b → a + b)
            (pot-val-eq vok₁ vok₂ (trans (sym eq₁)
                                         (trans (cong-app eq v) eq₂)))
            (pot-env-eq Γ ok₁ ok₂ eq)

    pot-ok-eq : ∀ {Ω : Env}
                  (Γ : Ctx)
                  (ok₁ : EnvOK Ω Γ)
                  (ok₂ : EnvOK Ω Γ)
                → pot Γ ok₁ ≡ pot Γ ok₂
    pot-ok-eq ∅ ok/nil ok/nil = refl
    pot-ok-eq (_ ∷ Γ) (ok/cons eq₁ vok₁ ok₁) (ok/cons eq₂ vok₂ ok₂) =
      cong₂ (λ a b → a + b)
            (pot-val-eq vok₁ vok₂ (trans (sym eq₁) eq₂))
            (pot-ok-eq Γ ok₁ ok₂)

    pot-concat : ∀ {Ω : Env}
                   (Γ₁ Γ₂ : Ctx)
                   (ok : EnvOK Ω (Γ₁ ++ Γ₂))
                   (ok₁ : EnvOK Ω Γ₁)
                   (ok₂ : EnvOK Ω Γ₂)
                 → pot (Γ₁ ++ Γ₂) ok ≡ pot Γ₁ ok₁ + pot Γ₂ ok₂
    pot-concat ∅ Γ₂ ok ok/nil ok₂ = pot-ok-eq Γ₂ ok ok₂
    pot-concat (_ ∷ Γ₁) Γ₂ (ok/cons eq vok ok) (ok/cons eq₁ vok₁ ok₁) ok₂ =
      let
        IH : pot (Γ₁ ++ Γ₂) ok ≡ pot Γ₁ ok₁ + pot Γ₂ ok₂
        IH = pot-concat Γ₁ Γ₂ ok ok₁ ok₂

        H₀ : pot-val vok + pot (Γ₁ ++ Γ₂) ok ≡ pot-val vok₁ + (pot Γ₁ ok₁ + pot Γ₂ ok₂)
        H₀ = cong₂ (λ a b → a + b) (pot-val-eq vok vok₁ (trans (sym eq) eq₁)) IH
      in  trans H₀ (sym (+-assoc (pot-val vok₁) (pot Γ₁ ok₁) (pot Γ₂ ok₂)))

    pot-cons : ∀ {n : ℕ}{v vs v₁ vs₁ : Val}{T : Tp}
                 (vokh : ValOK v T)
                 (vokt : ValOK vs (list n T))
                 (vokcons : ValOK (cons v₁ vs₁) (list n T))
                 (eq₁ : v ≡ v₁)
                 (eq₂ : vs ≡ vs₁)
               → pot-val vokh + pot-val vokt + n ≡ pot-val vokcons
    pot-cons {n} vokh vokt (vok/cons v vs vokh₁ vokt₁) refl refl =
       cong₂ (λ h t → h + t + n)
             (pot-val-eq vokh vokh₁ refl)
             (pot-val-eq vokt vokt₁ refl)


    pot-cons₂ : ∀ {n : ℕ}{v vs xs : Val}{T : Tp}
                 (vokh : ValOK v T)
                 (vokt : ValOK vs (list n T))
                 (vokcons : ValOK xs (list n T))
                 (eq : cons v vs ≡ xs)
               → pot-val vokh + pot-val vokt + n ≡ pot-val vokcons
    pot-cons₂ {n} vokh vokt (vok/cons v vs vokh₁ vokt₁) refl =
       cong₂ (λ h t → h + t + n)
             (pot-val-eq vokh vokh₁ refl)
             (pot-val-eq vokt vokt₁ refl)


    pot-extend-fresh : ∀ {x : Var}{V : Val}{Ω : Env}
                         (Γ : Ctx)
                         (ok₁ : EnvOK Ω Γ)
                         (ok₂ : EnvOK (x , V ∣ Ω) Γ)
                         → fresh x Γ
                         → pot Γ ok₁ ≡ pot Γ ok₂
    pot-extend-fresh ∅ ok/nil ok/nil frsh = refl
    pot-extend-fresh {x} ((v , T) ∷ Γ) (ok/cons eq₁ vok₁ ok₁)
                                 (ok/cons eq₂ vok₂ ok₂) frsh with x ≟ v
    ... | yes p = ⊥-elim (frsh T (here ((p , refl))))
    ... | no ¬p = cong₂ (λ a b → a + b)
                        (pot-val-eq vok₁ vok₂ (trans (sym eq₁) eq₂))
                        (pot-extend-fresh Γ ok₁ ok₂ (fresh-wkn frsh))

    import Statics as S

    pot-sub : ∀ {V : Val}{T₁ T₂ : Tp}{metric : Oper → ℕ}{Δ : Sig}
                (ok₁ : ValOK V T₁)
                (ok₂ : ValOK V T₂)
                (sub : S._<∶_ metric Δ T₁ T₂)
              → pot-val ok₂ ≤ pot-val ok₁
    pot-sub vok/tt vok/tt sub = z≤n
    pot-sub vok/ff vok/ff sub = z≤n
    pot-sub vok/nil vok/nil sub = z≤n
    pot-sub (vok/cons v vs vokh₁ vokt₁)
            (vok/cons .v .vs vokh₂ vokt₂)
            (S.sub/list sub le) =
      let
        IH₁ : pot-val vokh₂ ≤ pot-val vokh₁
        IH₁ = pot-sub vokh₁ vokh₂ sub

        IH₂ : pot-val vokt₂ ≤ pot-val vokt₁
        IH₂ = pot-sub vokt₁ vokt₂ (S.sub/list sub le)

        H₀ : pot-val vokh₂ + pot-val vokt₂ ≤ pot-val vokh₁ + pot-val vokt₁
        H₀ = le-add IH₁ IH₂
      in le-add H₀ le

    pot-share : ∀ {metric : Oper → ℕ}{Δ : Sig}{V V₁ : Val}{T T₁ T₂ : Tp}
                  (shareok : S.ShareOK metric Δ T T₁ T₂)
                  (vok : ValOK V T)
                  (vok₁ : ValOK V₁ T₁)
                  (vok₂ : ValOK V₁ T₂)
                  (eq : V ≡ V₁)
                → pot-val vok₁ + pot-val vok₂ ≡ pot-val vok
    pot-share S.sok/bool vok/tt vok/tt vok/tt refl = refl
    pot-share S.sok/bool vok/ff vok/ff vok/ff refl = refl
    pot-share (S.sok/list T T₁ T₂ x shareok) vok/nil vok/nil vok/nil refl = refl
    pot-share (S.sok/list T T₁ T₂ pok shareok)
              (vok/cons {p} v vs vokh vokt)
              (vok/cons {p₁} .v .vs vokh₁ vokt₁)
              (vok/cons {p₂} .v .vs vokh₂ vokt₂)
              refl =
      let
        IH₁ : pot-val vokh₁ + pot-val vokh₂ ≡ pot-val vokh
        IH₁ = pot-share shareok vokh vokh₁ vokh₂ refl

        IH₂ : pot-val vokt₁ + pot-val vokt₂ ≡ pot-val vokt
        IH₂ = pot-share (S.sok/list T T₁ T₂ pok shareok) vokt vokt₁ vokt₂ refl

        step₀ = sym (+-assoc (pot-val vokh₁ + pot-val vokt₁ + p₁)
                             (pot-val vokh₂ + pot-val vokt₂)
                             p₂)

        step₁ = cong (λ a → a + p₂) (+-assoc-comm (pot-val vokh₁ + pot-val vokt₁)
                                                  p₁
                                                  (pot-val vokh₂ + pot-val vokt₂))

        step₂ = +-assoc (pot-val vokh₁ + pot-val vokt₁ +
                                        (pot-val vokh₂ + pot-val vokt₂))
                        p₁
                        p₂

        step₃ = cong (λ a → a + (p₁ + p₂))
                     (sym (+-assoc (pot-val vokh₁ + pot-val vokt₁)
                                   (pot-val vokh₂)
                                   (pot-val vokt₂)))

        step₄ = cong (λ a → a + pot-val vokt₂ + (p₁ + p₂))
                     (+-assoc-comm (pot-val vokh₁)
                                   (pot-val vokt₁)
                                   (pot-val vokh₂))

        step₅ = cong (λ a → a + (p₁ + p₂))
                     (+-assoc (pot-val vokh₁ + pot-val vokh₂)
                              (pot-val vokt₁)
                              (pot-val vokt₂))

        step₆ = cong₂ (λ a b → a + b)
                      (cong₂ (λ a b → a + b) IH₁ IH₂)
                      (sym pok)

        H₀ = begin
          (pot-val vokh₁ + pot-val vokt₁ + p₁) + (pot-val vokh₂ + pot-val vokt₂ + p₂)
                                                                          ≡⟨ step₀ ⟩
          pot-val vokh₁ + pot-val vokt₁ + p₁ + (pot-val vokh₂ + pot-val vokt₂) + p₂
                                                                          ≡⟨ step₁ ⟩
          pot-val vokh₁ + pot-val vokt₁ + (pot-val vokh₂ + pot-val vokt₂) + p₁ + p₂
                                                                          ≡⟨ step₂ ⟩
          pot-val vokh₁ + pot-val vokt₁ + (pot-val vokh₂ + pot-val vokt₂) + (p₁ + p₂)
                                                                          ≡⟨ step₃ ⟩
          pot-val vokh₁ + pot-val vokt₁ + pot-val vokh₂ + pot-val vokt₂ + (p₁ + p₂)
                                                                          ≡⟨ step₄ ⟩
          pot-val vokh₁ + pot-val vokh₂ + pot-val vokt₁ + pot-val vokt₂ + (p₁ + p₂)
                                                                          ≡⟨ step₅ ⟩
          pot-val vokh₁ + pot-val vokh₂ + (pot-val vokt₁ + pot-val vokt₂) + (p₁ + p₂)
                                                                          ≡⟨ step₆ ⟩
          pot-val vokh + pot-val vokt + p                                 ∎
      in H₀

    pot-share-same : ∀ {T : Tp}{metric : Oper → ℕ}{Δ : Sig}
                     → (shareok : S.ShareOK metric Δ T T T)
                     → ∀ {V : Val}
                         (vok : ValOK V T)
                       → pot-val vok ≡ 0
    pot-share-same S.sok/bool vok/tt = refl
    pot-share-same S.sok/bool vok/ff = refl
    pot-share-same (S.sok/list T₁ .T₁ .T₁ x shareok) vok/nil = refl
    pot-share-same (S.sok/list {p} {.p} {.p} T₁ .T₁ .T₁ x shareok)
                   (vok/cons v vs vok vok₁) =
      let
        shareok' = S.sok/list T₁ T₁ T₁ x shareok

        H₀ : p ≡ 0
        H₀ = id-left-unique p p x
      in cong₂ (λ a b → a + b)
               (cong₂ (λ a b → a + b)
                      (pot-share-same shareok vok)
                      (pot-share-same shareok' vok₁) )
               H₀
