//  1394CameraControlSize.cpp: implementation of the C1394CameraControlSize class.
//
//	Version 4.0
//
//	Copyright 5/2000
// 
//	Iwan Ulrich
//	Robotics Institute
//	Carnegie Mellon University
//	Pittsburgh, PA
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//////////////////////////////////////////////////////////////////////

#include <windows.h>
#include "1394Camera.h"
#include "1394CameraControlSize.h"

extern "C" {
#include "pch.h"
}

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

C1394CameraControlSize::C1394CameraControlSize()
{
	m_maxH = 0;
	m_maxV = 0;
	m_unitH = 0;
	m_unitV = 0;
	m_top = 0;
	m_left = 0;
	m_width = 0;
	m_height = 0;
	m_colorCode = 0;
	for (int i=0; i<7; i++)
		m_colorCodes[i] = false;

	m_pixelsFrame = 0;
	m_bytesFrameHigh = 0;
	m_bytesFrameLow = 0;
	m_bytesPacketMin = 0;
	m_bytesPacketMax = 0;
	m_bytesPacket = 0;
}


C1394CameraControlSize::~C1394CameraControlSize()
{
}

int C1394CameraControlSize::Status()
{
	unsigned long status;
	unsigned long temp = 0x80000000;
	int ret,i;

	// size control present?
	if (!Supported())
		return CAM_ERROR;

	// mode supported?
	if (!ModeSupported(m_pCamera->m_videoMode))
		return CAM_ERROR_INVALID_VIDEO_SETTINGS;

	if(ret = m_pCamera->ReadQuadlet(0x2e0 + 4*m_pCamera->m_videoMode, &status))
		return ret;

	m_offset = 4*status + 0xf0000000;

	// inquire features
	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x00, &status)) 
		return ret;
	m_maxV = status & 0xffff;
	m_maxH = status >> 16;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x04, &status))
		return ret;
	m_unitV = status & 0xffff;
	m_unitH = status >> 16;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x08, &status)) 
		return ret;
	m_top = status & 0xffff;
	m_left = status >> 16;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x0c, &status))
		return ret;
	m_height = status & 0xffff;
	m_width = status >> 16;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x10, &status)) 
		return ret;
	m_colorCode = status & 0x00ff;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x14, &status))
		return ret;
	for (i=0; i<7; i++)
	{
		m_colorCodes[i] = (status & temp) != 0;
		temp >>= 1;
	}

	
	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x34,&m_pixelsFrame))
		return ret;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x38,&m_bytesFrameHigh))
		return ret;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x3c, &m_bytesFrameLow))
		return ret;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x40, &status)) 
		return ret;
	m_bytesPacketMax = status & 0xffff;
	m_bytesPacketMin = status >> 16;

	if(ret = m_pCamera->ReadQuadlet(m_offset + 0x44, &status)) 
		return ret;
	m_bytesPacket = status >> 16;
	return CAM_SUCCESS;
}


int C1394CameraControlSize::SetPosition(int left, int top)
{
	int ret;
	unsigned long quadlet;
	m_left = left;
	m_top = top;
	
	quadlet = left;
	quadlet <<= 16;
	quadlet += top;
	if(ret = m_pCamera->WriteQuadlet(m_offset + 0x08, quadlet))
		return ret;
	return Update();
}


int C1394CameraControlSize::SetSize(int width, int height)
{
	int ret;
	unsigned long quadlet;
	m_width = width;
	m_height = height;
	quadlet = width;
	quadlet <<= 16;
	quadlet += height;
	if(ret = m_pCamera->WriteQuadlet(m_offset + 0x0c, quadlet))
		return ret;
	return Update();
}


int C1394CameraControlSize::SetColorCode(int code)
{
	int ret;
	unsigned long quadlet;
	m_colorCode = code;
	quadlet = code;
	quadlet <<= 24;
	if(ret = m_pCamera->WriteQuadlet(m_offset + 0x10, quadlet))
		return ret;
	return Update();
}

	
int C1394CameraControlSize::SetBytesPerPacket(int bytes)
{
	int ret;
	unsigned long quadlet;
	m_bytesPacket = bytes;
	quadlet = bytes;
	quadlet <<= 16;
	if(ret = m_pCamera->WriteQuadlet(m_offset + 0x44, quadlet))
		return ret;

	return CAM_SUCCESS;
}


bool C1394CameraControlSize::ModeSupported(int mode)
{
	unsigned long status;
	unsigned int temp = 0x80000000;
	int ret;
	if(ret = m_pCamera->ReadQuadlet(0x19c,&status))
		return false;
	temp >>= mode;
	return (status & temp) != 0;
}


bool C1394CameraControlSize::Supported()
{
	unsigned long status;
	int ret;
	
	if(ret = m_pCamera->ReadQuadlet(0x100,&status))
		return false;

	return (status & 0x01000000) != 0;
}


int C1394CameraControlSize::Update()
{
	int ret;

	if(ret = Status())
		return ret;

	if(ret = SetBytesPerPacket(m_pCamera->m_controlSize.m_bytesPacketMax))
		return ret;
	m_pCamera->m_width = m_width;
	m_pCamera->m_height = m_height;

	return CAM_SUCCESS;
}
