//  1394Camera.cpp: implementation of the C1394Camera class.
//
//	Version 5.2 : 03/14/2001
//
//	Copyright 5/2000
// 
//	Iwan Ulrich
//	Robotics Institute
//	Carnegie Mellon University
//	Pittsburgh, PA
//
//  Copyright 3/2002
//
//  Christopher Baker
//  Robotics Institute
//  Carnegie Mellon University
//  Pittsburgh, PA
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//  10/2001 : Christopher Baker <cbaker@andrew.cmu.edu>
//  Made modifications and optimizations to take advantage
//  of the additional features provided by the new device driver
//
//  03/2002 : Christopher Baker <cbaker@andrew.cmu.edu>
//  
//   -  Fixed quadlets per packet stuff.
//   -  Added Support for 16-bit Monochrome modes
//   -  other assorted tweaks and fixes for 5.2
//
//////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <string.h>
#include <mmsystem.h>
#include <stdio.h>
#include "1394Camera.h"
#include "pch.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

// tables of format sizing data

int tableWidth[3][8] = {         // [format][mode]
	{ 160,  320,  640,  640,  640,  640,  640,    0},
	{ 800,  800,  800, 1024, 1024, 1024,  800, 1024},
	{1280, 1280, 1280, 1600, 1600, 1600, 1280, 1600}};

int tableHeight[3][8] = {        // [format][mode]
	{ 120,  240,  480,  480,  480,  480,  480,    0},
	{ 600,  600,  600,  768,  768,  768,  600,  768},
	{ 960,  960,  960, 1200, 1200, 1200, 1024, 1200}};

// tableMaxBufferSize is, for the most part, extraneous, but it's okay to have, for now

int tableMaxBufferSize[3][8] = { // [format][mode]
	{  57600,  153600,  460800,  614400,  921600,  307200,  614400,       0},
	{ 960000, 1440000,  480000, 1572864, 2359296,  786432,  960000, 1572864},
	{2457600, 3686400, 1228800, 3840000, 5760000, 1920000, 2457600, 3840000}};

int tableQuadletsPerPacket[3][8][6] = 
// [format][mode][rate]
{	
	{	
		// format 0
		{   0,   0,  15,  30,  60,   0},					
		{   0,  20,  40,  80, 160,   0},	
		{   0,  60, 120, 240, 480,   0},	
		{   0,  80, 160, 320, 640,   0},	
		{   0, 120, 240, 480, 960,   0},	
		{   0,  40,  80, 160, 320, 640},
		{   0,  80, 160, 320, 640,   0},
		{   0,   0,   0,   0,   0,   0}

	},{
		// format 1
		{   0, 125, 250, 500,1000,   0},
		{   0,   0, 735, 750,   0,   0},	
		{   0,   0, 125, 250, 500,1000},	
		{  96, 192, 384, 768,   0,   0},	
		{ 144, 288, 576,   0,   0,   0},	
		{  48,  96, 192, 384, 768,   0},
		{   0, 125, 250, 500,1000,   0},
		{  96, 192, 384, 768,   0,   0}
	
	},{	
		// format 2
		{ 160, 320, 640,   0,   0,   0},
		{ 240, 480, 960,   0,   0,   0},	
		{  80, 160, 320, 640,   0,   0},	
		{ 250, 500,1000,   0,   0,   0},	
		{ 375, 750,   0,   0,   0,   0},	
		{ 125, 250, 500,1000,   0,   0},
		{ 160, 320, 640,   0,   0,   0},
		{ 250, 500,1000,   0,   0,   0}
	}
};

/****************************************************/
/*                                                  */
/*            PUBLIC MEMBER FUNCTIONS               */
/*                                                  */
/****************************************************/


/*
 * Constructor
 *
 * Arguments
 *   hWnd: Defaulted to NULL: deprecated, will eventualy remove
 * 
 * Responsibilities:
 *   - init internal vars
 *   - setup member control classes
 */

C1394Camera::C1394Camera()
{
	DllTrace(DLL_TRACE_ENTER,"ENTER C1394Camera Constructor\n");
	//m_nBuffers = 2;
	//m_nDescriptors = 1;

	// this is the best place to trace the structure sizes
	DllTrace(DLL_TRACE_VERBOSE,"sizeof(C1394Camera) = %d\n",sizeof(C1394Camera));
	DllTrace(DLL_TRACE_VERBOSE,"sizeof(C1394CameraControl) = %d\n",sizeof(C1394CameraControl));
	DllTrace(DLL_TRACE_VERBOSE,"sizeof(C1394CameraControlSize) = %d\n",sizeof(C1394CameraControlSize));
	DllTrace(DLL_TRACE_VERBOSE,"sizeof(C1394CameraControlTrigger) = %d\n",sizeof(C1394CameraControlTrigger));
	
	m_pData = NULL;
	//m_allocateMemory = false;
	m_pName = NULL;
	m_linkChecked = false;
	m_cameraInitialized = false;
	m_node = 0;
	//NULL = hWnd;
	m_hDeviceAcquisition = NULL;
	m_hDeviceCapture = NULL;

	m_videoMode = 1;
	m_videoFrameRate = 4;
	m_videoFormat = 1;
	ZeroMemory(&m_spec,sizeof(CAMERA_SPECIFICATION));
	ZeroMemory(&m_DeviceData,sizeof(DEVICE_DATA));

	m_controlBrightness.m_pCamera = this;
	m_controlAutoExposure.m_pCamera = this;
	m_controlSharpness.m_pCamera = this;
	m_controlWhiteBalance.m_pCamera = this;
	m_controlHue.m_pCamera = this;
	m_controlSaturation.m_pCamera = this;
	m_controlGamma.m_pCamera = this;
	m_controlShutter.m_pCamera = this;
	m_controlGain.m_pCamera = this;
	m_controlZoom.m_pCamera = this;
	m_controlFocus.m_pCamera = this;
	m_controlIris.m_pCamera = this;
	m_controlTrigger.m_pCamera = this;
	m_controlSize.m_pCamera = this;

	m_controlBrightness.SetOffset(BRIGHTNESS_OFFSET);
	m_controlAutoExposure.SetOffset(AUTO_EXPOSURE_OFFSET);
	m_controlSharpness.SetOffset(SHARPNESS_OFFSET);
	m_controlWhiteBalance.SetOffset(WHITE_BALANCE_OFFSET);
	m_controlHue.SetOffset(HUE_OFFSET);
	m_controlSaturation.SetOffset(SATURATION_OFFSET);
	m_controlGamma.SetOffset(GAMMA_OFFSET);
	m_controlShutter.SetOffset(SHUTTER_OFFSET);
	m_controlGain.SetOffset(GAIN_OFFSET);
	m_controlZoom.SetOffset(ZOOM_OFFSET);
	m_controlFocus.SetOffset(FOCUS_OFFSET);
	m_controlIris.SetOffset(IRIS_OFFSET);
	// trigger offset isn't necessary, as it is a given
	//m_controlTrigger.SetOffset(0x30);
/*
	m_controlBrightness.m_offsetInquiry = 0x500;
	m_controlAutoExposure.m_offsetInquiry = 0x504;
	m_controlSharpness.m_offsetInquiry = 0x508;
	m_controlWhiteBalance.m_offsetInquiry = 0x50c;
	m_controlHue.m_offsetInquiry = 0x510;
	m_controlSaturation.m_offsetInquiry = 0x514;
	m_controlGamma.m_offsetInquiry = 0x518;
	m_controlShutter.m_offsetInquiry = 0x51c;
	m_controlGain.m_offsetInquiry = 0x520;
	m_controlZoom.m_offsetInquiry = 0x580;
	m_controlFocus.m_offsetInquiry = 0x528;
	m_controlIris.m_offsetInquiry = 0x524;
	m_controlTrigger.m_offsetInquiry = 0x530;

	m_controlBrightness.m_offsetStatus = 0x800;
	m_controlAutoExposure.m_offsetStatus = 0x804;
	m_controlSharpness.m_offsetStatus = 0x808;
	m_controlWhiteBalance.m_offsetStatus = 0x80c;
	m_controlHue.m_offsetStatus = 0x810;
	m_controlSaturation.m_offsetStatus = 0x814;
	m_controlGamma.m_offsetStatus = 0x818;
	m_controlShutter.m_offsetStatus = 0x81c;
	m_controlGain.m_offsetStatus = 0x820;
	m_controlZoom.m_offsetStatus = 0x880;
	m_controlFocus.m_offsetStatus = 0x828;
	m_controlIris.m_offsetStatus = 0x824;
	m_controlTrigger.m_offsetStatus = 0x830;
*/
	DllTrace(DLL_TRACE_EXIT,"EXIT C1394Camera Constructor\n");
}


/*
 * Destructor
 *
 * Responsibilities
 *
 * If we are grabbing images, stop it.  This is the last line
 * of defense against sloppy programming.
 */

C1394Camera::~C1394Camera()
{
	DllTrace(DLL_TRACE_ENTER,"ENTER C1394Camera Destructor\n");
	if(m_hDeviceAcquisition != NULL)
		StopImageAcquisition();

	if(m_hDeviceCapture != NULL)
		StopImageCapture();
	DllTrace(DLL_TRACE_EXIT,"EXIT C1394Camera Destructor\n");
}


/*
 * CheckLink
 *
 * Public
 *
 * Arguments
 *  - <none>
 *
 * Returns
 *  - CAM_SUCCESS: I found at least one camera and have selected camera 0
 *  - CAM_ERROR: I found no cameras that are available
 *
 * Calls:
 *   GetNumberCameras
 *   SelectCamera
 *
 * Mutates:
 *   - m_linkchecked (public) indicates that this function has been called
 *       maybe remove it.
 *
 * Comments:
 *   This function is very poorly named.  It is maintained for backwards compatibility.
 *   It may eventualy be replaced by something like RefreshCameraList().
 */

int C1394Camera::CheckLink()
{
	DllTrace(DLL_TRACE_ENTER,"ENTER CheckLink\n");
	m_linkChecked = false;
	if(GetNumberCameras() <= 0)
	{
		DllTrace(DLL_TRACE_ERROR,"CheckLink: no cameras found\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT CheckLink (-1)\n");
		return CAM_ERROR;
	}

	SelectCamera(0);

	m_linkChecked = true;
	DllTrace(DLL_TRACE_EXIT,"EXIT CheckLink (0)\n");
	return 0;
}


/*
 * SelectCamera
 *
 * Public
 *
 * Indexes into the current device list to point the class at a particular camera
 *
 * Arguments
 *   - node: the (zero-indexed) index of the camera to select
 *
 * Returns:
 *   - CAM_SUCCESS on success
 *   - CAM_ERROR_PARAM_OUT_OF_RANGE: you gave a bad camera number
 *   - CAM_ERROR: general I/O error (probably GetCameraSpecification)
 *
 * Comments:
 *   This is the only class function that will generate dialog boxes and will do so
 *   if and only if it believes that the device you selected isn't *really* a camera
 *   that complies with the 1394 Digital Camera Specification
 */

int C1394Camera::SelectCamera(int node)
{
	int oldNode;
	char *oldName;
	int format, mode, rate;
	ULONG ulRet;

	DllTrace(DLL_TRACE_ENTER,"ENTER SelectCamera (%d)\n",node);

	if(node < 0 || node >= (int) m_DeviceData.numDevices)
	{
		DllTrace(DLL_TRACE_ERROR,"SelectCamera: Camera %d out of range\n",node);
		DllTrace(DLL_TRACE_EXIT,"EXIT SelectCamera (%d)\n",CAM_ERROR_PARAM_OUT_OF_RANGE);
		return CAM_ERROR_PARAM_OUT_OF_RANGE;
	}
	oldNode = m_node;
	m_node = node;
	oldName = m_pName;
	m_pName = m_DeviceData.deviceList[m_node].DeviceName;

	// check the software version
	ZeroMemory(&m_spec,sizeof(CAMERA_SPECIFICATION));
	if((ulRet = GetCameraSpecification(m_pName,&m_spec)))
	{
		DllTrace(DLL_TRACE_ERROR,"SelectCamera: Error %08x getting Camera Specification\n",ulRet);
		m_pName = oldName;
		m_node = oldNode;
		DllTrace(DLL_TRACE_EXIT,"EXIT SelectCamera (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	if(m_spec.ulSpecification != 0x00A02D)
	{
		char buf[128];
		DllTrace(DLL_TRACE_WARNING, "SelectCamera: Warning: Camera specification (%06x) does not match 0x00A02D\n", m_spec.ulSpecification);
		sprintf(buf,"Camera Specification 0x%06X does not match IIDC 1394 Specification (0x00A02D)",m_spec.ulSpecification);
		MessageBox(NULL,buf,"1394 Camera Error",MB_OK | MB_ICONERROR);
	}

	if (m_spec.ulVersion  < 0x000100 || m_spec.ulVersion > 0x000102)
	{
		char buf[128];
		DllTrace(DLL_TRACE_WARNING,"SelectCamera: Warning: Camera software version (%06x) is not supported\n",m_spec.ulVersion);
		sprintf(buf,"Camera Firmware Version 0x%06X is not a valid version for the IIDC 1394 Digital Camera Specification",m_spec.ulSpecification);
		MessageBox(NULL,buf,"1394 Camera Error",MB_OK | MB_ICONERROR);
	}

	// whenever we switch cameras, reset our internal stuff

	m_cameraInitialized = false;

	// initialize entire video matrix to true
	for (format=0; format<3; format++)
		for (mode=0; mode<8; mode++)
			for (rate=0; rate<6; rate++)
				m_videoFlags[format][mode][rate] = true;


	DllTrace(DLL_TRACE_CHECK,"SelectCamera: Selected \"%s\"\n",m_pName);
	DllTrace(DLL_TRACE_EXIT,"EXIT SelectCamera (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;

}


/*
 * InitCamera
 *
 * Public
 *
 * Performs General initialization of the C1394Camera class for
 * the currently selected camera
 *
 * Arguments:
 *  -<none>
 *
 * Returns:
 *  CAM_SUCCESS
 *  CAM_ERROR_NOT_INITIALIZED: No camera selected (this is poorly named and may be changed)
 *  CAM_ERROR_BUSY: Usualy indicates broken invariants, but in and of itself prevents an
 *    init in the middle of image acquisition
 *  CAM_ERROR: Some register IO failed, GetLastError will tell why
 *
 */

int C1394Camera::InitCamera()
{
	GET_MAX_SPEED_BETWEEN_DEVICES maxSpeed;
	DWORD dwRet;
	ULONG maxSpeedNotLocal, maxSpeedLocal;

	DllTrace(DLL_TRACE_ENTER,"ENTER InitCamera\n");
	
	if(m_cameraInitialized)
	{
		// this isn't really an error case as much as just a dumb mistake
		DllTrace(DLL_TRACE_WARNING,"InitCamera: Warning: Duplicate Call to InitCamera\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_SUCCESS);
		return CAM_SUCCESS;
	}

	if(!m_pName)
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error: no camera selected\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	if(m_hDeviceAcquisition || m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error: Camera is busy, stop image acquisition first\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR_BUSY);
		return CAM_ERROR_BUSY;
	}

	// reset state
	// this frees up any isoch resources that may be left behind
	// from a previous program that didn't clean up after itself
	// properly (i.e. crashed)

	// this should *probably* have a return code, but if things
	// go wrong with resetting the state, other problems will rear up

	ResetCameraState(m_pName);

	// determine max speed
	// this is used for allocating bandwidth and stuff
	maxSpeed.fulFlags = 0;
	maxSpeed.ulNumberOfDestinations = 0;
	if (dwRet = GetMaxSpeedBetweenDevices(NULL, m_pName, &maxSpeed))
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error %08x on GetMaxSpeedBetweenDevices (NotLocal)\n",dwRet);
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}
	maxSpeedNotLocal = maxSpeed.fulSpeed;

	maxSpeed.fulFlags = 1;
	if (dwRet = GetMaxSpeedBetweenDevices(NULL, m_pName, &maxSpeed))
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error %08x on GetMaxSpeedBetweenDevices (Local)\n",dwRet);
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}
	maxSpeedLocal = maxSpeed.fulSpeed;

	// select the smaller of the two
	m_maxSpeed = (maxSpeedLocal < maxSpeedNotLocal ? maxSpeedLocal : maxSpeedNotLocal);

	// get the vendor and model names from the driver
	// the return codes aren't checked here because they are
	// simple data transfers from the device extension to our
	// buffers
	GetModelName(m_pName,m_nameModel,256);
	GetVendorName(m_pName,m_nameVendor,256);
	GetUniqueID(m_pName,&m_UniqueID);

	// determine video formats/modes/rates
	// private functions return bools and do their own respective tracing
	if(!InquireVideoFormats())
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error on InquireVideoFormats\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	if(!InquireVideoModes())
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error on InquireVideoModes\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	if(!InquireVideoRates())
	{
		DllTrace(DLL_TRACE_ERROR,"InitCamera: Error on InquireVideoRates\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	m_cameraInitialized = true;

	DllTrace(DLL_TRACE_EXIT,"EXIT InitCamera (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;
}


/*
 * ReadQuadlet
 *
 * Public
 *
 * Essentially a wrapper for ReadRegisterUL
 *
 * Arguments:
 *  - address: the offset into the camera register space to read from
 *      Addresses leading with "f" as in 0xf0000344 will be treated as absolute addresses.
 *      Those without a leading "f" will be treated as offsets into the 
 *        CSR space, so 0x00000344 will be read at CSR + 0x344, usually 0xf0f00344
 *
 *  - pData: the place to put the data read in from the register.  The data will be in
 *      machine order, so the most significant bit would be 0x80000000
 *
 * Returns:
 *  - CAM_SUCCESS
 *  - CAM_ERROR: something bad happened down in the bowels of the OS, use GetLastError() to find out.
 *  - CAM_ERROR_NOT_INITIALIZED: no camera has been selected
 *
 * Comments:
 *  ReadQuadlet catches ERROR_SEM_TIMEOUT, which means the camera was too busy to process the request.
 *  It will retry the request for the initial value of nretries times, by default this is 4, but
 *  it may become a registry variable.
 */

int C1394Camera::ReadQuadlet(unsigned long address, unsigned long *pData)
{
	unsigned long data = 0;
	int nretries = 4;
	DWORD dwRet;

	DllTrace(DLL_TRACE_ENTER,"ENTER ReadQuadlet (%08x,%08x)\n",address,pData);

	if(!m_pName)
	{
		DllTrace(DLL_TRACE_ERROR,"ReadQuadlet: No Camera has been selected\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT ReadQuadlet (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	if(!pData)
	{
		DllTrace(DLL_TRACE_ERROR,"ReadQuadlet: You gave me a NULL pointer you fool!\n");
		SetLastError(ERROR_INVALID_PARAMETER);
		DllTrace(DLL_TRACE_EXIT,"EXIT ReadQuadlet (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	// we're gonna try this nretries times, looking for
	// ERROR_SEM_TIMEOUT, which maps to STATUS_IO_TIMEOUT
	// meaning that the camera can't keep up.
	while((dwRet = ReadRegisterUL(m_pName,address,pData)) != 0 && nretries > 0)
	{
		if(dwRet != ERROR_SEM_TIMEOUT)
			// some other error, break out
			break;
		// Sleep for 10 ms
		Sleep(10);
		nretries--;
		DllTrace(DLL_TRACE_WARNING,"ReadQuadlet: Warning: Timeout on ReadRegister@0x%08x.  Retries Remaining: %d\n",
			address,nretries);
	}

	if(dwRet != 0)
	{
		DllTrace(DLL_TRACE_ERROR,"ReadQuadlet: Unrecoverable error %08x on ReadQuadlet\n",dwRet);
		// dwRet was gotten in ReadQuadlet by a call to GetLastError, so it's still there
		DllTrace(DLL_TRACE_EXIT,"EXIT ReadQuadlet (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	DllTrace(DLL_TRACE_EXIT,"EXIT ReadQuadlet (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;
}


/*
 * WriteQuadlet
 *
 * Public
 *
 * Essentially a wrapper for WriteRegisterUL
 *
 * Arguments:
 *  - address: the offset into the camera register space to write to
 *      Addresses leading with "f" as in 0xf0000344 will be treated as absolute addresses.
 *      Those without a leading "f" will be treated as offsets into the 
 *        CSR space, so 0x00000344 will be written at CSR + 0x344, usually 0xf0f00344
 *
 *  - data: the data write to the register
 *
 * Returns:
 *  - CAM_SUCCESS
 *  - CAM_ERROR: something bad happened down in the bowels of the OS, use GetLastError() to find out.
 *  - CAM_ERROR_NOT_INITIALIZED: no camera has been selected
 *
 * Comments:
 *  WriteQuadlet catches ERROR_SEM_TIMEOUT, which means the camera was too busy to process the request.
 *  It will retry the request for the initial value of nretries times, by default this is 4, but
 *  it may become a registry variable.
 */

int C1394Camera::WriteQuadlet(unsigned long address, unsigned long data)
{
	int nretries = 4;
	DWORD dwRet;

	DllTrace(DLL_TRACE_ENTER,"ENTER WriteQuadlet (%08x,%08x)\n",address,data);

	if(!m_pName)
	{
		DllTrace(DLL_TRACE_ERROR,"WriteQuadlet: No Camera has been selected\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT WriteQuadlet (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	// we're gonna try this nretries times, looking for
	// ERROR_SEM_TIMEOUT, which maps to STATUS_IO_TIMEOUT
	// meaning that the camera can't keep up.
	while((dwRet = WriteRegisterUL(m_pName,address,data)) != 0 && nretries > 0)
	{
		if(dwRet != ERROR_SEM_TIMEOUT)
			// some other error, break out
			break;
		// Sleep for 10 ms
		Sleep(10);
		nretries--;
		DllTrace(DLL_TRACE_WARNING,"WriteQuadlet: Warning: Timeout on WriteRegister@0x%08x.  Retries Remaining: %d\n",
			address,nretries);
	}

	if(dwRet != 0)
	{
		DllTrace(DLL_TRACE_ERROR,"WriteQuadlet: Unrecoverable error %08x on WriteRegisterUL\n",dwRet);
		// dwRet was gotten in WriteRegisterUL by a call to GetLastError, so it's still there
		DllTrace(DLL_TRACE_EXIT,"EXIT WriteQuadlet (%d)\n",CAM_ERROR);
		return CAM_ERROR;
	}

	DllTrace(DLL_TRACE_EXIT,"EXIT WriteQuadlet (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;
}


/*
 * SetVideoFormat
 *
 * Public
 *
 * Sets current video format unless the camera is actively grabbing frames
 *
 * Arguments
 *  - format: the format in [0,7] that you wish to set
 *
 * Returns:
 *  - CAM_SUCCESS
 *  - CAM_ERROR_NOT_INITIALIZED: No camera celected and/or camera not initialized
 *  - CAM_ERROR_BUSY: The camera is actively acquiring images
 *  - CAM_ERROR: WriteRegister has failed, use GetLastError() to find out why.
 *
 * Comments
 *   Something doesn't seem quite right about writing to the camera's register right
 *   away.  I will have to think on a better alternative.
 */

int C1394Camera::SetVideoFormat(unsigned long format)
{
	DWORD dwRet;

	DllTrace(DLL_TRACE_ENTER,"ENTER SetVideoFormat (%d)\n",format);

	if (!m_pName || !m_cameraInitialized)
	{
		DllTrace(DLL_TRACE_ERROR,"SetVideoFormat: Camera is not initialized\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFormat (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	if(m_hDeviceAcquisition || m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"SetVideoFormat: Camera is busy\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFormat (%d)\n",CAM_ERROR_BUSY);
		return CAM_ERROR_BUSY;
	}
	
	if (format <= 7)
	{
		if(!m_bxAvailableFormats[format])
		{
			// the desired format is not supported
			DllTrace(DLL_TRACE_ERROR,"SetVideoFormat: Format %d not supported\n",format);
			DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFormat (%d)\n",CAM_ERROR_INVALID_VIDEO_SETTINGS);
			return CAM_ERROR_INVALID_VIDEO_SETTINGS;
		}

		// shift it over into the most significant bits
		if(dwRet = WriteQuadlet(0x608, format << 29))
		{
			DllTrace(DLL_TRACE_ERROR,"SetVideoFormat: error %08x on WriteRegister\n",dwRet);
			DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFormat (%d)\n",CAM_ERROR);
			return CAM_ERROR;
		}

		m_videoFormat = format;
		// update parameters is a little funky, but leave it anyway
		UpdateParameters();
	} else {
		DllTrace(DLL_TRACE_ERROR,"SetVideoFormat: format %d out of range\n",format);
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFormat (%d)\n",CAM_ERROR_PARAM_OUT_OF_RANGE);
		return CAM_ERROR_PARAM_OUT_OF_RANGE;
	}

	if (format == 7)
		m_controlSize.Status();

	DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFormat (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;
}


/*
 * SetVideoMode
 *
 * Public
 *
 * Sets current video mode unless the camera is actively grabbing frames
 *
 * Arguments
 *  - mode: the desired mode in [0,7] that you wish to set
 *
 * Returns:
 *  - CAM_SUCCESS
 *  - CAM_ERROR_NOT_INITIALIZED: No camera celected and/or camera not initialized
 *  - CAM_ERROR_BUSY: The camera is actively acquiring images
 *  - CAM_ERROR: WriteRegister has failed, use GetLastError() to find out why.
 */

int C1394Camera::SetVideoMode(unsigned long mode)
{
	DWORD dwRet;

	DllTrace(DLL_TRACE_ENTER,"ENTER SetVideoMode (%d)\n",mode);

	if (!m_pName || !m_cameraInitialized)
	{
		DllTrace(DLL_TRACE_ERROR,"SetVideoMode: Camera is not initialized\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoMode (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	if(m_hDeviceAcquisition || m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"SetVideoMode: Camera is busy\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoMode (%d)\n",CAM_ERROR_BUSY);
		return CAM_ERROR_BUSY;
	}
		
	if (mode<=7)
	{
		if(!m_bxAvailableModes[m_videoFormat][mode])
		{
			DllTrace(DLL_TRACE_ERROR,"SetVideoMode: mode %d is not supported under format %d\n",mode,m_videoFormat);
			DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoMode (%d)\n",CAM_ERROR_INVALID_VIDEO_SETTINGS);
			return CAM_ERROR_INVALID_VIDEO_SETTINGS;
		}

		if(dwRet = WriteQuadlet(0x604, mode << 29))
		{
			DllTrace(DLL_TRACE_ERROR,"SetVideoMode: error %08x on WriteRegister\n",dwRet);
			DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoMode (%d)\n",CAM_ERROR);
			return CAM_ERROR;
		}

		m_videoMode = mode;
		UpdateParameters();
	} else {
		DllTrace(DLL_TRACE_ERROR,"SetVideoMode: mode %d out of range\n",mode);
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoMode (%d)\n",CAM_ERROR_PARAM_OUT_OF_RANGE);
		return CAM_ERROR_PARAM_OUT_OF_RANGE;
	}

	DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoMode (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;
}


/*
 * SetVideoFrameRate
 *
 * Public
 *
 * Sets current video FrameRate unless the camera is actively grabbing frames
 *
 * Arguments
 *  - rate: the desired frame rate in [0,5] that you wish to set
 *
 * Returns:
 *  - CAM_SUCCESS
 *  - CAM_ERROR_NOT_INITIALIZED: No camera celected and/or camera not initialized
 *  - CAM_ERROR_BUSY: The camera is actively acquiring images
 *  - CAM_ERROR: WriteRegister has failed, use GetLastError() to find out why.
 */

int C1394Camera::SetVideoFrameRate(unsigned long rate)
{
	DWORD dwRet;

	DllTrace(DLL_TRACE_ENTER,"ENTER SetVideoFramteRate (%d)\n",rate);

	if (!m_pName || !m_cameraInitialized)
	{
		DllTrace(DLL_TRACE_ERROR,"SetVideoFrameRate: Camera is not initialized\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFrameRate (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	if(m_hDeviceAcquisition || m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"SetVideoFrameRate: Camera is busy\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFrameRate (%d)\n",CAM_ERROR_BUSY);
		return CAM_ERROR_BUSY;
	}

	if (rate <= 5)
	{
		if(!m_videoFlags[m_videoFormat][m_videoMode][rate])
		{
			DllTrace(DLL_TRACE_ERROR,"SetVideoFrameRate: rate %d is not supported under format %d, mode %d\n",rate,m_videoFormat,m_videoMode);
			DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFrameRate (%d)\n",CAM_ERROR_INVALID_VIDEO_SETTINGS);
			return CAM_ERROR_INVALID_VIDEO_SETTINGS;
		}

		if(dwRet = WriteQuadlet(0x600, rate << 29))
		{
			DllTrace(DLL_TRACE_ERROR,"SetVideoFrameRate: error %08x on WriteRegister\n",dwRet);
			DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFrameRate (%d)\n",CAM_ERROR);
			return CAM_ERROR;
		}

		m_videoFrameRate = rate;

		UpdateParameters();

	} else {
		DllTrace(DLL_TRACE_ERROR,"SetVideoFrameRate: rate %d out of range\n",rate);
		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFrameRate (%d)\n",CAM_ERROR_PARAM_OUT_OF_RANGE);
		return CAM_ERROR_PARAM_OUT_OF_RANGE;
	}

		DllTrace(DLL_TRACE_EXIT,"EXIT SetVideoFrameRate (%d)\n",CAM_SUCCESS);
	return CAM_SUCCESS;
}


/*
 * GetVideoFrameRate
 *
 * Public
 *
 * one-line acessor for m_videoFrameRate
 */

int C1394Camera::GetVideoFrameRate()
{
	return (m_videoFrameRate);
}


/*
 * GetVideoMode
 *
 * Public
 *
 * one-line acessor for m_videoFrameRate
 */

int C1394Camera::GetVideoMode()
{
	return (m_videoMode);
}


/*
 * GetVideoFormat
 *
 * Public
 *
 * one-line acessor for m_videoFrameRate
 */

int C1394Camera::GetVideoFormat()
{
	return (m_videoFormat);
}


/*
 * GetMaxSpeed
 *
 * Public
 *
 * one-line accessor for m_maxSpeed
 * Note, the "*100" allows us to return the actual rate in Mbps
 */

int C1394Camera::GetMaxSpeed()
{
	return (m_maxSpeed * 100);
}


/*
 * GetVersion
 *
 * Public
 * 
 * One-line accessor for m_spec.ulVersion
 */

unsigned long C1394Camera::GetVersion()
{
	return m_spec.ulVersion;
}


/*
 * GetNumberCameras
 *
 * Public
 * 
 * Returns the stored result of GetDevideList if there is one to be had,
 * Otherwise returns the result of a direct call to GetDeviceList, which
 * the number of currently active 1394cmdr device objects.
 */

int C1394Camera::GetNumberCameras()
{
	if(m_linkChecked)
		return m_DeviceData.numDevices;
	else
		return GetDeviceList(&m_DeviceData);
}


/*
 * GetNode
 *
 * Public
 * 
 * One-line accessor for m_node, in case you forget (or are just lazy)
 */

int C1394Camera::GetNode()
{
	return m_node;
}

/****************************************************/
/*                                                  */
/*           PRIVATE MEMBER FUNCTIONS               */
/*                                                  */
/****************************************************/


/*
 * InitResources
 *
 * Private
 *
 * Allocates the Isochronous resources necessary to start an isochronous
 * streaming operation.
 *
 * Called by:
 *   StartImageCapture
 *   StartImageAcquisition
 */

BOOL C1394Camera::InitResources()
{
	ISOCH_ALLOCATE_BANDWIDTH AllocBandwidth;
	ISOCH_ALLOCATE_CHANNEL AllocChannel;
	ISOCH_ALLOCATE_RESOURCES AllocResources;
	ULONG ulRet,ulData;

	DllTrace(DLL_TRACE_ENTER,"ENTER InitResources\n");

	// Tripping either of these would indicate some sort of broken invariant
	// As they are also checked by StartImage*, which are the only functions
	// that call this.

	// however, we will check them anyway

	if (!m_pName)
	{
		DllTrace(DLL_TRACE_ERROR,"InitResources: Error: No camera selected\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitResources (FALSE)\n");
		return FALSE;
	}

	if(m_hDeviceAcquisition || m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"InitResources: Error: Camera is busy, stop image acquisition first\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT InitResources (FALSE)\n");
		return FALSE;
	}

	m_hBandwidth = m_hResource = NULL;
	m_lChannel = -1;

	// what the hell were these doing here?
	// the only mutators for the arguments to
	// these functions are the funcations themselves
	//SetVideoFormat(m_videoFormat);
	//SetVideoMode(m_videoMode);
	//SetVideoFrameRate(m_videoFrameRate);

//	AllocateBandwidth();
	ZeroMemory(&AllocBandwidth, sizeof(ISOCH_ALLOCATE_BANDWIDTH));
	AllocBandwidth.nMaxBytesPerFrameRequested = m_maxBytes;
	AllocBandwidth.fulSpeed = m_maxSpeed;
	ulRet = IsochAllocateBandwidth(NULL, m_pName, &AllocBandwidth);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"InitResources: Error %08x on IsochAllocateBandwidth\n",ulRet);
		goto exit_InitResources;
	}
	DllTrace(DLL_TRACE_VERBOSE,"InitResources: BytesPerFrameAvailable: %d\n",
		AllocBandwidth.BytesPerFrameAvailable);
	DllTrace(DLL_TRACE_VERBOSE,"InitResources: SpeedSelected: %d\n",
		AllocBandwidth.SpeedSelected);

	m_hBandwidth = AllocBandwidth.hBandwidth;

//	AllocateChannel();
	ZeroMemory(&AllocChannel,sizeof(ISOCH_ALLOCATE_CHANNEL));
	AllocChannel.nRequestedChannel = m_node;

	ulRet = IsochAllocateChannel(NULL, m_pName, &AllocChannel);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"InitResources: Error %08x on IsochAllocateChannel\n",ulRet);
		goto exit_InitResources;
	}
	DllTrace(DLL_TRACE_VERBOSE,"InitResources: Channel: %d\n",
		AllocChannel.Channel);
	DllTrace(DLL_TRACE_VERBOSE,"InitResources: ChannelsAvailable: %d\n",
		AllocChannel.ChannelsAvailable);

	m_lChannel = AllocChannel.Channel;

//	AllocateResources();
	ZeroMemory(&AllocResources,sizeof(ISOCH_ALLOCATE_RESOURCES));
	AllocResources.fulSpeed = m_maxSpeed;
	// BUGCHECK
	// OCHI compliant controllers are required to have RESOURCE_STRIP..., but maybe we should check anyway
	AllocResources.fulFlags = RESOURCE_USED_IN_LISTENING|RESOURCE_STRIP_ADDITIONAL_QUADLETS; // Listen Mode
	AllocResources.nChannel = m_lChannel;
	AllocResources.nMaxBytesPerFrame = m_maxBytes;
	AllocResources.nNumberOfBuffers = 10;
	AllocResources.nMaxBufferSize = m_maxBufferSize;
	AllocResources.nQuadletsToStrip = 1;
	ulRet = IsochAllocateResources(NULL, m_pName, &AllocResources);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"InitResources: Error %08x on IsochAllocateResources\n",ulRet);
		goto exit_InitResources;
	}
	DllTrace(DLL_TRACE_VERBOSE,"InitResources: hResource: %08x\n",
		AllocResources.hResource);

	m_hResource = AllocResources.hResource;

	//SetChannelSpeed();
	ulData = (ULONG) m_lChannel;
	ulData <<= 4;
	ulData += (m_maxSpeed/2);
	ulData <<= 24;
	ulRet = WriteQuadlet(0x60c, ulData);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"InitResources: Error %08x on WriteQuadlet(0x060C)\n",ulRet);
		goto exit_InitResources;
	}

	DllTrace(DLL_TRACE_EXIT,"EXIT InitResources (TRUE)\n");

	return TRUE;

exit_InitResources:

	// this is the common exit point for failed resource allocation
	// free them in reverse order

	if(m_hResource)
	{
		IsochFreeResources(NULL,m_pName,m_hResource);
		m_hResource = NULL;
	}

	if(m_lChannel != -1)

	{
		IsochFreeChannel(NULL,m_pName,m_lChannel);
		m_lChannel = -1;
	}

	if(m_hBandwidth)
	{
		IsochFreeBandwidth(NULL,m_pName,m_hBandwidth);
		m_hBandwidth = NULL;
	}

	DllTrace(DLL_TRACE_EXIT,"EXIT InitResources (FALSE)\n");
	return FALSE;
}


/*
 * FreeResources
 *
 * Private
 *
 * Frees the isochronous bandwidth, channel, and resource handle
 * that are allocated by InitResources
 *
 * Called by:
 *   StopImageCapture
 *   StopImageAcquisition
 */

BOOL C1394Camera::FreeResources()
{
	ULONG ulRet;
	BOOL bRet = TRUE;
	if (ulRet = IsochFreeResources(NULL, m_pName, m_hResource))
	{
		DllTrace(DLL_TRACE_ERROR,"FreeResources: Error %08x on IsochFreeResources\n",ulRet);
		return FALSE;
	}
	if (ulRet = IsochFreeChannel(NULL, m_pName, m_lChannel))
	{
		DllTrace(DLL_TRACE_ERROR,"FreeResources: Error %08x on IsochFreeChannel\n",ulRet);
		return FALSE;
	}
	if (IsochFreeBandwidth(NULL, m_pName, m_hBandwidth))
	{
		DllTrace(DLL_TRACE_ERROR,"FreeResources: Error %08x on IsochFreeBandwidth\n",ulRet);
		return FALSE;
	}
	return TRUE;
}


/*
 * InquireVideoFormats
 *
 * Private
 *
 * Reads the format register and fills in our stuff
 * CAM_ERROR indicates a failed ReadRegister
 */

BOOL C1394Camera::InquireVideoFormats()
{
	ULONG temp,value,format;
	DWORD dwRet;

	// inquire video formats
	if(dwRet = ReadQuadlet(0x100,&value))
	{
		DllTrace(DLL_TRACE_ERROR,"InquireVideoFormats, Error %08x on ReadRegister(0x100)\n",dwRet);
		return FALSE;
	}

	// temp will be our shifty bitmask
	temp = 0x80000000;
	for (format=0; format<8; format++)
	{
		m_bxAvailableFormats[format] = ((value & temp) != 0);
		if(format < 3 && !m_bxAvailableFormats[format])
			DisableVideoFormat(format); 
		temp >>= 1;
	}

	return TRUE;
}


/*
 * DisableVideoFormat
 *
 * Private
 *
 * Clears out all the videoFlags for a given format
 *
 * Note: possibly deprecated by m_bxAvailableFormats
 * Only called by InquireVideoFormats
 */

void C1394Camera::DisableVideoFormat(int format)
{
	int mode, rate;
	for (mode=0; mode<8; mode++)
	{
		m_bxAvailableModes[format][mode] = false;

		for (rate=0; rate<6; rate++)
			m_videoFlags[format][mode][rate] = false;
	}
}


/*
 * InquireVideoModes
 *
 * Private
 *
 * Reads the mode registers and fills in our stuff
 * CAM_ERROR indicates a failed ReadRegister
 */

BOOL C1394Camera::InquireVideoModes()
{
	ULONG value,temp,format,mode;
	DWORD dwRet;

	for (format=0; format<3; format++)
	{
		if (m_bxAvailableFormats[format])
		{
			// inquire video mode for current format
			if(dwRet = ReadQuadlet(0x180+format*4,&value))
			{
				DllTrace(DLL_TRACE_ERROR,"InquireVideoModes, Error %08x on ReadRegister(%03x)\n",dwRet,0x180+format*4);
				return FALSE;
			}
			temp = 0x80000000;
			for (mode=0; mode<8; mode++)
			{
				m_bxAvailableModes[format][mode] = ((value & temp) != 0);

				if (!(value & temp)) DisableVideoMode(format, mode); 
				temp >>= 1;
			}
		}
	}
	return TRUE;
}


/*
 * DisableVideoMode
 *
 * Private
 *
 * Clears out all the videoFlags for a given format and mode
 *
 * Note: possibly deprecated by m_bxAvailableModes
 * Only called by InquireVideoModes
 */

void C1394Camera::DisableVideoMode(int format, int mode)
{
	int rate;
	for (rate=0; rate<6; rate++)
		m_videoFlags[format][mode][rate] = false;
}


/*
 * InquireVideoRates
 *
 * Private
 *
 * Reads the rate registers and fills in our stuff
 * CAM_ERROR indicates a failed ReadRegister
 */

BOOL C1394Camera::InquireVideoRates()
{
	ULONG format, mode, rate, value, temp;
	DWORD dwRet;

	for (format=0; format<3; format++)
		for (mode=0; mode<8; mode++)
			if (m_videoFlags[format][mode][0])
			{
				// inquire video mode for current format
				if(dwRet = ReadQuadlet(0x200+format*32+mode*4,&value))
				{
					DllTrace(DLL_TRACE_ERROR,"InquireVideoRates, Error %08x on ReadRegister(%03x)\n",dwRet,0x200+format*32+mode*4);
					return FALSE;
				}
				//value = ReadQuadlet(0x200+format*32+mode*4);
				temp = 0x80000000;
				for (rate=0; rate<6; rate++)
				{
					if (!(value & temp)) m_videoFlags[format][mode][rate] = false; 
					temp >>= 1;
				}
			}
	return TRUE;
}


/*
 * UpdateParameters
 *
 * Private
 *
 * Helper function that tweaks m_width, m_height, m_maxBytes, m_maxBufferSize
 * whenever the video settings change.
 *
 * Doing this right away seems just as shady as writing the registers right away
 * but we will leave it for now
 *
 * Called by SetVideo*
 */

void C1394Camera::UpdateParameters()
{
	if (m_videoFormat != 7)
	{
		m_maxBytes = 4 * (tableQuadletsPerPacket[m_videoFormat][m_videoMode][m_videoFrameRate]);
		m_width = tableWidth[m_videoFormat][m_videoMode];
		m_height = tableHeight[m_videoFormat][m_videoMode];
		m_maxBufferSize = tableMaxBufferSize[m_videoFormat][m_videoMode];
	} else {
		m_maxBytes = m_controlSize.m_bytesPacket;
		m_width = m_controlSize.m_width;
		m_height = m_controlSize.m_height;
		m_maxBufferSize = m_controlSize.m_bytesFrameLow;
	}
}


/*
 * OneShot
 *
 * Private
 *
 * Writes to the ONE_SHOT register, telling the camera to transmit one frame
 *
 * Returns TRUE/FALSE as they would typically reflect good/bad
 *
 * Comments:
 *   This function is a wrapper for an otherwise simple procedure.  Maybe it should
 *   be inlined and just return whatever the WriteRegister returns
 *
 *   This function is currently not used
 */

BOOL C1394Camera::OneShot()
{
	ULONG ulRet;
	BOOL retval = TRUE;

	DllTrace(DLL_TRACE_ENTER,"ENTER OneShot\n");
	if((ulRet = WriteQuadlet(0x61c,0x80000000)))
	{
		DllTrace(DLL_TRACE_ERROR,"OneShot: error %08x on WriteQuadlet(0x61c)\n",ulRet);
		retval = FALSE;
	}

	DllTrace(DLL_TRACE_EXIT,"EXIT OneShot (%d)\n",retval);
	return retval;
}


// these video stream functions should probably leave
// they are more one-liners that would pop up error boxes

void C1394Camera::StartVideoStream()
{
	DllTrace(DLL_TRACE_ENTER,"ENTER StartVideoStream\n");
	if(WriteQuadlet(0x614,0x80000000) < 0)
		DllTrace(DLL_TRACE_ERROR,"Error Starting Video Stream");
	DllTrace(DLL_TRACE_EXIT,"EXIT StartVideoStream\n");
}


void C1394Camera::StopVideoStream()
{
	DllTrace(DLL_TRACE_ENTER,"ENTER StopVideoStream\n");
	if(WriteQuadlet(0x614,0) < 0)
		DllTrace(DLL_TRACE_ERROR,"Error Stopping Video Stream");
	DllTrace(DLL_TRACE_EXIT,"EXIT StopVideoStream\n");
}


/**************************************************/
/*                                                */
/*           DELETED MEMBER FUNCTIONS             */
/*                                                */
/*  These Functions have become unnecessary for   */
/*  one reason or another.  They will eventually  */
/*  be nuked altogether.                          */
/**************************************************/

/*

void C1394Camera::Speed(unsigned char value)
{
	DllTrace(DLL_TRACE_ENTER,"ENTER Speed\n");
	unsigned long data = value;
	data <<= 24;
	
	if(WriteRegisterUL(m_pName,0x60c,data) < 0)
		ERRORBOX("Problem Setting Speed");
}

// set channel and speed
void C1394Camera::SetChannelSpeed()
{
	int value = m_lChannel;			// set channel
	value <<= 4;
	value += (int) (m_maxSpeed/2);			// set speed
	value <<= 24;
	WriteQuadlet(0x60c, value);
}

BOOL C1394Camera::AllocateBandwidth()
{
	ISOCH_ALLOCATE_BANDWIDTH AllocBandwidth;
	ULONG ulRet;

	ZeroMemory(&AllocBandwidth, sizeof(ISOCH_ALLOCATE_BANDWIDTH));
	AllocBandwidth.nMaxBytesPerFrameRequested = m_maxBytes;
	AllocBandwidth.fulSpeed = m_maxSpeed;
	ulRet = IsochAllocateBandwidth(NULL, m_pName, &AllocBandwidth);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"AllocateBandwidth: Error %08x on IsochAllocateBandwidth\n",ulRet);
		return FALSE;
	}
	DllTrace(DLL_TRACE_VERBOSE,"AllocateBandwidth: BytesPerFrameAvailable: %d\n",
		AllocBandwidth.BytesPerFrameAvailable);
	DllTrace(DLL_TRACE_VERBOSE,"AllocateBandwidth: SpeedSelected: %d\n",
		AllocBandwidth.SpeedSelected);

	m_hBandwidth = AllocBandwidth.hBandwidth;
	return TRUE;
		
}

BOOL C1394Camera::AllocateChannel()
{
	ISOCH_ALLOCATE_CHANNEL AllocChannel;
	ULONG ulRet;

	ZeroMemory(&AllocChannel,sizeof(ISOCH_ALLOCATE_CHANNEL));
	AllocChannel.nRequestedChannel = m_node;

	ulRet = IsochAllocateChannel(NULL, m_pName, &AllocChannel);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"AllocateChannel: Error %08x on IsochAllocateChannel\n",ulRet);
		return FALSE;
	}
	DllTrace(DLL_TRACE_VERBOSE,"AllocateChannel: Channel: %d\n",
		AllocChannel.Channel);
	DllTrace(DLL_TRACE_VERBOSE,"AllocateChannel: ChannelsAvailable: %d\n",
		AllocChannel.ChannelsAvailable);

	m_lChannel = AllocChannel.Channel;
	return TRUE;
}

BOOL C1394Camera::AllocateResources()
{
	ISOCH_ALLOCATE_RESOURCES AllocResources;
	ULONG ulRet;

	ZeroMemory(&AllocResources,sizeof(ISOCH_ALLOCATE_RESOURCES));
	AllocResources.fulSpeed = m_maxSpeed;
	// BUGCHECK
	// OCHI compliant controllers are required to have RESOURCE_STRIP..., but maybe we should check anyway
	AllocResources.fulFlags = RESOURCE_USED_IN_LISTENING|RESOURCE_STRIP_ADDITIONAL_QUADLETS; // Listen Mode
	AllocResources.nChannel = m_lChannel;
	AllocResources.nMaxBytesPerFrame = m_maxBytes;
	AllocResources.nNumberOfBuffers = 10;
	AllocResources.nMaxBufferSize = m_maxBufferSize;
	AllocResources.nQuadletsToStrip = 1;
	ulRet = IsochAllocateResources(NULL, m_pName, &AllocResources);
	if(ulRet)
	{
		DllTrace(DLL_TRACE_ERROR,"AllocateResources: Error %08x on IsochAllocateResources\n",ulRet);
		return FALSE;
	}
	DllTrace(DLL_TRACE_VERBOSE,"AllocateResources: hResource: %08x\n",
		AllocResources.hResource);

	m_hResource = AllocResources.hResource;
	return TRUE;
}
*/

// deprecated, this function should not be necessary any more
/*
void C1394Camera::ResetLink(bool root)
{
	ULONG flags;
	//GetDeviceName();
	if (root)
		flags = 2;
	else 
		flags = 0;
	if (BusReset(NULL, m_pName, flags))
		ERRORBOX("Problem with Bus Reset");
}
*/
