///////////////////////////////////////////////////////////////////////////////
//
// 1394CamCap.cpp
//
// source for C1394Camera::
//  - StartImageCapture
//  - CaptureImage
//  - StopImageCapture
//
// Based on the following functions originally by Iwan Ulrich from isochapi.c:
//
//		IsochStartImageCapture
//		IsochAttachBufferCapture
//		IsochWaitImageCapture
//		IsochStopImageCapture
//
//	Copyright 1/2000
// 
//	Iwan Ulrich
//	Robotics Institute
//	Carnegie Mellon University
//	Pittsburgh, PA
//
//  Copyright 3/2002 - 7/2002
//
//  Christopher Baker
//  Robotics Institute
//  Carnegie Mellon University
//  Pittsburgh, PA
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//////////////////////////////////////////////////////////////////////

#include <windows.h>
#include <string.h>
#include <mmsystem.h>
#include <stdio.h>
#include "1394Camera.h"
#include "pch.h"

/* The Model
 * 
 * - Only one image buffer will exist, and is indexed by m_pFirstBuffer
 *
 * - On a call to CaptureImage, that buffer will be attached, filled, and
 *     detached.  This has the advantage of not using the CPU between calls
 *     to CaptureImage, unlike the Acquisition Method, which will be continuously
 *     using the CPU to some extent between Acquires
 *
 * - It is very likely that two consecutive calls to CaptureImage will miss
 *     at least one frame from the camera, so the percieved frame rate will be
 *     somewhat less than that which is set.
 */

/*
 * StartImageCapture
 *
 * Arguments:
 *   <none>
 *
 * Return Values:
 *   0 : success
 *   nonzero : some error ;)
 *
 * Reads Member Vars:
 *   m_cameraInitialized
 *   m_videoFlags
 *   m_videoFormat
 *   m_videoMode
 *   m_videoFrameRate
 *   m_hDeviceAcquisition
 *   m_hDeviceCapture
 *   m_pFirstBuffer
 *
 * Mutates Member Vars:
 *   m_hDeviceCapture
 *   m_pFirstBuffer
 *   
 * Comments:
 *   Allocates System Memory, Allocates Isochronous Channel, Bandwidth, and "Resources"
 */

int C1394Camera::StartImageCapture()
{
	PACQUISITION_BUFFER				pAcqBuffer;
	BOOL							b;	
	DWORD							dwRet, dwBytesRet;
	ISOCH_LISTEN					isochListen;
	OVERLAPPED						ListenOverlapped;						
	int return_value = 0;

	DllTrace(DLL_TRACE_ENTER,"ENTER StartImageCapture\n");

	if(!m_cameraInitialized)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Call InitCamera() first.\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (%d)",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	if(!(m_videoFlags[m_videoFormat][m_videoMode][m_videoFrameRate]))
	{
		DllTrace(DLL_TRACE_ERROR,
				"StartImageCapture: Current Video Settings (%d,%d,%d) are not Supported\n",
				m_videoFormat,
				m_videoMode,
				m_videoFrameRate);
		DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (%d)\n",CAM_ERROR_INVALID_VIDEO_SETTINGS);
		return CAM_ERROR_INVALID_VIDEO_SETTINGS;
	}

	if(m_hDeviceAcquisition != NULL)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: The Camera is already acquiring images, call StopImageAcquisition first\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (%d)\n",CAM_ERROR_BUSY);
		return CAM_ERROR_BUSY;
	}

	if(m_hDeviceCapture != NULL)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: The Camera is currently capturing images, call StopImageCapture first\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (%d)\n",CAM_ERROR_BUSY);
		return CAM_ERROR_BUSY;
	}

	if(!InitResources())
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: InitResources Failed\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (%d)\n",CAM_ERROR_INSUFFICIENT_RESOURCES);
		return CAM_ERROR_INSUFFICIENT_RESOURCES;
	}

	//////////////////////////////////////////
	// allocate and set up the frame buffer //
	//////////////////////////////////////////

	DllTrace(DLL_TRACE_CHECK,"StartImageCapture: Initializing Buffer...\n");

	// allocate the buffer header (stores data about a buffer)
	pAcqBuffer = (PACQUISITION_BUFFER) GlobalAlloc(LPTR,sizeof(ACQUISITION_BUFFER));
	if(!pAcqBuffer)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Error Allocating AcqBuffer\n");
		return_value = ERROR_OUTOFMEMORY;
		goto Exit_1394StartImageCapture;
	}

	// stuff it in as the First (and only) buffer
	m_pFirstBuffer = pAcqBuffer;

	// allocate the actual frame buffer
	// the buffer passed to ATTACH_BUFFER must be aligned on a page boundary
	// thus, allocate an extra 4K and generate a pointer to the first page boundary
	pAcqBuffer->pDataBuf = (unsigned char *)GlobalAlloc(LPTR,m_maxBufferSize + 4096);
	if(!pAcqBuffer->pDataBuf)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Error Allocating Data Buffer\n");
		return_value = ERROR_OUTOFMEMORY;
		goto Exit_1394StartImageCapture;
	}

	// point pFrameStart at the first page boundary
	pAcqBuffer->pFrameStart = pAcqBuffer->pDataBuf + (4096 - (((int)(pAcqBuffer->pDataBuf)) & 0xfff));

	// set the index (mostly for debugging purposes)
	pAcqBuffer->index = 0;

	// give the overlapped structure an event
	pAcqBuffer->overLapped.hEvent = CreateEvent(NULL,TRUE,FALSE,NULL);
	if(pAcqBuffer->overLapped.hEvent == INVALID_HANDLE_VALUE)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Error Creating Overlapped Event\n");
		return_value = GetLastError();
		goto Exit_1394StartImageCapture;
	}

	// allocate the AttachBuffers struct
	pAcqBuffer->pIsochAttachBuffers = (PISOCH_ATTACH_BUFFERS) GlobalAlloc(LPTR,sizeof(ISOCH_ATTACH_BUFFERS));
	if(!pAcqBuffer->pIsochAttachBuffers)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Error Allocating Isoch Attach Buffers\n");
		return_value = ERROR_OUTOFMEMORY;
		goto Exit_1394StartImageCapture;
	}

	// fill in the AttachBuffers struct
	pAcqBuffer->pIsochAttachBuffers->hResource = m_hResource;
	pAcqBuffer->pIsochAttachBuffers->nNumberOfDescriptors = 1;
	pAcqBuffer->pIsochAttachBuffers->ulBufferSize = m_maxBufferSize;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].fulFlags = DESCRIPTOR_SYNCH_ON_SY;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].ulLength = m_maxBufferSize;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].nMaxBytesPerFrame = m_maxBytes;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].ulSynch = 1;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].ulTag = 0;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].CycleTime.CL_CycleOffset = 0;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].CycleTime.CL_CycleCount = 0;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].CycleTime.CL_SecondCount = 0;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].bUseCallback = TRUE;
	pAcqBuffer->pIsochAttachBuffers->R3_IsochDescriptor[0].bAutoDetach = TRUE;

	DllTrace(DLL_TRACE_VERBOSE,"StartImageCapture: Allocated buffer\n");

	// open our long term device handle
	m_hDeviceCapture = OpenDevice(m_pName, TRUE);

	if(m_hDeviceCapture == INVALID_HANDLE_VALUE)
	{
		DllTrace(DLL_TRACE_ERROR,"StartImageCapture: error opening device (%s)\n",m_pName);
		m_hDeviceCapture = NULL;
		return_value = GetLastError();
		goto Exit_1394StartImageCapture;
	}

	// attach our buffer

	// reset the overlapped event
	ResetEvent(m_pFirstBuffer->overLapped.hEvent);

	// and attach the buffer
	DllTrace(DLL_TRACE_VERBOSE,"CaptureImage: Attaching buffer: index:%d, size:%d, R3qpp:%d, R3length: %d, FrameStart:%08x\n",
		m_pFirstBuffer->index,
		m_pFirstBuffer->pIsochAttachBuffers->ulBufferSize,
		m_pFirstBuffer->pIsochAttachBuffers->R3_IsochDescriptor->nMaxBytesPerFrame,
		m_pFirstBuffer->pIsochAttachBuffers->R3_IsochDescriptor->ulLength,
		m_pFirstBuffer->pFrameStart);

	b = DeviceIoControl(
			m_hDeviceCapture,
			IOCTL_ATTACH_BUFFER,
			m_pFirstBuffer->pIsochAttachBuffers,
			sizeof(ISOCH_ATTACH_BUFFERS),
			m_pFirstBuffer->pFrameStart,
			m_pFirstBuffer->pIsochAttachBuffers->ulBufferSize,
			&dwBytesRet,
			&m_pFirstBuffer->overLapped
			);

	dwRet = GetLastError();
	if ((dwRet != ERROR_IO_PENDING) && (dwRet != ERROR_SUCCESS)) 
	{
		DllTrace(DLL_TRACE_ERROR,"CaptureImage: Error %08x while attaching Buffer %d\n",dwRet,m_pFirstBuffer->index);
		DllTrace(DLL_TRACE_EXIT,"EXIT CaptureImage (%d)\n",dwRet);
		return dwRet;
	}

	// listen							
	ListenOverlapped.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
						
	isochListen.hResource = m_hResource;
	isochListen.fulFlags = 0;
	isochListen.StartTime.CL_SecondCount = 0;
	isochListen.StartTime.CL_CycleCount = 0;
	isochListen.StartTime.CL_CycleOffset = 0;
						
	b = DeviceIoControl( m_hDeviceCapture,
						IOCTL_ISOCH_LISTEN,
						&isochListen,
						sizeof(ISOCH_LISTEN),
						NULL,
						0,
						&dwBytesRet,
						&ListenOverlapped
						);

	if(!b)
	{
		dwRet = GetLastError();
		if(dwRet != ERROR_IO_PENDING && dwRet != ERROR_SUCCESS)
		{
			DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Error %08x on IOCTL_ISOCH_LISTEN\n",dwRet);
			DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (-1)");
			return dwRet;
		} else {
			GetOverlappedResult(m_hDeviceCapture,&ListenOverlapped,&dwBytesRet,TRUE);
		}
	}
				
	StartVideoStream();

	// block-wait on the overlapped result from the next buffer

	if(!GetOverlappedResult(m_hDeviceCapture, &m_pFirstBuffer->overLapped, &dwBytesRet, TRUE))
	{
		dwRet = GetLastError();
		DllTrace(DLL_TRACE_ERROR,"CaptureImage: Error %08x while Getting Overlapped Result\n",dwRet);
		DllTrace(DLL_TRACE_EXIT,"EXIT CaptureImage (%d)\n",dwRet);
		return dwRet;
	}

	CloseHandle(ListenOverlapped.hEvent);		

	DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (0)\n");
	return 0;

/////////////////////////////////////
// Exit_1394StartImageCapture: //
/////////////////////////////////////

Exit_1394StartImageCapture:

    // we go here if something breaks and we need to clean up anything we've allocated thus far
	DllTrace(DLL_TRACE_ERROR,"StartImageCapture: Starting Error Cleanup\n");
		
	if(m_pFirstBuffer)
	{
		DllTrace(DLL_TRACE_VERBOSE,"StartImageCapture: Removing buffer %d\n",m_pFirstBuffer->index);

		if(m_pFirstBuffer->overLapped.hEvent)
			CloseHandle(m_pFirstBuffer->overLapped.hEvent);
		if(m_pFirstBuffer->pDataBuf)
			GlobalFree(m_pFirstBuffer->pDataBuf);
		if(m_pFirstBuffer->pIsochAttachBuffers)
			GlobalFree(m_pFirstBuffer->pIsochAttachBuffers);

	}

	m_pFirstBuffer = NULL;

	StopVideoStream();
	FreeResources();

	DllTrace(DLL_TRACE_EXIT,"EXIT StartImageCapture (%d)\n",return_value);
	return return_value;
}


/*
 * CaptureImage
 *
 * Arguments:
 *   <none>
 *
 * Return Values:
 *   0 : success
 *   nonzero : some error ;)
 *
 * Reads Member Vars:
 *   m_hDeviceCapture
 *   m_pFirstBuffer
 *
 * Mutates Member Vars:
 *   m_pData
 *
 * Comments:
 *   Attaches a buffer, waits for it to be filled, then returns.  The expected block
 *     time for this call will range from 1-2 sample period(s).
 */

int C1394Camera::CaptureImage()
{
	DWORD							dwRet, dwBytesRet;
	BOOL							b;

	DllTrace(DLL_TRACE_ENTER,"ENTER CaptureImage\n");

	if(!m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"CaptureImage: Not Capturing Images: Call StartImageCapture First");
		DllTrace(DLL_TRACE_EXIT,"EXIT CaptureImage (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	// attach our buffer

	// reset the overlapped event
	ResetEvent(m_pFirstBuffer->overLapped.hEvent);

	// and attach the buffer
	DllTrace(DLL_TRACE_VERBOSE,"CaptureImage: Attaching buffer: index:%d, size:%d, R3qpp:%d, R3length: %d, FrameStart:%08x\n",
		m_pFirstBuffer->index,
		m_pFirstBuffer->pIsochAttachBuffers->ulBufferSize,
		m_pFirstBuffer->pIsochAttachBuffers->R3_IsochDescriptor->nMaxBytesPerFrame,
		m_pFirstBuffer->pIsochAttachBuffers->R3_IsochDescriptor->ulLength,
		m_pFirstBuffer->pFrameStart);

	b = DeviceIoControl(
			m_hDeviceCapture,
			IOCTL_ATTACH_BUFFER,
			m_pFirstBuffer->pIsochAttachBuffers,
			sizeof(ISOCH_ATTACH_BUFFERS),
			m_pFirstBuffer->pFrameStart,
			m_pFirstBuffer->pIsochAttachBuffers->ulBufferSize,
			&dwBytesRet,
			&m_pFirstBuffer->overLapped
			);

	dwRet = GetLastError();
	if ((dwRet != ERROR_IO_PENDING) && (dwRet != ERROR_SUCCESS)) 
	{
		DllTrace(DLL_TRACE_ERROR,"CaptureImage: Error %08x while attaching Buffer %d\n",dwRet,m_pFirstBuffer->index);
		DllTrace(DLL_TRACE_EXIT,"EXIT CaptureImage (%d)\n",dwRet);
		return dwRet;
	}
	// block-wait on the overlapped result from the next buffer

	if(!GetOverlappedResult(m_hDeviceCapture, &m_pFirstBuffer->overLapped, &dwBytesRet, TRUE))
	{
		dwRet = GetLastError();
		DllTrace(DLL_TRACE_ERROR,"CaptureImage: Error %08x while Getting Overlapped Result\n",dwRet);
		DllTrace(DLL_TRACE_EXIT,"EXIT CaptureImage (%d)\n",dwRet);
		return dwRet;
	}

	DllTrace(DLL_TRACE_VERBOSE,"CaptureImage: Snagged Frame: Index = %d\n",m_pFirstBuffer->index);

	// update the public data pointer
	m_pData = m_pFirstBuffer->pFrameStart;
	DllTrace(DLL_TRACE_EXIT,"EXIT CaptureImage (0)\n");
	return CAM_SUCCESS;
}


/*
 * StopImageAcquisition
 *
 * Arguments:
 *   <none>
 *
 * Return Values:
 *   0 : success
 *   nonzero : some error ;)
 *
 * Reads Member Vars:
 *   m_hDeviceCapture
 *   m_pFirstBuffer
 *
 * Mutates Member Vars:
 *   m_hDeviceCapture
 *   m_pFirstBuffer
 *
 * Comments:
 *   Frees all resources (memory, events, handles, etc.) allocated by
 *     StartImageCapture
 */

int C1394Camera::StopImageCapture()
{
	DWORD							dwBytesRet;
	ISOCH_STOP						isochStop;
	OVERLAPPED						StopOverlapped;
	BOOL							b;
	
	DllTrace(DLL_TRACE_ENTER,"ENTER StopImageCapture\n");

	if(!m_hDeviceCapture)
	{
		DllTrace(DLL_TRACE_ERROR,"StopImageCapture: Not Capturing Images: Call StartImageCapture First\n");
		DllTrace(DLL_TRACE_EXIT,"EXIT StopImageCapture (%d)\n",CAM_ERROR_NOT_INITIALIZED);
		return CAM_ERROR_NOT_INITIALIZED;
	}

	// stop listening
	StopOverlapped.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	isochStop.hResource = m_hResource;
	isochStop.fulFlags = 0;
			
	b = DeviceIoControl(m_hDeviceCapture,
				IOCTL_ISOCH_STOP,
				&isochStop,
				sizeof(ISOCH_STOP),
				NULL,
				0,
				&dwBytesRet,
				&StopOverlapped
				);
	
	if(!GetOverlappedResult(m_hDeviceCapture,&StopOverlapped,&dwBytesRet,TRUE))
		// some error, but we really can't do anything about it now
		DllTrace(DLL_TRACE_ERROR,"StopImageCapture: Error %08x on Overlapped Isoch Stop",GetLastError());
	
	CloseHandle(StopOverlapped.hEvent);
		
	// free up all resources
	CloseHandle(m_hDeviceCapture);
	m_hDeviceCapture = NULL;

	DllTrace(DLL_TRACE_CHECK,"StopImageCapture: Cleaning up Buffer\n");

	// free the event
	CloseHandle(m_pFirstBuffer->overLapped.hEvent);
	// free the frame buffer
	GlobalFree(m_pFirstBuffer->pDataBuf);
	// free the Attach Buffers struct
	GlobalFree(m_pFirstBuffer->pIsochAttachBuffers);
	// free the buffer header
	GlobalFree(m_pFirstBuffer);

	m_pFirstBuffer = NULL;

	StopVideoStream();
	FreeResources();
	DllTrace(DLL_TRACE_EXIT,"EXIT StopImageCapture (0)\n");
	return 0;
}
