//  1394Camera.h: interface for the C1394Camera class.
//
//	Version 5.2
//
//	Copyright 5/2000
// 
//	Iwan Ulrich
//	Robotics Institute
//	Carnegie Mellon University
//	Pittsburgh, PA
//
//  Copyright 3/2002
//
//  Christopher Baker
//  Robotics Institute
//  Carnegie Mellon University
//  Pittsburgh, PA
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//////////////////////////////////////////////////////////////////////

#ifndef __1394CAMERA_H__
#define __1394CAMERA_H__

#ifdef MY1394CAMERA_EXPORTS

// compiling library, reference
// private, potentially modified
// version of the header
#include "../inc/1394camapi.h"

#else

// using library, use global version
#include <1394camapi.h>

#endif // MY1394CAMERA_EXPORTS

#include "1394CameraControl.h"
#include "1394CameraControlTrigger.h"
#include "1394CameraControlSize.h"

class CAMAPI C1394Camera  
{
friend class C1394CameraControlSize;  

public:
	// constructor
	C1394Camera(HWND hWnd = NULL);
	// destructor
	~C1394Camera();

	// Selection/Control

	void InitCamera();
	int GetNode();
	void SelectCamera(int node);
	unsigned long GetVersion();
	int GetNumberCameras();
	int GetMaxSpeed();
	void ResetLink(bool root);
	int CheckLink();
	void InquireControlRegisters();
	void StatusControlRegisters();
	void WriteQuadlet(unsigned long address, unsigned long value);
	unsigned long ReadQuadlet(unsigned long address);

	// Video format/mode/rate

	int GetVideoFormat();
	void SetVideoFormat(int format);
	int GetVideoMode();
	void SetVideoMode(int mode);
	int GetVideoFrameRate();
	void SetVideoFrameRate(int rate);
	
	// Image Capture (1394CamCap.cpp)

	int StartImageCapture();
	int CaptureImage();
	int StopImageCapture();

	// Image Acquire (1394CamAcq.cpp)

	int StartImageAcquisition();
	int AcquireImage();
	int StopImageAcquisition();

	// Color Format Conversion (1394CamRGB.cpp)

	// convert data to standard: RGB, upper-left corner
	// based on video format/mode
	void getRGB(unsigned char *pBitmap);

	// same as getRGB, except data is returned in the
	// bottom-up, BGR format the MS calls a DIB
	void getDIB(unsigned char *pBitmap);

	// individual converters
	void YtoRGB(unsigned char *pBitmap);
	void Y16toRGB(unsigned char *pBitmap);
	void YUV411toRGB(unsigned char* pBitmap);
	void YUV422toRGB(unsigned char* pBitmap);
	void YUV444toRGB(unsigned char* pBitmap);

	// Control Wrappers

	void SetIris(int value);
	void SetFocus(int value);
	void SetZoom(int value);
	void SetBrightness(int value);
	void SetAutoExposure(int value);
	void SetSharpness(int value);
	void SetWhiteBalance(int u, int v);
	void SetHue(int value);
	void SetSaturation(int value);
	void SetGamma(int value);
	void SetShutter(int value);
	void SetGain(int value);

	// Control Members
	// Feature_Hi
	C1394CameraControl m_controlBrightness;
	C1394CameraControl m_controlAutoExposure;
	C1394CameraControl m_controlSharpness;
	C1394CameraControl m_controlWhiteBalance;
	C1394CameraControl m_controlHue;
	C1394CameraControl m_controlSaturation;
	C1394CameraControl m_controlGamma;
	C1394CameraControl m_controlShutter;
	C1394CameraControl m_controlGain;
	C1394CameraControl m_controlIris;
	C1394CameraControl m_controlFocus;
	C1394CameraControlTrigger m_controlTrigger;

	// Feature_Lo
	C1394CameraControl m_controlZoom;
	// there's also pan, tilt, and optical filter controls

	// Partial Scan Control class
	C1394CameraControlSize m_controlSize;

	// utility members
	int m_width;
	int m_height;
	bool m_linkChecked;
	bool m_cameraInitialized;
	char m_nameModel[256];
	char m_nameVendor[256];
	unsigned char* m_pData;
	bool m_videoFlags[3][8][6];		// [format][mode][rate]

private:
	void DisableVideoMode(int format, int mode);
	void InquireVideoRates();
	void InquireVideoModes();
	void DisableVideoFormat(int format);
	void InquireVideoFormats();
	void UpdateParameters();
	void AllocateResources();
	void AllocateChannel();
	void AllocateBandwidth();
	void GetDeviceName();
	void StopListen();
	void Listen();
	void OneShot();
	void SetChannelSpeed();
	void Speed(unsigned char value);
	void StopVideoStream();
	void StartVideoStream();
	void InitResources();
	void FreeResources();

	int m_videoFrameRate;
	int m_videoMode;
	int m_videoFormat;
	int m_maxBytes;
	int m_maxBufferSize;
	int m_nBuffers;
	int m_nDescriptors;
	int m_node;
	CAMERA_SPECIFICATION m_spec;
	bool m_allocateMemory;
	char* m_pName;
	DEVICE_DATA m_DeviceData;
	ISOCH_ALLOCATE_BANDWIDTH m_bandwidth;
	ISOCH_ALLOCATE_CHANNEL m_channel;
	ISOCH_ALLOCATE_RESOURCES m_resource;
	PISOCH_ATTACH_BUFFERS m_pIsochAttachBuffers;
	ISOCH_GET_IMAGE_PARAMS m_isochGetImageParams;

	ACQUISITION_BUFFER	m_acquisitionBuffer[6];
	ACQUISITION_BUFFER	*m_pFirstBuffer;
	ACQUISITION_BUFFER	*m_pLastBuffer;
	ACQUISITION_BUFFER	*m_pCurrentBuffer;
	HANDLE		m_hDeviceAcquisition;
	ULONG		m_acquisitionBufferSize;

	HANDLE		m_hDeviceCapture;
	ULONG		m_captureBufferSize;
	OVERLAPPED	m_captureOverlapped;

	ISOCH_LISTEN m_listen;
	ISOCH_STOP m_stop;
	HWND m_hWnd;
	ULONG m_maxSpeed;
};

#endif // __1394CAMERA_H__
