#!/usr/bin/perl -w

package Geckobot::Module;
use FileHandle;

$VERSION = "0.01_00";
use strict;

sub new {
  # Geckobot::Module is an abstract class.  Derive 
  # modules from it and let it do the dirty work. 
  # See the example modules.

  my $proto = shift ;
  my %params = @_;

  my $self = 
  {
    name => 'No name set',	# a name for this module
    usage => 'No usage set',	# text showing how invoked 
    descrip => 'No description set', # text of 'what it does'
    code => sub {},		# the action()
    enabled => 1,		# on by default
    weight => 0,		# process weight; high => fork
    regex => qr/^\000$/,	# triggered by this regex
  };
  for (my ($k, $v) = (each %params)) {
    $self->{$k} = $v;
  }

  bless $self, $proto;
  return $self;
}

sub status {
  # the generic console/status message reporting facility.
  my $self = shift;

  if (defined $self->{status}) {
    $self->{status}->(@_);
  } else {
    print $self->name, " @_\n";
  }
}

sub status_fn {
  # set the status function.  this allows the bot to control
  # all the output at the top level. good for logging, etc.
  my ($self, $code) = (shift, shift);
  if (defined $code) {
    $self->{status} = $code;
  }
  return $code;
}

sub try {
  # 'try' to do something.  
  # this also decides whether or not to farm
  # the operation out to another process, based
  # on the 'weight'.

  my $self = shift ;
  my $message = shift ;

  warn "no 'bot' passed to ",$self->name unless $self->{bot};

  return undef unless $self->enabled;
  return undef unless $self->matches($message);

  # if the 'matches' method already did the work,
  # don't do more.
  if (defined $message->{reply}) {
    return $self->name;
  }

  if ($self->weight >= 1) {
    # we've decided to fork (or later perhaps thread)
    my $pid = fork();

    if (!$pid) {
      # send of the child to do the work. 
      # note that clients that fork always
      # consider the event handled if they
      # pass the regex test above.

      my $result = $self->action($message);
      $message->{reply} = $result;

      return 'CHILD';

    } else {
      # the parent can just go on its merry way
      # since the child is now responsible for
      # the message.

      $self->status(2, $self->name, "child [$pid] spawned");
      $self->{proc}->{$pid} = $self->name;
      return 'FORKED';
    }

  } else {
    # staying within the current process. this is
    # mainly for very quick things, things that 
    # should remain atomic, and systems that can't
    # fork.

    my $result = $self->action($message);
    if ($result) {
      $message->{reply} = $result;
      return $self->name;
    }
  }

  return undef;
}

sub name {
  # you should be able to figure this one out.
  # this, like the attribute methods that follow, are 
  # set-if-value-present; name('foo') sets the name
  # to foo and returns 'foo', while name() just returns
  # 'foo'.

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $self->{'name'} = $arg;
  }
  return $self->{'name'};
}

sub priority {
  # helps define a partial order or execution on otherwise 
  # random rules.  

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $self->{'priority'} = $arg;
  }
  return $self->{'priority'};
}

sub code {
  # yield the code point for this action

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $self->{'code'} = $arg;
  } 

  return \$self->{'code'};
}

sub action {
  # the code entry point. this is what you might
  # call a virtual method.

  return 'No action defined. This should be overriden in the subclass.';
}

sub usage {
  # show how we're invoked

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $self->{'usage'} = $arg;
  }
  return $self->{'usage'};
}

sub matches {
  # check if we match. if we don't, give back undef,
  # since empty true matches can be interesting.

  my $self = shift;
  my $message = shift ;

  if ($message->{text} =~ /$self->{'match'}/) {
    my @args = ($1,$2,$3,$4,$5,$6,$7,$8,$9);
    @{$message->{args}} = @args;
    return $self->{name};
  } else {
    return undef;
  }
}

sub regex {
  # the regular expression we agree to match,
  # and consequently, the things we get when
  # the match occurs: $1,$2,$3, etc are passed
  # as an array to action() after this object's
  # regex() is matched().
  #
  # a method should probably just override
  # matches if they want to do fancier stuff.

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $self->{'match'} = $arg;
  }

  return $self->{'match'};
}

sub descrip {
  # a text description of what this object does, 
  # the attribution, or whatever

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $self->{'descrip'} = $arg;
  }
  return $self->{'descrip'};
}

sub enabled {
  # module instances can be turned on and off at
  # run-time, so this sets the objects state.

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $arg = 0 if lc($arg) eq 'false';
    $self->{'enabled'} = $arg;
  }
  return $self->{'enabled'};
}

sub weight {
  # the 'process weight'.  a weight of zero
  # means it should be handled in-line, because
  # it doesn't cost must.  higher weights mean
  # we may want to fork to handle it.

  my $self = shift;
  my $arg = shift;
  if (defined $arg) {
    $arg = 0 if lc($arg) eq 'false';
    $self->{'weight'} = $arg;
  }
  return $self->{'weight'};
}

'a true value';
