#!/usr/bin/perl -w

# Geckobot Mark II 
#
# (C) 1998 Kevin Lenzo 
#

use Geckobot;
use Net::IRC;

$VERSION = "0.01_00";
use strict;

package Geckobot::IRC ;
use POE qw(Wheel::SocketFactory Wheel::ReadWrite Driver::SysRW Filter::Line);

sub Geckobot::IRC::new {
  my $proto = shift ;
  my %params = @_;
  my $self = {};
  bless $self, $proto;

  print "IRC: new\n";

  $self->{irc} = new Net::IRC;
  $self->{connection} = $self->{irc}->newconn
    ('Server'   => $params{'Server'},
     'Network'  => $params{'Network'},
     'Port'     => $params{'Port'} || 6667,
     'Nick'     => $params{'Nick'} || "Infobot",
     'Ircname'  => $params{'Ircname'} || "Infobot",
     'Username' => $params{'Username'} || "Infobot"
    )
      or warn "Can't connect to IRC server.\n";

  $self->{bot} = $params{'bot'};
  $self->{session} = new POE::Session
      ($self, [qw(_start _stop cycle output on_msg)]);
  $self->{session}->option(trace => 1, default => 0);

  $self->add_handlers;
  $self->{irc}->timeout(0.2);
  $self->{irc}->do_one_loop;
  return $self ;
}

sub Geckobot::IRC::_start {
  my ($self, $kernel) = @_[OBJECT, KERNEL];
  $self->{running} = 1;
  $kernel->yield('cycle');
}

sub Geckobot::IRC::_stop {
  my ($self) = $_[OBJECT];
  $self->{running} = 0;
}

sub Geckobot::IRC::cycle {
  my ($self, $kernel) = @_[OBJECT, KERNEL];
  $self->{irc}->do_one_loop;
  $kernel->yield('cycle') if $self->{running};
}

sub Geckobot::IRC::dcc_msg {
    my ($self, $connection, $socket, $content) = @_;
    print ("-=$self->{connections}->{DCC}->{users}->{$socket}->{nick}=- $content");
    $connection->privmsg($socket, $content);
}

sub Geckobot::IRC::output {
    my ($self, $content, $message) = @_[OBJECT, ARG0, ARG1];
    print(">$message->{sender}< $content\n");
    $self->{connection}->privmsg($message->{sender}, $content);
}

sub Geckobot::IRC::notice {
    my ($self, $connection, $to, $content) = @_;

    print("-$to- $content");
    $connection->notice($to, $content);
}

sub Geckobot::IRC::on_connect {
    my ($self, $connection, $event) = @_;
    my $c = $self->{param}->{'join_channels'};
    my @channels;
    @channels = split /\s+/, $c if defined $c;

    @channels = ('#oznux');
    foreach (@channels) {
	print("Joining $_ ...\n");
	$connection->join("$_");
    }
}

sub Geckobot::IRC::on_init {
    my ($self, $connection, $event) = @_;
    my (@args) = ($event->args);
    shift (@args);
    foreach(@args) {
      print "** $_\n";
    }
}


sub Geckobot::IRC::on_version {
    my ($self, $connection, $event) = @_;
    my (@args) = ($event->args);
    shift (@args);

    $self->notice($connection, $event->{nick}, $self->{version});
    print("*** replied to VERSION from $event->{nick}");
}

sub Geckobot::IRC::on_part {
    my ($self, $connection, $event) = @_;
    my ($channel) = ($event->to)[0];

    print(sprintf "*** %s has left channel %s", 
		  $event->nick, $channel);
}

sub Geckobot::IRC::on_join {
    my ($self, $connection, $event) = @_;
    my ($channel) = ($event->to)[0];

    print(sprintf "*** %s (%s) has joined channel %s\n",
		 $event->nick, $event->userhost, $channel);
}

sub Geckobot::IRC::on_msg {
  my ($self, $kernel, $connection, $event) = @_[OBJECT, KERNEL,
						ARG0, ARG1];
    my ($nick) = $event->nick;
    my ($arg)  = $event->args;

    my $message = new Geckobot::Message(
					protocol => "IRC",
					sender => $nick, 
					receiver => $event->to,
					content => $arg, 
					type => 'private',
					session => $self->{session},
					bot => $self->{bot},
				       );
    print("[$nick] $arg\n");
    $kernel->call($self->{bot}->{session}, 'sense', $message);
    0;
}

sub Geckobot::IRC::on_public {
    my ($self, $connection, $event) = @_;

    my @to = $event->to;
    my $to = $to[0];
    my ($nick) = $event->nick;
    my @args = $event->args;
    my $args = $args[0];

    print("<".$event->nick().":$to>  $args");

    my $message = new Geckobot::Message(
					protocol => "IRC",
					connection => $self,
					sender => $nick, 
					receiver => $to,
					content => $args, 
					type => 'public',
					session => $self->{session},
					bot => $self->{bot},
				       );
    $self->{kernel}->call($self->{bot}->{session}, 'sense', $message);
}

sub Geckobot::IRC::on_chat {
    # We have a chat message coming in, Houston
    my ($self, $connection, $event) = @_;
    my ($nick) = $event->nick;
    my ($sock) = $event->to;
    my ($arg)  = $event->args;
 
    print("=$nick=  $arg");
    $self->dcc_msg($connection, $sock, "reply") ;
}

sub Geckobot::IRC::on_names {
    my ($self, $connection, $event) = @_;
    my (@list, $channel) = ($event->args);
    ($channel, @list) = splice @list, 2;

    my $nick;
    foreach $nick (split /\s+/, $list[0]) {
	if ($nick =~ s/^([+\@])//) {
	    $self->{connections}->{IRC}->{$channel}->{"+$1"}->{$nick}++;
	}
	$self->{connections}->{IRC}->{users}->{$nick}->{nick} = $nick;
	$self->{connections}->{IRC}->{$channel}->{$nick} = time();
    }


    print("$channel: @list");
}

# DCC SEND or CHAT request
sub Geckobot::IRC::on_dcc {
    my ($self, $connection, $event) = @_;
    my ($sock) = $event->to;
    my ($nick) = $event->nick;

    my $type = ($event->args)[1];

    if (uc($type) eq 'SEND') {
	# $connection->new_get($event);
	# do nothing
    } elsif(uc($type) eq 'CHAT') {
	my $x = $connection->new_chat($event);
	print("DCC CHAT connection established with $nick");
	
	$self->{connections}->{DCC}->{users}->{$x->socket}->{nick} = $nick;
	$self->{connections}->{DCC}->{users}->{$x->socket}->{host} = $event->host;
	$self->{connections}->{DCC}->{users}->{$x->socket}->{time} = time();

	$self->dcc_msg($connection, $x->socket, "hi, $nick");
    } else {
	print("Unknown DCC type: " . $type);
    }
}

sub Geckobot::IRC::on_dcc_close {
    my ($self, $connection, $event) = @_;
    my ($sock) = $event->to;
    my ($nick) = $event->nick;

    my $type = ($event->args)[1];
    if (defined $self->{connections}->{DCC}->{users}->{$sock}) {
	print("DCC connection closed by $self->{connections}->{DCC}->{$sock}->{nick}");	
	delete $self->{connections}->{DCC}->{users}->{$sock};
    }
}

sub Geckobot::IRC::on_ping {
    my ($self, $connection, $event) = @_;
    my ($args, $nick) = (($event->args)[0], $event->nick);

    $connection->ctcp_reply($nick, $args);
    print("*** CTCP PING request from $nick received");
}

# Gives lag results for outgoing PINGs.
sub Geckobot::IRC::on_ping_reply {
    my ($self, $connection, $event) = @_;
    my ($args) = ($event->args)[1];
    my ($nick) = $event->nick;

    $args = time - $args;
    print("*** CTCP PING reply from $nick: $args sec.");
}

# Change our nick if we're told to
sub Geckobot::IRC::on_nick_taken {
    my ($self, $connection, $event) = @_;

    $connection->nick(substr($connection->nick, -1).(substr($connection->nick, 0, 8)));
#    $self->param('nickname', $connection->nick());
}

# Display formatted CTCP ACTIONs.
sub Geckobot::IRC::on_action {
    my ($self, $connection, $event) = @_;
    my ($nick, @args) = ($event->nick, $event->args);

    # :FIXME: this should not be necessary:
    shift @args;

    print("* $nick @args");
}

sub Geckobot::IRC::add_handlers {
    my ($self) = shift;
    my ($connection) = $self->{connection};

    $connection->add_handler('cping',  sub { $self->on_ping(@_) });
    $connection->add_handler('crping', sub { $self->on_ping_reply(@_) } );
    $connection->add_handler('msg',    sub { 
			       $poe_kernel->yield('on_msg', @_ ) }
			    );
    $connection->add_handler('chat',   sub { $self->on_chat(@_)} );
    $connection->add_handler('public',    sub { 
			       $poe_kernel->yield('on_msg', @_) }
			    );
    $connection->add_handler('caction',sub { $self->on_action(@_ ) } );
    $connection->add_handler('join',   sub { $self->on_join(@_) } );
    $connection->add_handler('part',   sub { $self->on_part(@_) } );
    $connection->add_handler('cdcc',   sub { $self->on_dcc(@_) } );
    $connection->add_handler('dcc_close',   sub { $self->on_dcc_close(@_) } );

    $connection->add_handler('cversion', sub { $self->on_version(@_) } );

    $connection->add_global_handler([ 251,252,253,254,302,255 ], 
				    sub { $self->on_init(@_) } );
    $connection->add_global_handler(376, sub { $self->on_connect(@_) } );
    $connection->add_global_handler(433, sub { $self->on_nick_taken(@_) } );
    $connection->add_global_handler(353, sub { $self->on_names(@_) } );
}

1;
