#!/usr/bin/perl -w 

BEGIN {
  require 5.005;
}

package Geckobot;

use Geckobot::Message;
use Geckobot::Entity;
use Geckobot::Client;
use Geckobot::Module;

$VERSION = "0.01_00";

use strict;
use Carp;

sub Geckobot::new {
    my $proto = shift ;
    my %args = @_;
    my $self = {};

    bless $self, $proto;
    
    # bootstrap the parameters. 
    %{$self->{param}} = %args;
    $self->{param}->{can_fork} = 1 unless defined $self->{param}->{can_fork};

    $self->read_config();

    # then make sure that the args override the config file
    while (my ($k, $v) = (each %args)) {
      $self->{param}->{$k} = $v;
    }

    $self->load_modules();

    $self->status(1, "Started at ".localtime()." [pid $$]");
    return $self;
}

sub Geckobot::status {
  my ($self, $level) = (shift, shift);

  if ($self->param('gaudy')) {
    if ($level == 1) { 
      print "\e[33m@_\e[0m";
    } elsif ($level == 2) {
      print "\e[32m@_\e[0m";
    } elsif ($level == 3) {
      print "\e[1m\e[31m@_\e[0m";
    } else {
      print "\e[37m@_\e[0m";
    }
  } else {
    print "@_";
  }

  print "\n" unless substr($_[-1],-1,1) eq "\n";
}

sub Geckobot::start {
    my $self = shift;
    $| = 1;

    $self->{running} = 1;

    my $console = new FileHandle;
    $console->fdopen('STDIN', 'r');

    push @{$self->{clients}}, new Geckobot::Client('tty', $console);
    $self->status(1, "** Console active");

    while ($self->{running}) {
	$self->do_one_turn;
    }
}

sub Geckobot::do_one_turn {
    my $self = shift ;
    my $client;

    foreach $client (@{$self->{clients}}) {
	my $message = $client->pop;
	$self->sense($message) if $message;
    }
}

sub read_config  {
  my $self = shift ;

  if (!$self->param('config_file')) {
    $self->status(0, "No config file given");
    return undef;
  }

  if (!open IN, $self->param('config_file')) {
    $self->status(0, "No config file $self->{param}->{config_file}: $!");
  } else {
    while (<IN>) {
      chomp;
      s/\#.*//;
      my ($feature, $value) = split /\s+/, $_, 2;
      $self->param($feature, $value) if defined $feature;
    }
    close IN;
  }
}

sub param {
  my $self = shift;
  my $feature = shift;
  if (@_) {
    my $value = shift;
    $self->status(3, "$feature set to $value");
    $self->{param}->{$feature} = $value;
  }
  $self->{param}->{$feature};
}

sub write_config  {
  my $self = shift ;
  my $file = shift ;
  return undef unless $file;

  if (open OUT, ">$file") {
    foreach (sort keys %{$self->{param}}) {
      print "$_\t$self->{param}->{$_}\n";
    }
    close OUT;
  } else {
    return "$!";
  }
}

sub load_modules {
  my $self = shift ;
  my ($module, $filename, $mdir);

  foreach $mdir (split /\t/, $self->param('module_dirs')) {
    $mdir =~ s|/$||;
    if (opendir DIR, $mdir) {
      $self->status(1, "Adding modules in $mdir");

      foreach $module (grep /\.pm$/, readdir DIR) {
	$filename = $module;
	$module =~ s/\.pm$//;
	$module = "Geckobot::Module::$module";

	eval  { require "$mdir/$filename" };
	warn $@ if $@;

	my $instance;
	eval { $instance = $module->new('bot' => $self ) };
	if (!$@) {
	  if ($instance) {
	    $self->status
		(
		 2,
		 $instance->name, 
		 $instance->VERSION,
		 " instantiated OK\n",
		 $instance->regex()
		 );

	    # channel all status info up through the bot
	    # by setting the status function to ours
	    $instance->status_fn(sub { $self->status(@_) });

	    push @{$self->{modules}}, $instance;

	  } else {
	    warn "$module not instantiated\n";
	  }
	} else {
	  warn "Skipping $filename: $@";
	}
      }
      closedir DIR;
    } else {
      warn "Can't add module directory $mdir: $!\n";
    }
  }
}

sub sense {
  # interface to the world.  'sense' receives
  # a message object, does some preliminary tests,
  # and routes the message on.
  
  my ($self, $message) = @_;

  # $i18n->normalize($message, $language);
  # $daemon->accept($message);

  # preparse($message, language):
  #   normalize

  my $text = $message->{text};

  chomp $text;
  $text =~ s/\s+/ /g;

  my $botname = qr/\bbot\b/i;

  #   named/addressing
  if ($text =~ s/^((\S+)[-:]+ )\S//) {
    $message->{tag} = $2;
    $message->{addressed}++ if $message->{tag} =~ $botname;
  }

  $text =~ s/^($botname, )// and 
      $message->{addressed} = $1;

  not $message->{addressed} and
      $text =~ s/(, $botname)$// and 
	  $message->{addressed} = $1;

  #   negation/correction
  $text =~ s/(^no, |, no, )//i 
      and $message->{negation} = $1;

  not $message->{addressed} and
      $text =~ s/^($botname, )\b// and 
	  $message->{addressed} = $1;
  
  #   polite/rudeness
  $text =~ s/^(excuse me,?( but)? )// 
      and $message->{polite} = $1;

  # tell
  if ($text =~ s/^tell (\S+) //i) {
    $message->{target} = $1;
    $text =~ s/^about //i;
  } else {
    $message->{target} = $message->{sender}->{name};
  }
  $message->{target} = $message->{sender}->{name} if $message->{target} =~ m/^me$/i;

  $message->{text} = $text;

  #   headverb
  if ($text =~ /^(.*?) (is|are) ?(.*)?$/i) {
    my ($X, $verb, $Y) = ($1, $2, $3);

    my $preQ = qr/^can anyone tell me\s+/;
    if ($X =~ s/($preQ)//) {
      $message->{preQ} = $1;
    }

    #   qWord
    my $qWord = qr/what|where|who|when|how/i;

    if ($X =~ s/^($qWord)\s*//) {
      # a question explicitly marked by a q-word
      $message->{qWord} = $1;
      if ($Y) {
	$X = $Y;
      }
    } else {
      $message->{value} = $Y if $Y;
    }

    $message->{text} = $X;
    $message->{verb} = $verb;
  }


  # function/module:
  #   modules, query, delete, update, confused

  if ($text =~ m/^quit/) {
    exit 0;
  } else {
    if (1) {
      my $stat = '';
      foreach (sort keys %$message) {
	$stat .= "$_ -> $message->{$_}\n";
      }
      $self->status(2,$stat);
    }

    if (my $which = $self->try_modules($message)) { 
      $self->status(1, "caught by $which");
    } else {
      goto &question;
    }
  } 
}

sub try_modules {
  my ($self, $message) = @_;
  
  my $result;
  
  # check each module to see it it wants to fire
  # on this piece of input. first module to 
  # return non-undef wins.

  my $which;
  foreach (@{$self->{modules}}) {
    next unless $_;
    $result = $_->try($message); 
    if ($result) {
      $which = $_->name;
      last;
    }
  }

  if (defined $result) {
    $self->effect($message) unless $result eq 'FORKED';

    if ($result eq 'CHILD') {
      $self->status(2, "child [$$] exiting");
      exit 0;
    }

    return $which;
  }

  return undef;
}

sub question {
    # checks if the message is a valid query, and then
    # checks the database(s) to see if we know anything
    # about it

    my ($self, $message) = @_;
    my $content = $message->{content};
    my $handled = 0;

    if ($content =~ /\b(who|what|when|where|why)\b/i) {
	$message->{reply} = "that appears to be a question";
	$self->effect($message);
    } else {
	$self->statement($message);
    }
}

sub statement {
    # statement handling.  this is where the database is
    # checked, and possibly updated, if the incoming message
    # is appropriate.

    my ($self, $message) = @_;

    if ($message->{content} =~ / (is|are) /) {
      $message->{reply} = "that could be a statement";
      goto &effect;
    } else {
      goto &confused;
    }
}

sub confused {
    # the fallback. nothing else caught the message,
    # so either discard it, or admit we're confused.

    my ($self, $message) = @_;
    $message->{reply} = "<confused>";
    $self->effect($message);
}

sub effect {
  my ($self, $message) = @_;
  $self->status(3, $message->{reply});
}

1;

__END__

=head1 NAME

Geckobot - A Multiprotocol Daemon That Remembers

=head1 SYNOPSIS

 blech

=head1 DESCRIPTION

Geckobot is balch claj foo, a liikei kie wikle nunsd fork 
kle unke mimts.

=head2 Motivation

There are these foo:

=over 4

=item *

pickle vroom

Eep op orp ah ah

=item *

knuckle ambrosia

Wankle foo glip-glip!

Foom dela woot nugurad.

=back

=head2 An a Grain of Salt

wump.

=head2 Adding Modules

faleen bak nuptial.

=head2 Instantiating a Geckobot

Winky woo.

=head1 RETURN VALUE

A whole lotta gecko.

=head1 ERRORS 

exceptions 'n 'at.

=head1 EXAMPLES

=over 4
=item * 

akljf and poerp

Yes, aren't they exciting.

=head1 ENVIRONMENT

=head1 FILES

=head1 SEE ALSO

=over 4
=item * 

=head1 NOTES

=head1 CAVEATS

=head1 DIAGNOSTICS

=head1 BUGS

Far too numerous to enumerate.

=head1 RESTRICTIONS

Bugs I can't work around, or am unwilling too in the near future.

=head1 AUTHORS & COPYRIGHTS

=over 4

=item Geckobot has been brought to you by the equation

C<1782**12 + 1841**12 == 1922**12>.

=head2 Contributors

Many many.

=head2 Author

=over 4
=item * Kevin A. Lenzo 

Kevin A. Lenzo is <lenzo@cs.cmu.edu>, and the Geckobot has grown
from the Infobot, also of his design, with some of work dating 
back to 1994, running under MacPerl.

Except where otherwise noted, Geckobot is Copyright 1999 Kevin A. Lenzo.
All rights reserved.  Geckobot is free software; you may redistribute it
and/or modify it under the same terms as Perl itself.

=cut

