signature PARSE_STATE =
  sig
    val reset : unit -> unit

    val setfile : string -> unit
    val newline : int -> unit

    val curline : unit -> int

    val ext : int * int -> Mark.ext option
  end

structure ParseState :> PARSE_STATE =
  struct
    structure M = BinaryMapFn (struct
                                 type ord_key = string
                                 val compare = String.compare
                               end)

    val files : {name : string,
                 curline : int ref,
                 starts : int list ref} M.map ref = ref M.empty

    val curfile : string list ref = ref []

    fun reset () = (files := M.empty; curfile := [])

    val bug = ErrorMsg.impossible

    fun setfile file = 
        (case M.find (!files, file) of
           SOME {name, curline, ...} => (curfile := List.tl (!curfile); curline := 1)
         | NONE => 
           let
             val fstate = {name = file, starts = ref [1], curline = ref 1}
           in
             files := M.insert (!files, file, fstate);
             curfile := file :: (!curfile)
           end)

    fun curf () =
        (case !curfile of
           (f :: _) => f
         | _ => bug "badcur")

    fun newline pos =
        (case M.find (!files, curf ()) of
           SOME {curline, starts, ...} => (starts := pos :: (!starts); curline := (!curline + 1))
         | NONE => bug "nocurfile")

    fun curline () =
        (case M.find (!files, curf ()) of
           SOME {curline, ...} => !curline
         | NONE => bug "nocurline")

    fun look (pos, a :: rest, n) =
         if a < pos then (n, pos - a)
         else look (pos, rest, n - 1) 
       | look _ = (0, 0)

    fun history [] = ""
      | history [f] = f
      | history (f :: fs) = f ^ " (from " ^ history fs ^ ")"

    fun ext (left, right) =
        (case M.find (!files, curf ()) of
           SOME {starts, curline, ...} => SOME (look (left, !starts, !curline),
                                                look (right, !starts, !curline),
                                                history (!curfile))
         | NONE => NONE)
             
  end
