
(* lambda-terms as higher-order inductive datatypes.*)

(* exp represented as deBruijn terms; G never occurs in closed values *)
datatype exp = A of exp * exp | L of exp | V of int | G of unit ref

(* val app : exp -> exp -> exp *)
fun app e1 e2 = A (e1,e2)

(* val lam : (exp -> exp) -> exp *)
fun lam h =
    let val g = ref ()
	fun ll (A (e1,e2)) n = A (ll e1 n, ll e2 n)
	  | ll (L e) n = L (ll e (n+1))
	  | ll (V m) n = V m
	  | ll (G g') n = if g' = g then V n else G g'
    in L (ll (h (G g)) 0) end

(* a few test cases *)
val combS = lam (fn f => lam (fn g => lam (fn x => app (app f x) (app g x))));
val combK = lam (fn x => lam (fn y => x))
val combI' = app (app combS combK) (combK)

(* val eiter : exp -> ('a -> 'a -> 'a) -> (('a -> 'a) -> 'a) -> 'a *)
fun eiter e a l =
    let fun itr (A (e1,e2)) r = a (itr e1 r) (itr e2 r)
	  | itr (L e) r = l (fn x => itr e (x :: r))
	  | itr (V _) [] = raise Fail "eiter V"
	  | itr (V 0) (x::r) = x
	  | itr (V n) (x::r) = itr (V (n-1)) r
	  | itr (G _) r = raise Fail "eiter G"
    in itr e [] end

(* note that, e.g., lam (fn x=>eiter x a l), which raises an exception,
   violates the modality restrictions because x is not closed... *)

(* should return true for any closed exp-value *)
fun testiter e = (eiter e app lam = e)

fun nvars e = eiter e (fn n1=>fn n2=>n1+n2) (fn f=>f 1)
fun nlams e = eiter e (fn n1=>fn n2=>n1+n2) (fn f=>1+f 0)
fun napps e = eiter e (fn n1=>fn n2=>1+n1+n2) (fn f=>f 0)

(* standard CPS translation *)
fun cps e = 
    eiter e (fn e1c => fn e2c =>
	     lam (fn k=>app e1c (lam (fn f=>
		        app e2c (lam (fn a=>app (app f a) k))))))
            (fn hc =>
	     lam (fn k=>app k (lam (fn x=>hc (lam (fn k=>app k x))))))

(* "optimizing" CPS translation *)
fun optcps e =
    lam (fn c=>
    eiter e (fn e1c => fn e2c =>
	     fn k => e1c (fn f => e2c (fn a => app (app f a) (lam k))))
            (fn hc =>
	     fn k => k (lam (fn x => lam (fn c => hc (fn k=>k x) (app c)))))
	    (app c))

(* ecase : exp -> (exp -> exp -> 'a) -> ((exp -> exp) -> 'a) -> 'a *)
fun ecase (A (e1,e2)) a l = a e1 e2
  | ecase (L e) a l =
    l (fn x => let fun sub (A (e1,e2)) n = A (sub e1 n, sub e2 n)
		     | sub (L e) n = L (sub e (n+1))
		     | sub (V n') n = if n' = n then x else V n'
		     | sub (G _) n = raise Fail "ecase sub G"
	       in sub e 0 end)
  | ecase (V _) a l = raise Fail "ecase V"
  | ecase (G _) a l = raise Fail "ecase G";

(* should return true for any closed exp-value *)
fun testcase e = (ecase e app lam = e)

(* Doesn't seem possible to express ecase in terms of eiter.  Really
   need some form of primitive recursion to generalize both. *)

(* conversion to deBruijn form (just identity on representation...) *)
fun todb e = eiter e (fn e1=>fn e2=>fn n=>A (e1 n, e2 n))
                   (fn h=>fn n=>L (h (fn m=>V (m-(n+1))) (n+1))) 0

(* external representation, with named variables *)
datatype eexp = VAR of string | APP of eexp * eexp | LAM of string * eexp

(* conversion to external representation *)
fun toext e = eiter e (fn m1=>fn m2=>fn n=>APP (m1 n, m2 n))
                      (fn h=>fn n=>let val x = "x" ^ makestring n
				   in LAM (x, h (fn n'=>VAR x) (n+1)) end) 0

val cpsS = toext (optcps combS)

(* conversion from external representation *)
fun fromext m =
    let fun fe (APP (m1,m2)) r = app (fe m1 r) (fe m2 r)
	  | fe (LAM (x,m)) r = 
	    lam (fn y => fe m (fn x' => if x'=x then y else r x'))
	  | fe (VAR x) r = r x
    in fe m (fn x=>raise Fail ("unbound "^x)) end

val combI'' = fromext (toext combI');
